# -*- coding: utf-8 -*-

from sandbox.projects.release_machine.components import configs as cfg
from sandbox.projects.release_machine.core import const as rm_const
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
from sandbox.projects.common.constants import constants as sandbox_constants


PACKAGES = {
    "dler": "extsearch/video/robot/rt_transcoder/dler/docker_image/pkg.json",
    "frontend": "extsearch/video/robot/rt_transcoder/frontend/bin/docker_image/pkg.json",
    "nitc": "extsearch/video/robot/rt_transcoder/nitc/docker_image/pkg.json",
    "output_cleaner": "extsearch/video/robot/rt_transcoder/output_cleaner/bin/docker_image/pkg.json",
    "recent_tasks_metrics": "extsearch/video/robot/rt_transcoder/recent-task-metrics/docker_image/pkg.json",
    "transcoded": "yabs/vh/cms-pgaas/transcoded/docker_image/pkg.json",
    "video_manager": "extsearch/video/robot/rt_transcoder/video-manager/docker_image/pkg.json",
    "waiting_task_sender": "extsearch/video/robot/rt_transcoder/waiting-task-sender/docker_image/pkg.json",
    "webhook": "extsearch/video/robot/rt_transcoder/ng/webhookbin/docker_image/pkg.json",
    "scheduler": "yweb/video/transcoder/scheduler/docker_image/pkg.json",
}


class TranscoderCfg(cfg.ReferenceBranchedConfig):
    name = "transcoder"
    responsible = "mvtomilov"

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "RT_TRANSCODER"
        testenv_db_owners = ["mvtomilov", "d-vilkova", "ezabolotskikh", "serjk86", "iemelyanov", "abramov-al", "dlukshto", "iadobryaev", "renkens"]

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrek()
                ]

            @property
            def _branch_part(self):
                """
                    Part of job graph for Testenv with jobs in branch, i.e. tests, builds.
                    :return: list with jobs
                """
                branch_part = super(self.__class__, self)._branch_part
                resource_type = "TRANSCODER_DOCKER_IMAGE"
                for target, path in PACKAGES.items():
                    branch_part.append(
                        jg_build.JobGraphElementBuildBranched(
                            task_name="YA_PACKAGE",
                            out={resource_type: 14},
                            ctx={
                                sandbox_constants.USE_AAPI_FUSE: True,
                                sandbox_constants.BUILD_TYPE_KEY: sandbox_constants.RELEASE_BUILD_TYPE,
                                "packages": path,
                                "resource_type": resource_type,
                                "package_type": "docker",
                                "docker_image_repository": "video",
                                "docker_push_image": True,
                                "docker_save_image": True,
                                "docker_user": "robot-transcoder",
                                "docker_token_vault_name": "robot-transcoder-docker",
                                "custom_version": "{branch}-{revision}-{sandbox_task_id}",
                            },
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 15 << 30,  # 15 Gb
                                    },
                                },
                                "job_name_parameter": target.upper(),
                            },
                        )
                    )

                # graphs
                branch_part.append(jg_build.JobGraphElementBuildBranched(
                    task_name="BUILD_TRANSCODER_GRAPHS_2",
                    out={"TRANSCODER_GRAPHS": 14},
                    ctx={
                        sandbox_constants.USE_AAPI_FUSE: True,
                        sandbox_constants.BUILD_TYPE_KEY: sandbox_constants.RELEASE_BUILD_TYPE,
                        "arts": "yweb/video/faas/graphs/package",
                        "result_rt": "TRANSCODER_GRAPHS",
                        "result_rd": "Transcoder valhalla graphs",
                        "result_single_file": True,
                        "targets": "yweb/video/faas/graphs/package",
                    },
                    job_params={
                        "apiargs": {
                            "requirements": {
                                "disk_space": 5 << 30,  # 5 Gb
                            },
                        },
                        "job_name_parameter": "graphs",
                    },
                ))

                # nitc binary
                branch_part.append(jg_build.JobGraphElementBuildBranched(
                    task_name="YA_MAKE_RELEASE_TO_NANNY",
                    out={"TRANSCODER_NITC_BINARY": 14},
                    ctx={
                        sandbox_constants.USE_AAPI_FUSE: True,
                        sandbox_constants.BUILD_TYPE_KEY: sandbox_constants.RELEASE_BUILD_TYPE,
                        "arts": "extsearch/video/robot/rt_transcoder/nitc/bin/nitc",
                        "result_rt": "TRANSCODER_NITC_BINARY",
                        "result_rd": "Transcoder nitc binary",
                        "result_single_file": True,
                        "targets": "extsearch/video/robot/rt_transcoder/nitc/bin",
                    },
                    job_params={
                        "job_name_parameter": "nitc_binary",
                    },
                ))

                return branch_part

            @property
            def _release(self):
                """
                    Release part of job graph for Testenv with arrows and dependencies
                    :return: list with jobs
                """
                release_part = super(self.__class__, self)._release

                # docker images
                for target in PACKAGES.keys():
                    for release_stage in [rm_const.ReleaseStatus.prestable, rm_const.ReleaseStatus.stable]:
                        release_part.append(
                            jg_release.JobGraphElementReleaseBranched(
                                release_item=target,
                                release_to=release_stage,
                                job_arrows=(
                                    jg_job_triggers.JobTriggerBuild(
                                        job_name_parameter=target,
                                        parent_job_data=(
                                            jg_job_data.ParentDataDict(
                                                "component_resources",
                                                target + "_docker_image",
                                                "TRANSCODER_DOCKER_IMAGE",
                                            )
                                        )
                                    )
                                )
                            )
                        )
                        release_part.append(
                            jg_release.JobGraphElementActionReleaseBranched(
                                release_to=release_stage,
                                release_item=target,
                            )
                        )

                # graphs
                for release_stage in [rm_const.ReleaseStatus.prestable, rm_const.ReleaseStatus.stable]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_item="graphs",
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="graphs",
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "transcoder_graphs",
                                            "TRANSCODER_GRAPHS",
                                        )
                                    )
                                )
                            )
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                            release_item="graphs",
                        )
                    )

                # nitc binary
                for release_stage in [rm_const.ReleaseStatus.prestable, rm_const.ReleaseStatus.stable]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_item="nitc_binary",
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="nitc_binary",
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "nitc_binary",
                                            "TRANSCODER_NITC_BINARY",
                                        )
                                    )
                                )
                            )
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                            release_item="nitc_binary",
                        )
                    )
                return release_part

    class Notify(cfg.ReferenceBranchedConfig.Notify):
        """ Notification settings """
        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            """
            Startrek notifications

            https://wiki.yandex-team.ru/releasemachine/startrekoptions/
            """
            assignee = "mvtomilov"
            use_task_author_as_assignee = True
            queue = "TRANSCODER"
            dev_queue = "TRANSCODER"
            summary_template = "Релиз транскодера {}"
            notify_on_robot_comments_to_tickets = True
            followers = [
                "d-vilkova",
                "ezabolotskikh",
                "iemelyanov",
                "mvtomilov",
                "abramov-al"
            ]
            add_commiters_as_followers = False
            notify_on_deploy_to_feature_st_tickets = True
            deadline = 7

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        """
        Changelog configuration

        https://wiki.yandex-team.ru/releasemachine/changelogs/
        """
        dirs = [
            "arcadia/extsearch/video/robot/rt_transcoder",
            "arcadia/yweb/video/faas"
        ]
        wiki_page = ""

    class SvnCfg(cfg.ReferenceBranchedConfig.SvnCfg):
        """
            Path to branches: /[REPO_NAME]/[branches_folder]/[branch_name]/[branch_folder_name]
            Path to tags: /[REPO_NAME]/[tag_folder]/[tag_name]/[tag_folder_name]
        """
        allow_autobranches = False
        branches_folder = "branches/video/robot"
        branch_name = "video-rtt"

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        resources_info = []

        # docker images
        for each in PACKAGES.keys():
            resources_info.append(
                cfg.ReleasedResourceInfo(
                    name=each + "_docker_image",
                    resource_type="TRANSCODER_DOCKER_IMAGE",
                )
            )

        # graphs
        resources_info.append(
            cfg.ReleasedResourceInfo(
                name="transcoder_graphs",
                resource_type="TRANSCODER_GRAPHS",
            )
        )

        # nitc binary
        resources_info.append(
            cfg.ReleasedResourceInfo(
                name="nitc_binary",
                resource_type="TRANSCODER_NITC_BINARY",
            )
        )

        allow_robots_to_release_stable = True
        release_followers_permanent = [
            "d-vilkova",
            "ezabolotskikh",
            "iemelyanov",
            "mvtomilov",
            "serjk86",
            "abramov-al",
        ]
        deploy_system = rm_const.DeploySystem.nanny
        wait_for_deploy_time_sec = 4 * 60 * 60
