# coding: utf-8

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs as cfg
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.presets as jg_presets
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class TravelHotelsAdministratorCfg(cfg.ReferenceBranchedConfig):
    name = "travel_hotels_administrator"
    display_name = "Travel - Hotels Administrator"
    responsible = "mbobrov"

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "TRAVEL"

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):
            _preset = jg_presets.SinglePackagePreset(
                build_apiargs={"requirements": {"disk_space": 10 * 1024 ** 3}},
                package_names="travel/hotels/hotels_administrator/package/package.json",
                build_ctx={
                    "use_new_format": True,
                    "run_tests": True,
                },
                deploy_system=rm_const.DeploySystem.nanny.name,
                stages=[
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.stable),
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.testing),
                ],
            )

        class JobPatch(cfg.ReferenceBranchedConfig.Testenv.JobPatch):
            # List of jobs that should be deactivated in cloned DB
            deactivate = []

            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "{}_BUILD_{}".format(self.name, "HOTELS_ADMINISTRATOR"): rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [ri.ReleasableItem(
                name="travel_hotels_administrator",
                data=ri.SandboxResourceData("TRAVEL_HOTELS_ADMINISTRATOR_BINARY", ttl=100),
                deploy_infos=[
                    ri.single_nanny_service("travel_hotels_administrator_prod"),
                    ri.single_nanny_service(
                        "travel_hotels_administrator_testing", stage=rm_const.ReleaseStatus.testing,
                    ),
                ],
            )]
        allow_robots_to_release_stable = True

        release_followers_permanent = []

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        ya_make_targets = [
            "arcadia/travel/hotels/hotels_administrator"
        ]
        wiki_page = "travel/hotels/dev/releases/administrator"
        wiki_page_owner = ""
        svn_paths_filter = cfg.ChangelogPathsFilter(rm_const.PermissionType.ALLOWED, ["arcadia/travel"])

    class Notify(cfg.ReferenceBranchedConfig.Notify):

        notifications = [
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="travel_releases",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="travel_releases",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="travel_releases",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="travel_releases",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.STARTREK_TICKET_CREATED_CONDITIONS,
                    join_strategy="OR",
                ),
                message_template_file="notifications/ticket_created.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="travel_releases",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="travel_releases",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "mbobrov"  # Ignored when use_task_author_as_assignee = True
            use_task_author_as_assignee = True
            queue = "YATRAVELREL"
            dev_queue = "HOTELS"
            summary_template = u"Релиз Hotels Administrator Путешествий {}"
            add_commiters_as_followers = False
            followers = [
                "tivelkov",
                "sandello",
                "mbobrov",
                "alexcrush",
            ]
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }

        class Telegram(cfg.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["travel_releases"]
            config = cfg.RmTelegramNotifyConfig(chats=chats)
            invite_link = "https://t.me/joinchat/Co-lrhZao6uefuoeNP9_sg"
