# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
from sandbox.projects.voicetech.resource_types import testenv_resources


class TtsServerCfg(configs.ReferenceBranchedConfig):
    name = "tts_server"
    responsible = "vvkiri"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_task_owner = "VOICESERV_RELEASERS"
        trunk_db = "voicetech-tts-server-trunk"
        branch_db_template = "voicetech-tts-server-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _trunk_part(self):
                default_trunk_part = super(self.__class__, self)._trunk_part
                trunk_part = [
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_TTS_SERVER_GPU",
                        build_item="GPU_TRUNK",
                        filter_targets=[
                            "alice/rtlog/evlogdump",
                            "voicetech/tts/server",
                        ],
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 60 * (1024 ** 3),  # 60 GB
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 60),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=lambda x, rm_config: "GPU r{}: {}".format(
                                    x.revision,
                                    x.commit_message.encode(
                                        "utf-8") if x.commit_message else "<testenv manual runs lost this value>",
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                        ),
                        ctx={
                            "release_type": rm_const.ReleaseStatus.testing,
                            "cuda_version": "10.1",
                        },
                        out={
                            "VOICETECH_TTS_SERVER_GPU": 14,
                            "VOICETECH_TTS_SERVER_RUNNER": 14,
                            "VOICETECH_EVLOGDUMP": 14,
                        },
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_ALICE_CUTTLEFISH_TTS_ADAPTER_PACKAGE",
                        build_item="CUTTLEFISH_TTS_ADAPTER_PACKAGE_TRUNK",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "observed_paths": [
                                "arcadia/alice/cuttlefish/bin/tts_adapter",
                            ],
                            "apiargs": {
                                "requirements": {
                                    # disc space: 15GB (for arcadia src)
                                    "disk_space": 15 * (1024 ** 3),  # 15 Gb
                                },
                            },
                        },
                        out={"ALICE_CUTTLEFISH_TTS_ADAPTER_PACKAGE": 14},
                    ),

                ]
                return default_trunk_part + trunk_part

            @property
            def _branch_part(self):
                default_branch_part = super(self.__class__, self)._branch_part
                branch_part = []

                class Lingware:
                    def __init__(
                        self,
                        released_resource_name,
                        sandbox_resource_name,
                        testenv_tasks_suffix,
                        beta_conf_type,
                        func_tests_group,
                        test_data_resource_type,
                        speaker,
                        language
                    ):
                        self.released_resource_name = released_resource_name
                        self.sandbox_resource_name = sandbox_resource_name
                        self.testenv_tasks_suffix = testenv_tasks_suffix
                        self.beta_conf_type = beta_conf_type
                        self.test_data_resource_type = test_data_resource_type
                        self.speaker = speaker
                        self.language = language
                        self.beta_generator = None
                        self.resource_info = [
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                self.released_resource_name,
                                self.sandbox_resource_name,
                            ),
                        ]
                        self.func_tests_group = func_tests_group
                        self.func_test = None
                        self.quality_test = None

                LINGWARES = [
                    Lingware(
                        released_resource_name="ru_e2e",
                        # got from sandbox/projects/release_machine/components/configs/tts_server.py
                        sandbox_resource_name="VOICETECH_TTS_RU_GPU",
                        testenv_tasks_suffix="RU_E2E",
                        beta_conf_type="ru_e2e",
                        func_tests_group="tts_ru_shitova_e2e",
                        language="ru",
                        speaker="shitova",
                        test_data_resource_type="VOICETECH_TTS_RU_DATA",
                    ),
                    Lingware(
                        released_resource_name="ru_multispeaker_e2e",
                        sandbox_resource_name="VOICETECH_TTS_RU_MULTISPEAKER_GPU",
                        testenv_tasks_suffix="RU_MULTISPEAKER_E2E",
                        beta_conf_type="ru_multispeaker_e2e",
                        func_tests_group="tts_ru_oksana_e2e",
                        language="ru",
                        speaker="oksana",
                        test_data_resource_type="VOICETECH_TTS_RU_DATA_OKSANA",
                    ),
                    Lingware(
                        released_resource_name="ru_valtz_e2e",
                        sandbox_resource_name="VOICETECH_TTS_RU_VALTZ_GPU",
                        testenv_tasks_suffix="RU_VALTZ_E2E",
                        beta_conf_type="ru_valtz_e2e",
                        func_tests_group="tts_ru_valtz_e2e",
                        language="ru",
                        speaker="valtz",
                        test_data_resource_type="VOICETECH_TTS_RU_DATA_VALTZ",
                    ),
                    Lingware(
                        released_resource_name="tr_e2e",
                        sandbox_resource_name="VOICETECH_TTS_TR_GPU",
                        testenv_tasks_suffix="TR_E2E",
                        beta_conf_type="tr_e2e",
                        func_tests_group="tts_tr_selay_e2e",
                        language="tr",
                        speaker="selay",
                        test_data_resource_type="VOICETECH_TTS_TR_DATA",
                    ),
                ]

                for lingware_type in testenv_resources.TTS_LINGWARE_TYPES:
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="BUILD_TTS_LINGWARE",
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 100 * (1024 ** 3),  # 100 GB -- for huge lingwares
                                    },
                                },
                                "job_name_parameter": lingware_type,
                                "observed_paths": [
                                    "voicetech/tts/lingware/src/" + lingware_type.lower(),
                                    "voicetech/tts/lingware/src/common",
                                ],
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key="result_rd",
                                    transform=jg_utils.construct_resource_description,
                                ),
                                jg_arrows.ParamsData(
                                    input_key="ap_arcadia_revision",
                                    transform=lambda x, rm_config: x.revision,
                                ),
                            ),
                            ctx={
                                "resource_type": testenv_resources.TTS_LINGWARE_TYPES[
                                    lingware_type].lingware_resource.Name,
                                "release_type": rm_const.ReleaseStatus.stable,
                            },
                            out={
                                testenv_resources.TTS_LINGWARE_TYPES[lingware_type].lingware_resource.Name: 14,
                            },
                        )
                    )

                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_TTS_SERVER_GPU",
                        build_item="GPU",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 60 * (1024 ** 3),  # 60 GB
                                },
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=lambda x, rm_config: "GPU r{}: {}".format(
                                    x.revision,
                                    x.commit_message.encode(
                                        "utf-8") if x.commit_message else "<testenv manual runs lost this value>",
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                        ),
                        ctx={
                            "release_type": rm_const.ReleaseStatus.stable,
                            "cuda_version": "10.1",
                            "with_hfg": True,
                        },
                        out={
                            testenv_resources.VOICETECH_TTS_SERVER_GPU.Name: 14,
                            testenv_resources.VOICETECH_TTS_SERVER_RUNNER.Name: 14,
                            testenv_resources.VOICETECH_TTS_SERVER_EVLOGDUMP.Name: 14,
                        },
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="BUILD_ALICE_CUTTLEFISH_TTS_ADAPTER_PACKAGE",
                        build_item="CUTTLEFISH_TTS_ADAPTER_PACKAGE",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                            "observed_paths": [
                                "arcadia/alice/cuttlefish/bin/tts_adapter",
                            ],
                            "apiargs": {
                                "requirements": {
                                    # disc space: 15GB (for arcadia src)
                                    "disk_space": 15 * (1024 ** 3),  # 15 Gb
                                },
                            },
                        },
                        out={"ALICE_CUTTLEFISH_TTS_ADAPTER_PACKAGE": 14},
                    ),
                )
                build_data = (
                    jg_job_data.ParentDataDict(
                        "component_resources",
                        "tts-server",  # got from sandbox/projects/release_machine/components/configs/tts_server.py
                        "VOICETECH_TTS_SERVER_GPU",
                    ),
                    jg_job_data.ParentDataDict(
                        "component_resources",
                        "run_sh",  # got from sandbox/projects/release_machine/components/configs/tts_server.py
                        "VOICETECH_TTS_SERVER_RUNNER",
                    ),
                    jg_job_data.ParentDataDict(
                        "component_resources",
                        "evlogdump",  # got from sandbox/projects/release_machine/components/configs/tts_server.py
                        "VOICETECH_EVLOGDUMP",
                    ),
                )
                prev_lingware_suffix = None
                for lingware, required_strings_func in zip(
                    LINGWARES,
                    [
                        lambda x, params: [x.split(":")[1] + "/arcadia", "lingware/ru_e2e"],
                        lambda x, params: [x.split(":")[1] + "/arcadia", "lingware/ru_multispeaker_e2e"],
                        lambda x, params: [x.split(":")[1] + "/arcadia", "lingware/ru_valtz_e2e"],
                        lambda x, params: [x.split(":")[1] + "/arcadia", "lingware/tr_e2e"],
                    ]
                ):
                    job_arrows = [
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="GPU",
                            parent_job_data=build_data,
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter=lingware.testenv_tasks_suffix,
                            parent_job_data=lingware.resource_info,
                        ),
                        jg_arrows.ParentsData(
                            input_key="tasks_to_wait",
                            triggers=(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="GPU",
                                    parent_job_data=(
                                        jg_job_data.ParentDataId(
                                            input_key="build_id",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter=lingware.testenv_tasks_suffix,
                                    parent_job_data=(
                                        jg_job_data.ParentDataId(
                                            input_key="lingware_testenv_tasks_suffix",
                                        )
                                    ),
                                ),
                            ),
                            transform=lambda build_id, lingware_testenv_tasks_suffix: jg_utils.join_args_with_sep(
                                ",",
                                build_id,
                                lingware_testenv_tasks_suffix,
                            ),
                        ),
                    ]
                    if prev_lingware_suffix is not None:
                        job_arrows.append(
                            jg_job_triggers.JobTriggerActionRunAcceptanceBranchedTest(
                                job_name_parameter=prev_lingware_suffix
                            )
                        )
                    branch_part.append(
                        jg_test.JobGraphElementYappyBetaGeneratorBranched(
                            beta_conf_type=lingware.beta_conf_type,
                            job_arrows=tuple(job_arrows),
                            ctx={"beta_name_source": "STRING"},
                        )
                    )
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="WAIT_VOICE_SERVER",
                            job_params={
                                "job_name_parameter": "WAIT_YAPPY_BETA_{}".format(lingware.testenv_tasks_suffix),
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerGenerateBeta(
                                    job_name_parameter=lingware.beta_conf_type,
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key="xxx_server_url",
                                            output_key="beta_name",
                                            transform=lambda x,
                                                             params: "https://{}.hamster.yandex.ru/ru/gpu/info".format(
                                                x),
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerNewTag(
                                    jg_job_data.ParentDataCtx(
                                        input_key="required_strings",
                                        output_key="result_path",
                                        transform=required_strings_func,
                                    )
                                ),
                                jg_arrows.ParamsData(
                                    input_key="release_number",
                                    transform=jg_utils.get_major_release_number,
                                ),
                                jg_arrows.ParentsData(
                                    input_key="tasks_to_wait",
                                    triggers=(
                                        jg_job_triggers.JobTriggerGenerateBeta(
                                            job_name_parameter=lingware.beta_conf_type,
                                            parent_job_data=(
                                                jg_job_data.ParentDataId(
                                                    input_key="beta_id",
                                                ),
                                            ),
                                        ),
                                        jg_job_triggers.JobTriggerNewTag(
                                            parent_job_data=(
                                                jg_job_data.ParentDataId(
                                                    input_key="tag_id",
                                                )
                                            ),
                                        ),
                                    ),
                                    transform=lambda beta_id, tag_id: jg_utils.join_args_with_sep(",", beta_id, tag_id),
                                ),
                            ),
                            ctx={
                                "server_url": "http://alice-tts-rm.yappy.yandex.ru/ru/gpu/info",
                                # "required_strings": <set later>
                                "checks_limit": 10,
                                "checks_period": 300,
                                "beta_description": lingware.released_resource_name,
                                "tests_group": lingware.func_tests_group,
                                # "release_number": <set later>
                            },
                        )
                    )
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="CIT_TEST_VOICE_SERVER",
                            job_params={
                                "job_name_parameter": "FUNC_ACCEPTANCE_{}".format(lingware.testenv_tasks_suffix),
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter="WAIT_YAPPY_BETA_{}".format(lingware.testenv_tasks_suffix),
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key="xxx_beta_description",
                                            output_key="beta_description",
                                            transform=lambda x, params: str(x),
                                        ),
                                        jg_job_data.ParentDataId(
                                            input_key="tasks_to_wait",
                                        ),
                                    ),
                                ),
                                jg_arrows.ParamsData(
                                    input_key="release_number",
                                    transform=jg_utils.get_major_release_number,
                                ),
                            ),
                            ctx={
                                "uniproxy_websocket_url": "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-tts-rm/uni.ws",
                                "tests_group": lingware.func_tests_group,
                                # "release_number": <set later>
                            }
                        )
                    )
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="LOAD_TEST_TTS_SERVER",
                            job_params={
                                "job_name_parameter": "LOAD_ACCEPTANCE_{}".format(lingware.testenv_tasks_suffix),
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key="release_number",
                                    transform=jg_utils.get_major_release_number,
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter="FUNC_ACCEPTANCE_{}".format(lingware.testenv_tasks_suffix),
                                    parent_job_data=(
                                        jg_job_data.ParentDataId(
                                            input_key="tasks_to_wait",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter="WAIT_YAPPY_BETA_{}".format(lingware.testenv_tasks_suffix),
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key="xxx_beta_description",
                                            output_key="beta_description",
                                            transform=lambda x, params: str(x),
                                        ),
                                    ),
                                ),

                            ),
                            ctx={
                                "lang": lingware.language,
                                "voice": lingware.speaker,
                                "min_rps": 10,
                                "max_rps": 60,
                                "num_steps": 6,
                                "step_duration": 300,
                                "num_ramps": 2,
                                "texts_resource_type": lingware.test_data_resource_type,
                                "tts_beta_url": "http://alice-tts-rm.yappy.yandex.ru/ru/gpu/info"
                            }
                        )
                    )
                    branch_part.append(
                        jg_test.JobGraphElementActionRunAcceptanceBranched(
                            frequency=(jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            job_params={
                                "job_name_parameter": lingware.testenv_tasks_suffix,
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter="FUNC_ACCEPTANCE_{}".format(lingware.testenv_tasks_suffix),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter="LOAD_ACCEPTANCE_{}".format(lingware.testenv_tasks_suffix),
                                ),
                            ),
                        )
                    )
                    prev_lingware_suffix = lingware.testenv_tasks_suffix
                    for release_stage in [rm_const.ReleaseStatus.prestable, rm_const.ReleaseStatus.stable]:
                        branch_part.append(
                            jg_release.JobGraphElementReleaseBranched(
                                release_to=release_stage,
                                release_item=lingware.testenv_tasks_suffix,
                                job_arrows=(
                                    jg_job_triggers.JobTriggerBuild(
                                        job_name_parameter="GPU",
                                        parent_job_data=build_data,
                                    ),
                                    jg_job_triggers.JobTriggerBuild(
                                        job_name_parameter=lingware.testenv_tasks_suffix,
                                        parent_job_data=lingware.resource_info,
                                    ),
                                )
                            )
                        )
                        branch_part.append(
                            jg_release.JobGraphElementActionReleaseBranched(
                                release_to=release_stage,
                                release_item=lingware.testenv_tasks_suffix,
                            )
                        )
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name='BUILD_TTS_FAST_DATA_SMOKE_TEST',
                        job_params={
                            "job_name_parameter": "SMOKE_TESTS"
                        },
                        ctx={
                            'build_type': 'release'
                        },
                        out={
                            'VOICETECH_TTS_FASTDATA_SMOKE_TEST_CASES': 14,
                            'VOICETECH_TTS_FASTDATA_SMOKE_TEST': 14
                        }
                    )
                )
                for release_stage in [rm_const.ReleaseStatus.prestable, rm_const.ReleaseStatus.stable]:
                    branch_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            release_item="GPU",
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="GPU",
                                    parent_job_data=build_data,
                                ),
                            )
                        )
                    )
                    branch_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                            release_item="GPU",
                        )
                    )
                branch_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="SMOKE_TESTS",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="SMOKE_TESTS",
                                parent_job_data=[
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "smoke_test_cases",
                                        "VOICETECH_TTS_FASTDATA_SMOKE_TEST_CASES"
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "smoke_test",
                                        "VOICETECH_TTS_FASTDATA_SMOKE_TEST"
                                    )
                                ],
                            )
                        )
                    )
                )
                branch_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="SMOKE_TESTS"
                    )
                )
                branch_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="CUTTLEFISH_TTS_ADAPTER_PACKAGE",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CUTTLEFISH_TTS_ADAPTER_PACKAGE",
                                parent_job_data=[
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "tts_adapter",
                                        "ALICE_CUTTLEFISH_TTS_ADAPTER_PACKAGE"
                                    )
                                ],
                            )
                        )
                    )
                )
                branch_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="CUTTLEFISH_TTS_ADAPTER_PACKAGE"
                    )
                )
                return default_branch_part + branch_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                uppercase_name = self.name.upper()
                tests.update({
                    "_{}_UMBRELLA_ACCEPT_SCHEDULED".format(uppercase_name): rm_const.TestFrequencies.FORTY_TWO_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        resources_info = [
            configs.ReleasedResourceInfo(
                name="tts-server",
                resource_type="VOICETECH_TTS_SERVER_GPU",
            ),
            configs.ReleasedResourceInfo(
                name="evlogdump",
                resource_type="VOICETECH_EVLOGDUMP",
            ),
            configs.ReleasedResourceInfo(
                name="run_sh",
                resource_type="VOICETECH_TTS_SERVER_RUNNER",
            ),
            configs.ReleasedResourceInfo(
                name="ru_e2e",
                resource_type="VOICETECH_TTS_RU_GPU",
            ),
            configs.ReleasedResourceInfo(
                name="ru_valtz_e2e",
                resource_type="VOICETECH_TTS_RU_VALTZ_GPU",
            ),
            configs.ReleasedResourceInfo(
                name="ru_multispeaker_e2e",
                resource_type="VOICETECH_TTS_RU_MULTISPEAKER_GPU",
            ),
            configs.ReleasedResourceInfo(
                name="tr_e2e",
                resource_type="VOICETECH_TTS_TR_GPU",
            ),
            configs.ReleasedResourceInfo(
                name="smoke_test",
                resource_type="VOICETECH_TTS_FASTDATA_SMOKE_TEST",
            ),
            configs.ReleasedResourceInfo(
                name="smoke_test_cases",
                resource_type="VOICETECH_TTS_FASTDATA_SMOKE_TEST_CASES",
            ),
            configs.ReleasedResourceInfo(
                name="tts_adapter",
                resource_type="ALICE_CUTTLEFISH_TTS_ADAPTER_PACKAGE"
            )
        ]
        wait_for_deploy_time_sec = 3600
        allow_robots_to_release_stable = True
        allow_old_releases = True

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            """Mail notifications configuration"""
            mailing_list = [
                "vvkiri@yandex-team.ru",
            ]

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "vvkiri"
            queue = "ALICERELEASE"
            dev_queue = "ALICE"
            summary_template = u"Релиз tts-server-а и/или его данных {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "av-kotikov",
                "vvkiri",
                "plato",
            ]
            add_commiters_as_followers = False
            deadline = 7
            hide_commits_under_cut = True
            important_changes_limit = 100

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chat = "tts_release"
            config = configs.RmTelegramNotifyConfig(chats=[
                chat
            ])
            invite_link = u"https://t.me/joinchat/Bu6WMhUORoqEvgzwBhjEqA"

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "voicetechnology/dev/serverinfr/tts-server/releases/"
        dirs = [
            "arcadia/voicetech/tts/core",
            "arcadia/voicetech/tts/lingware",
            "arcadia/voicetech/tts/server",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "voicetech/tts_server"

    class Yappy(yappy_cfg.YappyBaseCfg):
        betas = {
            "ru_e2e": yappy_cfg.YappyBetaCfg(
                beta_name="alice-tts-rm",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyBetaPatch(
                        patch_file="alice-tts-alice-tts-rm",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(param_name="tts-server", local_path="tts-server"),
                            yappy_cfg.YappyParametrizedResource(param_name="run_sh", local_path="run.sh"),
                            yappy_cfg.YappyParametrizedResource(param_name="ru_e2e", local_path="lingware"),
                        ],
                        instance_spec=[
                            yappy_cfg.YappyEnvVar(container="supervisor", name="CUDA_VISIBLE_DEVICES", value="0,1"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="TTS_MAXCONNECTION", value="400"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="TTS_TOTALCONNECTION", value="400"),
                        ],
                        parent_service="tts-ru-gpu-yp-vla"
                    )
                ]
            ),
            "ru_multispeaker_e2e": yappy_cfg.YappyBetaCfg(
                beta_name="alice-tts-rm",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyBetaPatch(
                        patch_file="alice-tts-alice-tts-rm",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(param_name="tts-server", local_path="tts-server"),
                            yappy_cfg.YappyParametrizedResource(param_name="run_sh", local_path="run.sh"),
                            yappy_cfg.YappyParametrizedResource(param_name="ru_multispeaker_e2e",
                                                                local_path="lingware"),
                        ],
                        instance_spec=[
                            yappy_cfg.YappyEnvVar(container="supervisor", name="CUDA_VISIBLE_DEVICES", value="0,1,2,3"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="TTS_MAXCONNECTION", value="400"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="TTS_TOTALCONNECTION", value="400"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="CUDNN_CONV_WSCAP_DBG", value="16"),
                        ],
                        parent_service="tts-ru-oksana-gpu-yp-vla"
                    )
                ]
            ),
            "ru_valtz_e2e": yappy_cfg.YappyBetaCfg(
                beta_name="alice-tts-rm",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyBetaPatch(
                        patch_file="alice-tts-alice-tts-rm",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(param_name="tts-server", local_path="tts-server"),
                            yappy_cfg.YappyParametrizedResource(param_name="run_sh", local_path="run.sh"),
                            yappy_cfg.YappyParametrizedResource(param_name="ru_valtz_e2e", local_path="lingware"),
                        ],
                        instance_spec=[
                            yappy_cfg.YappyEnvVar(container="supervisor", name="CUDA_VISIBLE_DEVICES", value="0,1,2"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="TTS_MAXCONNECTION", value="400"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="TTS_TOTALCONNECTION", value="400"),
                        ],
                        parent_service="tts-ru-valtz-gpu-yp-vla"
                    )
                ]
            ),
            "tr_e2e": yappy_cfg.YappyBetaCfg(
                beta_name="alice-tts-rm",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyBetaPatch(
                        patch_file="alice-tts-alice-tts-rm",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(param_name="tts-server", local_path="tts-server"),
                            yappy_cfg.YappyParametrizedResource(param_name="run_sh", local_path="run.sh"),
                            yappy_cfg.YappyParametrizedResource(param_name="tr_e2e", local_path="lingware"),
                        ],
                        instance_spec=[
                            yappy_cfg.YappyEnvVar(container="supervisor", name="CUDA_VISIBLE_DEVICES", value="0,1,2"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="TTS_MAXCONNECTION", value="400"),
                            yappy_cfg.YappyEnvVar(container="supervisor", name="TTS_TOTALCONNECTION", value="400"),
                        ],
                        parent_service="tts-tr-gpu-yp-vla"
                    )
                ]
            )
        }
