# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.presets as jg_presets
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


class TurboInitCfg(configs.ReferenceBranchedConfig):
    name = "turbo_init"
    responsible = "shubert"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "turbo-init-trunk"
        trunk_task_owner = "TURBO"
        branch_db_template = "turbo-init-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            _preset = jg_presets.SinglePackagePreset(
                build_task_name="BUILD_TURBO",
                build_ctx={"checkout_mode": "auto"},
                package_names="packages/debian/runtime/web/turbopages/init/pkg.json",
                deploy_system=rm_const.DeploySystem.nanny.name,
                stages=[jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.stable)],
            )

            @property
            def _release(self):
                return super(self.__class__, self)._release + [
                    jg_release.JobGraphElementScheduleRelease(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_params={
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny_push.name,
                                "flows": ["deploy"],
                                "warden_component_name": "turbo_init",
                                "warden_parent_component_name": "turbo",
                            }
                        },
                        job_arrows=jg_job_triggers.JobTriggerBuild(
                            parent_job_data=jg_job_data.ParentDataDict(
                                "component_resources", "turbo_init_binary", "TURBO_INIT_PACKAGE"
                            ),
                        ),
                    ),
                    jg_release.JobGraphElementActionScheduleBranched(
                        rm_const.ReleaseStatus.stable,
                        release_item="SCHEDULE"
                    )
                ]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="turbo_init_binary",
                    data=ri.SandboxResourceData("TURBO_INIT_PACKAGE", ttl=30),
                    deploy_infos=[
                        ri.NannyDeployInfo(
                            [
                                ri.DeployService("turbo_api_init_production_yp_sas"),
                                ri.DeployService("turbo_api_init_production_yp_man"),
                                ri.DeployService("turbo_api_init_production_yp_vla"),
                                ri.DeployService("turbo_init_production_yp_sas"),
                                ri.DeployService("turbo_init_production_yp_man"),
                                ri.DeployService("turbo_init_production_yp_vla"),
                                ri.DeployService("turbo_init_hamster_yp_sas"),
                                ri.DeployService("turbo_init_hamster_yp_man"),
                                ri.DeployService("turbo_init_hamster_yp_vla"),
                            ],
                        ),
                    ],
                ),
                ri.DynamicReleasableItem(
                    name="scheduled_bundle",
                    data=ri.SandboxResourceData("SCHEDULED_RM_RELEASE_DATA"),
                    deploy_infos=[ri.NannyDeployInfo()],
                ),
            ]

        allow_old_releases = True
        allow_robots_to_release_stable = True

    class Notify(configs.ReferenceBranchedConfig.Notify):

        notifications = [
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="turbo_devops",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="turbo_devops",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["turbo_devops"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "shubert"
            queue = "TURBOINITREL"
            dev_queue = "TURBO"
            summary_template = u"priemka turbo_init {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "shubert",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "jandekspoisk/kachestvopoiska/turbopages/dev/releases/turbo_int/"
        observed_paths = [
            "arcadia/quality/functionality/turbo/init/",
        ]
        review_groups = [
            "shubert",
        ]
