# -*- coding: utf-8 -*-
#
# Links to release machine docs:
# * https://wiki.yandex-team.ru/releasemachine/

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers


MOBILE_UAC_BACKEND_ABC_SERVICE_ID = 32978
ROLE_DUTY = 413


class UacBackendCfg(configs.ReferenceBranchedConfig):
    name = "uac_backend"
    responsible = "kigan"

    class MergesCfg(configs.ReferenceBranchedConfig.MergesCfg):
        number_last_branches_to_merge = 1

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "RMP"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        use_arc=True,
                        job_params={
                            "observed_paths": ["yabs/rmp/backend"],
                        },
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(17, 17)],
                            weekday_list=[2, 4],  # Wednesday and Friday
                            once_a_day=True
                        )
                    ),
                ]

            output_build_resources = {
                "UAC_BACKEND_API_SERVER": 40,
                "UAC_BACKEND_CELERY_WORKER": 40,
                "UAC_BACKEND_CELERY_BEAT": 40,
                "UAC_BACKEND_CONF_LOCAL_PRODUCTION": 40,
                "UAC_BACKEND_CONF_LOCAL_TESTING": 40,
                "UAC_WORKER_CONF_LOCAL_PRODUCTION": 40,
                "UAC_WORKER_CONF_LOCAL_TESTING": 40,
            }

            output_export_configs = {
                "UAC_BACKEND_INSTANCECTL_CONFIG": 40,
                "UAC_BACKEND_UPLOAD_INSTANCECTL_CONFIG": 40,
                "UAC_WORKER_INSTANCECTL_CONFIG": 40,
                "UAC_BEAT_INSTANCECTL_CONFIG": 40,
                "UAC_BACKEND_UWSGI_PRODUCTION_CONFIG": 40,
                "UAC_BACKEND_UPLOAD_UWSGI_PRODUCTION_CONFIG": 40,
                "UAC_BACKEND_UWSGI_TESTING_CONFIG": 40,
                "UAC_WORKER_UWSGI_CONFIG": 40,
                "UAC_BEAT_UWSGI_CONFIG": 40,
                "UAC_BACKEND_PUSH_CLIENT_PRODUCTION_CONFIG": 40,
                "UAC_BACKEND_PUSH_CLIENT_TESTING_CONFIG": 40,
                "UAC_WORKER_PUSH_CLIENT_PRODUCTION_CONFIG": 40,
                "UAC_WORKER_PUSH_CLIENT_TESTING_CONFIG": 40,
                "UAC_LOGROTATE_PRODUCTION_CONFIG": 40,
                "UAC_LOGROTATE_TESTING_CONFIG": 40,
                "UAC_PUSH_CLIENT_HIDE_SECRETS_SCRIPT": 40,
            }

            @property
            def _trunk_part(self):
                trunk_part = super(self.__class__, self)._trunk_part
                trunk_part.append(
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_UAC_BACKEND",
                        ctx={"clear_build": True},
                        out=self.output_build_resources,
                        job_params={
                            "apiargs": {"hidden": True},
                            "job_name_parameter": "TRUNK",
                        },
                    )
                )
                trunk_part.append(
                    jg_test.JobGraphElementTestCommon(
                        task_name='EXPORT_TEMPLATE_RESOURCES_AND_NOTIFY_TELEGRAM',
                        ctx={
                            "notify": True,
                            "people": UacBackendCfg.Notify.Telegram.chats,
                            "template_resources": self.output_export_configs.keys(),
                            "export_name": "UAC backend configs",
                            "release_statuses_to_notify": ["testing", "stable"],
                        },
                        out=self.output_export_configs,
                        job_params={
                            "observed_paths": [
                                "yabs/rmp/backend/configs",
                            ],
                            "job_name_parameter": "EXPORT_CONFIGS",
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key=sandbox_constants.ARCADIA_URL_KEY,
                                transform=lambda x, rm_config: "{svn_ssh_url}/arcadia@{revision}".format(
                                    svn_ssh_url=x.svn_ssh_url,
                                    revision=x.revision,
                                ),
                                override=True,
                            ),
                        )
                    )
                )
                for res_type, observed_paths in [
                    (
                        "RMP_SAAS_INDEXER",
                        [
                            "yabs/rmp/tools/rmp_saas_indexer",
                        ],
                    ),
                    (
                        "RMP_APP_INFO_PARSER",
                        [
                            "yabs/rmp/backend/src/app_stores",
                            "yabs/rmp/tools/app_info/parser",
                            "yabs/rmp/tools/libs/app_stores_parser",
                            "yabs/rmp/tools/libs/app_stores_utils",
                            "yabs/rmp/tools/libs/logbroker_client",
                        ],
                    ),
                    (
                        "RMP_APP_TO_URL_MATCHER",
                        [
                            "yabs/rmp/tools/app_info/app_to_web_urls",
                        ],
                    ),
                    (
                        "RMP_YT_STATES_CLEANER",
                        [
                            "yabs/rmp/tools/yt_states_cleaner",
                        ],
                    ),
                    (
                        "RMP_APP_DATA_BUILDER",
                        [
                            "yabs/rmp/tools/app_info/app_data_builder",
                        ],
                    ),
                    (
                        "RMP_NIRVANA_TMP_CLEANER",
                        [
                            "yabs/rmp/tools/nirvana_tmp_cleaner",
                        ]
                    ),
                    (
                        "RMP_DATA_CHECKER",
                        [
                            "yabs/rmp/tools/app_info/data_checker",
                        ],
                    ),
                    (
                        "RMP_AURORA_DATA_BUILDER",
                        [
                            "yabs/rmp/tools/app_info/aurora_data_builder",
                        ],
                    ),
                    (
                        "RMP_APP_NAMES",
                        [
                            "yabs/rmp/tools/app_info/app_names",
                        ],
                    ),
                    (
                        "RMP_AURORA_URL_SYNCER",
                        [
                            "yabs/rmp/tools/app_info/aurora_url_syncer",
                        ],
                    ),
                    (
                        "RMP_SK_NETWORK_BUILDER",
                        [
                            "yabs/rmp/tools/app_info/sk_ad_netwok_builder",
                        ],
                    ),
                    (
                        "RMP_GET_DEEPLINKS",
                        [
                            "yabs/rmp/tools/app_info/get_deeplinks",
                        ],
                    ),
                    (
                        "RMP_TABLES_MONITORINGS",
                        [
                            "yabs/rmp/tools/app_info/tables_monitorings",
                        ],
                    ),
                    (
                        "RMP_TABLES_COMPARATOR",
                        [
                            "yabs/rmp/tools/tables_comparator",
                        ],
                    ),
                    (
                        "RMP_BANNER_TO_APP_LINKS",
                        [
                            "yabs/rmp/tools/app_info/banner_to_app_links",
                        ],
                    ),
                    (
                        "RMP_VIDEO_FACTORS",
                        [
                            "yabs/rmp/tools/app_info/video_factors",
                        ],
                    ),
                    (
                        "RMP_POPULAR_BANNERS",
                        [
                            "yabs/rmp/tools/app_info/popular_banners",
                        ],
                    ),
                    (
                        "RMP_TRACKER_URL_STAT",
                        [
                            "yabs/rmp/tools/tracker_url_stat",
                        ],
                    ),
                    (
                        "RMP_CLICKS_DATA_BUILDER",
                        [
                            "yabs/rmp/tools/app_info/clicks_data_builder",
                        ],
                    ),
                    (
                        "RMP_LINEAR_TREND_ESTIMATOR",
                        [
                            "yabs/rmp/tools/app_info/linear_trend_estimator",
                        ],
                    ),
                    (
                        "RMP_TRACKER_URL_RETRIER",
                        [
                            "yabs/rmp/tools/tracker_url_retrier",
                        ],
                    ),
                    (
                        "RMP_RECOMMENDED_COST",
                        [
                            "yabs/rmp/tools/recommended_cost"
                        ],

                    ),
                    (
                        "RMP_BANNER_TO_APP_BUILDER",
                        [
                            "yabs/rmp/tools/app_info/banner_to_app_builder",
                        ],
                    ),
                    (
                        "RMP_FERRYMAN_UPLOADER",
                        [
                            "yabs/rmp/tools/ferryman_uploader",
                        ],
                    ),
                    (
                        "RMP_DEEPLINKS_MONITORING",
                        [
                            "yabs/rmp/tools/app_info/deeplinks_monitoring",
                        ],
                    ),
                ]:
                    trunk_part.append(
                        jg_test.JobGraphElementTestCommon(
                            task_name='BUILD_RESOURCE_AND_NOTIFY_TELEGRAM',
                            ctx={
                                "clear_build": True,
                                "notify": True,
                                "people": UacBackendCfg.Notify.Telegram.chats,
                                "result_rt": res_type,
                            },
                            out={res_type: 40},
                            job_params={
                                "observed_paths": observed_paths,
                                "job_name_parameter": "BUILD_TOOL_{}".format(res_type),
                                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key=sandbox_constants.ARCADIA_URL_KEY,
                                    transform=lambda x, rm_config: "{svn_ssh_url}/arcadia@{revision}".format(
                                        svn_ssh_url=x.svn_ssh_url,
                                        revision=x.revision,
                                    ),
                                    override=True,
                                ),
                            )
                        )
                    )

                for dashboard_group in ("testing", "production"):
                    trunk_part.append(
                        jg_test.JobGraphElementTestCommon(
                            task_name='VALIDATE_UAC_SERVICES_ENV_SPEC',
                            ctx={
                                "notify": True,
                                "people": UacBackendCfg.Notify.Telegram.chats,
                                "dashboard_name": "uac_backend",
                                "dashboard_groups": [dashboard_group],
                            },
                            job_params={
                                "job_name_parameter": "VALIDATE_{}_SERVICES_ENV".format(dashboard_group.upper()),
                                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                                "frequency": (jg_utils.TestFrequency.LAZY, None),
                            },
                            job_arrows=(
                                jg_arrows.ParamsData(
                                    input_key=sandbox_constants.ARCADIA_URL_KEY,
                                    transform=lambda x, rm_config: "{svn_ssh_url}/arcadia@{revision}".format(
                                        svn_ssh_url=x.svn_ssh_url,
                                        revision=x.revision,
                                    ),
                                    override=True,
                                ),
                            )
                        )
                    )
                trunk_part.append(
                    jg_test.JobGraphElementTestCommon(
                        task_name="DELIVER_AND_NOTIFY_TELEGRAM",
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="message",
                                transform=lambda x, rm_config: "Trunk build {name} r{revision}".format(
                                    name=rm_config.name,
                                    revision=x.revision,
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="TRUNK",
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="task",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="VALIDATE_TESTING_SERVICES_ENV"
                            )
                        ),
                        job_params={
                            "commit_path_prefixes": ["/trunk/arcadia"],
                            "observed_paths": ["yabs/rmp/backend"],
                            "job_name_parameter": "DELIVER_BINARIES_AUTO",
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        ctx={
                            "notify": True,
                            "people": UacBackendCfg.Notify.Telegram.chats,
                            "component_name": UacBackendCfg.name,
                            "resource_type": list(self.output_build_resources.keys())[0],
                            "use_already_built": True,
                            "env": "testing",
                            "include_commit_info_in_release_message": True,
                        }
                    )
                )
                trunk_part.append(
                    jg_test.JobGraphElementTestCommon(
                        task_name="DELIVER_AND_NOTIFY_TELEGRAM",
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="message",
                                transform=lambda x, rm_config: "Nightly build {name} r{revision}".format(
                                    name=rm_config.name,
                                    revision=x.revision,
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="TRUNK",
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="task",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="VALIDATE_TESTING_SERVICES_ENV"
                            )
                        ),
                        job_params={
                            "job_name_parameter": "DELIVER_BINARIES_NIGHTLY",
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                            "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                time_interval_list=[(7, 7)],
                                weekday_list=list(range(7)),  # every day
                                once_a_day=True,
                            ),
                        },
                        ctx={
                            "notify": True,
                            "people": UacBackendCfg.Notify.Telegram.chats,
                            "component_name": UacBackendCfg.name,
                            "resource_type": list(self.output_build_resources.keys())[0],
                            "use_already_built": True,
                            "env": "testing",
                            "include_commit_info_in_release_message": False,
                        }
                    )
                )
                trunk_part.append(
                    jg_test.JobGraphElementTestCommon(
                        task_name="DEPLOY_BINARY_TASK",
                        out={"SANDBOX_TASKS_BINARY": 30},
                        ctx={
                            "target": "sandbox/projects/rmp/uac/GenerateExtServicesYasmPanels/GenerateExtServicesYasmPanels",
                            "attrs": {"task_cls": "UacBackendGenerateExtServicesYasmPanels"},
                            "use_yt_cache": False,
                        },
                        job_params={
                            "job_name_parameter": "BUILD_GENERATE_EXT_SERVICES_YASM_PANELS_TASK",
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key=sandbox_constants.ARCADIA_URL_KEY,
                                transform=lambda x, rm_config: "{svn_ssh_url}/arcadia@{revision}".format(
                                    svn_ssh_url=x.svn_ssh_url,
                                    revision=x.revision,
                                ),
                                override=True,
                            ),
                        ),
                    )
                )
                trunk_part.append(
                    jg_test.JobGraphElementTestCommon(
                        task_name="UAC_BACKEND_GENERATE_EXT_SERVICES_YASM_PANELS",
                        ctx={
                            "ext_services_base_path": "/rmp/uac/backend/{ctype}/ext_services",
                            "ext_services_alert_prefix": "uac.backend.ext_services",
                            "ctypes": ["test", "prod"],
                            "abc_slug": "mobileuacbackend",
                            "juggler_namespace": "rmp",
                            "juggler_tags": {},
                            "use_last_binary": True,
                        },
                        job_params={
                            "observed_paths": [
                                "yabs/rmp/backend/yasm",
                                "sandbox/projects/rmp/uac/GenerateExtServicesYasmPanels",
                            ],
                            "job_name_parameter": "GENERATE_EXT_SERVICES_YASM_PANELS",
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="BUILD_GENERATE_EXT_SERVICES_YASM_PANELS_TASK",
                            ),
                        ),
                    )
                )
                return trunk_part

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_UAC_BACKEND",
                        ctx={"clear_build": True},
                        out=self.output_build_resources,
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                validate_jobs = {
                    rm_const.ReleaseStatus.testing: "VALIDATE_TESTING_SERVICES_ENV",
                    rm_const.ReleaseStatus.stable: "VALIDATE_PRODUCTION_SERVICES_ENV"
                }
                for release_stage in [rm_const.ReleaseStatus.stable, rm_const.ReleaseStatus.testing]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            task_name="RELEASE_RM_COMPONENT_2",
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "api_server",
                                            "UAC_BACKEND_API_SERVER",
                                        ),
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "celery_worker",
                                            "UAC_BACKEND_CELERY_WORKER",
                                        ),
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "celery_beat",
                                            "UAC_BACKEND_CELERY_BEAT",
                                        ),
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "conf_local_backend_production",
                                            "UAC_BACKEND_CONF_LOCAL_PRODUCTION",
                                        ),
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "conf_local_backend_testing",
                                            "UAC_BACKEND_CONF_LOCAL_TESTING",
                                        ),
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "conf_local_worker_production",
                                            "UAC_WORKER_CONF_LOCAL_PRODUCTION",
                                        ),
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "conf_local_worker_testing",
                                            "UAC_WORKER_CONF_LOCAL_TESTING",
                                        ),
                                    )
                                ),
                                jg_job_triggers.JobTriggerNewTag([
                                    jg_job_data.ParentDataOutput("major_release_num", "branch_number_for_tag"),
                                    jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
                                ]),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter=validate_jobs[release_stage]
                                ),
                            ),
                            job_params={
                                "ctx": {
                                    "deploy_system": rm_const.DeploySystem.nanny.name,
                                }
                            },
                        )
                    )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.testing,
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        }
                    )
                )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_TESTING_RELEASE_UAC_BACKEND": rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

            @property
            def ignore_match(self):
                """List of tests which should be ignored in branch db"""
                return super(self.__class__, self).ignore_match + [
                    "_BUILD__UAC_BACKEND__TRUNK",
                    "_TEST__UAC_BACKEND__EXPORT_CONFIGS",
                    "_TEST__UAC_BACKEND__BUILD_GENERATE_EXT_SERVICES_YASM_PANELS_TASK",
                    "_TEST__UAC_BACKEND__GENERATE_EXT_SERVICES_YASM_PANELS",
                ]

            @property
            def ignore_prefix(self):
                return super(self.__class__, self).ignore_prefix + [
                    "_TEST__UAC_BACKEND__DELIVER_BINARIES_",
                    "_TEST__UAC_BACKEND__BUILD_TOOL_",
                ]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def _backend_production_services(self):
            return [
                ri.DeployService("uac-backend-production-{}".format(loc))
                for loc in rm_const.MAIN_LOCATIONS
            ] + [
                ri.DeployService("uac-backend-production-upload-{}".format(loc))
                for loc in rm_const.MAIN_LOCATIONS
            ]

        @property
        def _worker_production_services(self):
            return [
                ri.DeployService("uac-worker-production-{}".format(loc))
                for loc in rm_const.MAIN_LOCATIONS
            ]

        @property
        def _beat_production_services(self):
            return [
                ri.DeployService("uac-beat-production-{}".format(loc))
                for loc in rm_const.MAIN_LOCATIONS
            ]

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="api_server",
                    data=ri.SandboxResourceData("UAC_BACKEND_API_SERVER"),
                    deploy_infos=[
                        ri.NannyDeployInfo(self._backend_production_services),
                        ri.single_nanny_service("uac-backend-test", stage=rm_const.ReleaseStatus.testing)
                    ]
                ),
                ri.ReleasableItem(
                    name="celery_worker",
                    data=ri.SandboxResourceData("UAC_BACKEND_CELERY_WORKER"),
                    deploy_infos=[
                        ri.NannyDeployInfo(self._worker_production_services),
                        ri.single_nanny_service("uac-worker-test", stage=rm_const.ReleaseStatus.testing)
                    ]
                ),
                ri.ReleasableItem(
                    name="celery_beat",
                    data=ri.SandboxResourceData("UAC_BACKEND_CELERY_BEAT"),
                    deploy_infos=[
                        ri.NannyDeployInfo(self._beat_production_services),
                        ri.single_nanny_service("uac-beat-test", stage=rm_const.ReleaseStatus.testing)
                    ]
                ),
                ri.ReleasableItem(
                    name="conf_local_backend_production",
                    data=ri.SandboxResourceData("UAC_BACKEND_CONF_LOCAL_PRODUCTION"),
                    deploy_infos=[
                        ri.NannyDeployInfo(self._backend_production_services)
                    ]
                ),
                ri.ReleasableItem(
                    name="conf_local_backend_testing",
                    data=ri.SandboxResourceData("UAC_BACKEND_CONF_LOCAL_TESTING"),
                    deploy_infos=[
                        ri.single_nanny_service("uac-backend-test", stage=rm_const.ReleaseStatus.testing)
                    ]
                ),
                ri.ReleasableItem(
                    name="conf_local_worker_production",
                    data=ri.SandboxResourceData("UAC_WORKER_CONF_LOCAL_PRODUCTION"),
                    deploy_infos=[
                        ri.NannyDeployInfo(self._worker_production_services + self._beat_production_services)
                    ]
                ),
                ri.ReleasableItem(
                    name="conf_local_worker_testing",
                    data=ri.SandboxResourceData("UAC_WORKER_CONF_LOCAL_TESTING"),
                    deploy_infos=[
                        ri.NannyDeployInfo(
                            services=[
                                ri.DeployService("uac-worker-test"),
                                ri.DeployService("uac-beat-test")
                            ],
                            stage=rm_const.ReleaseStatus.testing
                        ),
                    ]
                ),
            ]

        wait_for_deploy_time_sec = 3600
        allow_robots_to_release_stable = True

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = [
                "uac-releases@yandex-team.ru",
            ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["rmp_backend_alerts"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            use_task_author_as_assignee = False
            abc_service = MOBILE_UAC_BACKEND_ABC_SERVICE_ID
            assignee = configs.Responsible(
                login="kigan",
                abc=configs.Abc(component_id=MOBILE_UAC_BACKEND_ABC_SERVICE_ID, role_id=ROLE_DUTY)
            )
            queue = "UACBACKREL"
            dev_queue = "RMP"
            summary_template = u"UAC Backend release {}"
            components = "Backend"
            workflow = rm_const.Workflow.BETA_TEST
            add_commiters_as_followers = False
            deadline = 3
            hide_commits_under_cut = True

            def on_release_close_tickets_by_query(self, st_issue_key=None):
                if st_issue_key:
                    return '"Linked To": {key} AND (Status: "RC" OR Status: "Tested" OR Status: "awaitingRelease") AND ({queues})'.format(
                        key=st_issue_key,
                        queues=' OR '.join(['Queue: "{}"'.format(q) for q in (self.dev_queue, 'UC', 'DIRECT')]),
                    )

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "uac/backend/releases/"
        dirs = [
            "arcadia/yabs/rmp/backend",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        use_arc = True
