# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.presets as jg_presets
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class UgcServerCfg(configs.ReferenceBranchedConfig):
    name = "ugc_server"
    responsible = "ivannik"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "ugc-server-trunk"
        trunk_task_owner = "UGC"
        branch_db_template = "ugc-server-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(1, 3)],
                            weekday_list=[1],
                            last_run_finished=False,
                            once_a_day=True
                        )
                    ),
                ]

            _preset = jg_presets.SingleBuildPreset(
                build_task_name="BUILD_UGC_SERVER",
                build_ctx={
                    "ugc_server_release": True,
                    sandbox_constants.CLEAR_BUILD_KEY: True
                },
                deploy_system=rm_const.DeploySystem.nanny.name,
                stages=[
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.stable),
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.prestable, jg_utils.CheckEachCommit()),
                ],
            )

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="SANDBOX_CI_UGC_PAGES_E2E",
                        job_params={
                            "job_name_parameter": "RUN_E2E_TEST",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=lambda x, rm_config: jg_utils.get_major_release_number(x, rm_config),
                            ),
                            jg_job_triggers.JobTriggerActionRelease(
                                job_name_parameter=rm_const.ReleaseStatus.prestable,
                            ),
                        ),
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="UGC_SERVER_FUNCTIONAL_TEST",
                        job_params={
                            "job_name_parameter": "FUNCTIONAL_TEST",
                        }
                    )
                )
                return branch_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_TESTING_RELEASE_UGC_SERVER": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_RUN_UGC_FRONT_E2E_TEST": rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="ugc_server_executable",
                    data=ri.SandboxResourceData("UGC_SERVER_EXECUTABLE", ttl=14),
                    deploy_infos=[ri.single_nanny_service("ugc-server-vla-yp")],
                )
            ]

        release_followers_permanent = [
            "ivannik",
            "ovarene",
            "bartimaeous",
            "ksburaya",
            "qqq",
        ]
        allow_robots_to_release_stable = True

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "ivannik"
            queue = "UGC"
            dev_queue = "UGC"
            summary_template = u"Приемка UGC сервера {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "ivannik",
                "ovarene",
                "bartimaeous",
                "ksburaya",
                "loreley",
                "qqq",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7

            @property
            def banned_people(self):
                return super(self.__class__, self).banned_people | {"zhshishkin"}

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "JandeksPoisk/UGC/DevOps/releases/ugcserver/"
        observed_paths = [
            "arcadia/entity/ugc/data",
            "arcadia/entity/ugc/server",
            "arcadia/entity/ugc/tools",
            "arcadia/entity/ugc/util",
            "arcadia/kernel/ugc",
        ]
        review_groups = [
            "ugc",
        ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "ugc/server"
        tag_name = "ugc/server"
