# -*- coding: utf-8 -*-

from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.voicetech.resource_types import testenv_resources

import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg


class UniproxyCfg(configs.ReferenceBranchedConfig):
    name = "uniproxy"
    responsible = "paxakor"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "VOICESERV_RELEASERS"
        trunk_db = "alice-uniproxy-trunk"
        branch_db_template = "alice-uniproxy-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _trunk_part(self):
                return [
                    # Build
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_DOCKER_IMAGE_V6",
                        build_item="DOCKER_TRUNK",
                        job_params={
                            "observed_paths": [
                                "alice/uniproxy/bin/uniproxy-unistat",
                                "alice/uniproxy/bin/uniproxy-subway",
                                "alice/uniproxy/docker",
                                "alice/uniproxy/library",
                                "alice/uniproxy/server",
                            ],
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                            # "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 3600),
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 2 * (1024 ** 3),  # 2 GB
                                },
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="docker_package_checkout_arcadia_from_url",
                                transform=lambda x, rm_config: "arcadia:/arc/trunk/arcadia@{revision}".format(
                                    revision=x.revision,
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="registry_tags",
                                transform=lambda x, rm_config: ["voicetech/uniproxy:{}.r{}".format(
                                    x.svn_path.split("/arcadia")[0],
                                    x.revision,
                                )]
                            ),
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            "release_type": "testing",
                            "_container": 667332969,
                            "build_type": "release",
                            "docker_build_args": [],
                            "docker_package_arcadia_patch": "",
                            "docker_package_json": "alice/uniproxy/docker/uniproxy/package.json",
                            "registry_login": "robot-voicetechbugs",
                            "vault_item_name": "robot-voicetechbugs-registry-token",
                            "vault_item_owner": "robot-voicetechbugs",
                        },
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_UNIPROXY_BINARY",
                        build_item="BINARY_TRUNK",
                        job_params={
                            "observed_paths": [
                                "alice/uniproxy/bin/uniproxy",
                                "alice/uniproxy/library",
                            ],
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        out={"VOICETECH_UNIPROXY_BINARY": 14},
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_CUTTLEFISH_BINARY",
                        build_item="CUTTLEFISH_TRUNK",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        out={"VOICETECH_CUTTLEFISH_BINARY": 14},
                    ),

                    jg_build.JobGraphElementBuildTrunk(
                        task_name=testenv_resources.VOICETECH_UNIPROXY_EXPERIMENTS.NameBuilderTest,
                        build_item="EXPERIMENTS_TRUNK",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "observed_paths": [
                                "alice/uniproxy/experiments",
                            ],
                            "apiargs": {
                                "requirements": {
                                    # disc space: 15GB (for arcadia src)
                                    "disk_space": 15 * (1024 ** 3),  # 15 Gb
                                },
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            # and42@: build experiments task can not check experiments json at pre_build step
                            # when build from arcadia tag
                            # YaMakeTask yet not checkout uniproxy sources at this moment for this case (tag)
                            "skip_pre_build_check_experiments": False,
                            "release_type": "testing",
                        },
                        out={testenv_resources.VOICETECH_UNIPROXY_EXPERIMENTS.Name: 14},
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_UNIPROXY_PACKAGE",
                        build_item="PACKAGE_TRUNK",
                        filter_targets=[
                            "voicetech/tools/evlog_to_qloud",
                            "voicetech/uniproxy2",
                        ],
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "observed_paths": [
                                "alice/uniproxy/bin/uniproxy",
                                "alice/uniproxy/bin/uniproxy-unistat",
                                "alice/uniproxy/bin/uniproxy-subway",
                                "alice/uniproxy/docker",
                                "alice/uniproxy/library",
                                "alice/uniproxy/server",
                                "voicetech/library/proto_api",
                            ],
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 16 * (1024 ** 3),  # 16 Gb
                                },
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            # BUILD_UNIPROXY_PACKAGE is gen 1 sandbox task, so fill ctx instead custom_fields
                            # import param from gen 2 task params (filled at base class)
                            "build_type": "release",
                            "run_tests": True,
                            "run_medium_tests": True,
                            "run_long_tests": True,
                            "ignore_fail_tests": False,
                            "release_type": "testing",
                            "use_aapi_fuse": True,
                            "use_arc_instead_of_aapi": True,
                            "aapi_fallback": True,
                            "arc_secret": "sec-01eyzmpjy2qcp91f0vkhrqfsx7#arc.token"
                        },
                        out={testenv_resources.VOICETECH_UNIPROXY_PACKAGE.Name: 14},
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_MARKER_TESTS_EXE",
                        build_item="MARKER_TESTS_EXE",
                        job_params={
                            "job_name_parameter": "BUILD_MARKER_TESTS_EXE_NIGHTLY",
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 8 * (1024 ** 3),  # 8 Gb
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            "build_type": "release",
                            "release_type": "testing",
                        },
                        out={"ALICE_MARKER_TESTS_EXE": 14},
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_MARKER_TESTS_CONFIG",
                        build_item="MARKER_TESTS_CONFIG",
                        job_params={
                            "job_name_parameter": "BUILD_MARKER_TESTS_CONFIG_NIGHTLY",
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 1 * (1024 ** 3),  # 1 Gb
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            "uniproxy_websocket_url": "wss://beta.uniproxy.alice.yandex.net/uniproxy-nightly/uni.ws",
                        },
                        out={"ALICE_MARKER_TESTS_CONFIG": 14},
                    ),
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_MARKER_TESTS_DATA",
                        build_item="MARKER_TESTS_DATA",
                        job_params={
                            "job_name_parameter": "BUILD_MARKER_TESTS_DATA_NIGHTLY",
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 8 * (1024 ** 3),  # 8 Gb
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        out={"ALICE_MARKER_TESTS_DATA": 14},
                    ),

                    # Deploy
                    jg_release.JobGraphElementReleaseBase(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=rm_const.ReleaseStatus.testing,
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            # "frequency": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 120),
                            "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                            "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                time_interval_list=[(22, 23)],
                                seconds_from_last_run_commit=1.5 * 3600,
                                once_a_day=True),
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny.name,
                            }
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="EXPERIMENTS_TRUNK",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "uniproxy_experiments",
                                        "VOICETECH_UNIPROXY_EXPERIMENTS",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="PACKAGE_TRUNK",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "uniproxy",
                                        "VOICETECH_UNIPROXY_PACKAGE",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CUTTLEFISH_TRUNK",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cuttlefish",
                                        "VOICETECH_CUTTLEFISH_BINARY"
                                    )
                                )
                            )
                        )
                    ),
                    jg_release.JobGraphElementReleaseBase(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=rm_const.ReleaseStatus.unstable,
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.RUN_N_MINUTES_AFTER_LAST_RUN, 60),
                            "observed_paths": [
                                "alice/cuttlefish",
                                "alice/uniproxy/library/settings",
                                "apphost/conf/backends",
                                "apphost/conf/verticals/VOICE"
                            ],
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny.name,
                            }
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="EXPERIMENTS_TRUNK",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "uniproxy_experiments",
                                        "VOICETECH_UNIPROXY_EXPERIMENTS",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="PACKAGE_TRUNK",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "uniproxy",
                                        "VOICETECH_UNIPROXY_PACKAGE",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CUTTLEFISH_TRUNK",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cuttlefish",
                                        "VOICETECH_CUTTLEFISH_BINARY"
                                    )
                                )
                            )
                        )
                    ),

                    # Tests
                    jg_test.JobGraphElementTestTrunkCommon(
                        task_name="UNIPROXY_PERF_TEST",
                        job_params={
                            "job_name_parameter": "PERF_TEST_QUASAR_NIGHTLY",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="process_id",
                                transform=lambda x, rm_config: "nightly-{revision}".format(
                                    revision=x.revision,
                                ),
                            ),
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="TESTING",
                            ),
                        ),
                        ctx={
                            "app_type": "quasar",
                            "lang": "ru-RU",
                            "topic": "quasar-general",
                            "yt_table": "//home/voice/asr/buckets/feb2020/accept/ya.station",
                            "uniproxy_websocket_url": "wss://beta.uniproxy.alice.yandex.net/uniproxy-nightly/uni.ws"
                        },
                        out={
                            "VOICETECH_UNIPROXY_PERF_TESTER_RESULT": 40,
                        },
                    ),
                    jg_test.JobGraphElementTestTrunkCommon(
                        task_name="UNIPROXY_MARKER_TESTS",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                            "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                time_interval_list=[(22, 23)],
                                seconds_from_last_run_commit=1.5 * 3600,
                                once_a_day=True),
                            "job_name_parameter": "UNIPROXY_MARKER_TESTS_NIGHTLY",
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 8 * (1024 ** 3),  # 8 Gb
                                },
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="BUILD_MARKER_TESTS_CONFIG_NIGHTLY",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        "config_resource_id",  # sandbox task parameter
                                        "ALICE_MARKER_TESTS_CONFIG",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="BUILD_MARKER_TESTS_EXE_NIGHTLY",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        "exe_resource_id",  # sandbox task parameter
                                        "ALICE_MARKER_TESTS_EXE",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="BUILD_MARKER_TESTS_DATA_NIGHTLY",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        "data_resource_id",  # sandbox task parameter
                                        "ALICE_MARKER_TESTS_DATA",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="TESTING",
                            ),
                        ),
                        ctx={
                            "vins_url": "http://vins.alice.yandex.net/speechkit/app/pa/",
                            "checks_limit": 60,
                            "checks_period": 600,  # seconds
                        },
                    ),
                    jg_test.JobGraphElementTestTrunkCommon(
                        task_name="CIT_TEST_VOICE_SERVER",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                            "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                time_interval_list=[(22, 23)],
                                seconds_from_last_run_commit=1.5 * 3600,
                                once_a_day=True),
                            "job_name_parameter": "FUNC_ACCEPTANCE_NIGHTLY",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="TESTING",
                            ),
                        ),
                        ctx={
                            "uniproxy_websocket_url": "wss://beta.uniproxy.alice.yandex.net/uniproxy-nightly/uni.ws",
                            "tests_group": "all",
                            # "release_number": <set later>
                        },

                    ),
                    jg_test.JobGraphElementTestTrunkCommon(
                        task_name="CIT_TEST_VOICE_SERVER",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.RUN_N_MINUTES_AFTER_LAST_RUN, 60),
                            "job_name_parameter": "FUNC_ACCEPTANCE_TRUNK",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="UNSTABLE",
                            ),
                        ),
                        ctx={
                            "uniproxy_websocket_url": "wss://beta.uniproxy.alice.yandex.net/uniproxy-cuttlefish-trunk/uni.ws",
                            "tests_group": "all",
                            # "release_number": <set later>
                        },

                    ),

                ]

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_test.JobGraphElementTestTagCommon(
                        task_name="UNIPROXY_REGRESS_TESTS",
                        job_params={
                            "job_name_parameter": "TESTPALM",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerStartrek(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="target_ticket",
                                        output_key="startrek_issue",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerNewBranch(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_number",
                                        output_key="new_branch_number",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="uniproxy_websocket_url",
                                        output_key="new_branch_number",
                                        transform=lambda x,
                                                         params: "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-stable-{}/uni.ws".format(
                                            x),
                                    ),
                                )
                            ),
                        ),
                        ctx={
                            # "uniproxy_websocket_url": <set by depency>
                            "testsuite": "alice",
                            # "release_number": <set by dependency>
                        }
                    ),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="TESTPALM",
                            )
                        )
                    ),
                ]

            @property
            def _branch_part(self):
                default_branch_part = super(self.__class__, self)._branch_part
                branch_part = [
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_DOCKER_IMAGE_V6",
                        build_item="DOCKER",
                        job_params={
                            "observed_paths": [
                                "alice/uniproxy/bin/uniproxy-unistat",
                                "alice/uniproxy/bin/uniproxy-subway",
                                "alice/uniproxy/docker",
                                "alice/uniproxy/library",
                                "alice/uniproxy/server",
                            ],
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 2 * (1024 ** 3),  # 2 GB
                                },
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=[
                                    jg_job_data.ParentDataOutput(
                                        input_key="docker_package_checkout_arcadia_from_url",
                                        output_key=sandbox_constants.ARCADIA_URL_KEY,
                                    ),
                                ],
                            ),
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="registry_tags",
                                        output_key="result_path",
                                        transform=lambda x, params: ["voicetech/uniproxy:{}".format(
                                            x.split("uniproxy/")[1].split("/arcadia")[0],
                                        )]
                                    ),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            "release_type": "stable",
                            "_container": 667332969,
                            "build_type": "release",
                            "docker_build_args": [],
                            "docker_package_arcadia_patch": "",
                            "docker_package_json": "alice/uniproxy/docker/uniproxy/package.json",
                            "registry_login": "robot-voicetechbugs",
                            "vault_item_name": "robot-voicetechbugs-registry-token",
                            "vault_item_owner": "robot-voicetechbugs",
                        },
                    ),
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name=testenv_resources.VOICETECH_UNIPROXY_EXPERIMENTS.NameBuilderTest,
                        build_item="EXPERIMENTS",
                        job_params={
                            "observed_paths": [
                                "alice/uniproxy/experiments",
                            ],
                            "apiargs": {
                                "requirements": {
                                    # disc space: 15GB (for arcadia src)
                                    "disk_space": 15 * (1024 ** 3),  # 15 Gb
                                },
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            # and42@: build experiments task can not check experiments json at pre_build step
                            # when build from arcadia tag
                            # YaMakeTask yet not checkout uniproxy sources at this moment for this case (tag)
                            "skip_pre_build_check_experiments": True,
                            "release_type": "stable",
                        },
                        out={"VOICETECH_UNIPROXY_EXPERIMENTS": 14},
                    ),
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_UNIPROXY_PACKAGE",
                        build_item="PACKAGE",
                        job_params={
                            "observed_paths": [
                                "alice/uniproxy/bin/uniproxy",
                                "alice/uniproxy/bin/uniproxy-unistat",
                                "alice/uniproxy/bin/uniproxy-subway",
                                "alice/uniproxy/docker",
                                "alice/uniproxy/library",
                                "alice/uniproxy/rtc",
                                "alice/uniproxy/server",
                                "voicetech/library/proto_api",
                            ],
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 8 * (1024 ** 3),  # 8 Gb
                                },
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            # BUILD_UNIPROXY_PACKAGE is gen 1 sandbox task, so fill ctx instead custom_fields
                            # import param from gen 2 task params (filled at base class)
                            "build_type": "release",
                            "run_tests": True,
                            "run_medium_tests": True,
                            "run_long_tests": False,
                            "ignore_fail_tests": False,
                            "release_type": "stable",
                            "use_aapi_fuse": True,
                            "use_arc_instead_of_aapi": True,
                            "aapi_fallback": True,
                            "arc_secret": "sec-01eyzmpjy2qcp91f0vkhrqfsx7#arc.token"
                        },
                        out={testenv_resources.VOICETECH_UNIPROXY_PACKAGE.Name: 14},
                    ),
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_CUTTLEFISH_PACKAGE",
                        build_item="CUTTLEFISH_PACKAGE",
                        job_params={
                            "observed_paths": [
                                "alice/uniproxy/rtc"
                            ],
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 8 * (1024 ** 3),  # 8 Gb
                                },
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            # BUILD_CUTTLEFISH_PACKAGE is gen 1 sandbox task, so fill ctx instead custom_fields
                            # import param from gen 2 task params (filled at base class)
                            "build_type": "release",
                            "run_tests": True,
                            "run_medium_tests": True,
                            "run_long_tests": False,
                            "ignore_fail_tests": False,
                            "release_type": "stable",
                            "use_aapi_fuse": True,
                            "use_arc_instead_of_aapi": True,
                            "aapi_fallback": True,
                            "arc_secret": "sec-01eyzmpjy2qcp91f0vkhrqfsx7#arc.token"
                        },
                        out={testenv_resources.VOICETECH_CUTTLEFISH_PACKAGE.Name: 14},
                    ),

                    jg_build.JobGraphElementBuildBranched(
                        task_name="BUILD_CUTTLEFISH_BINARY",
                        build_item="CUTTLEFISH",
                        job_params={
                            "observed_paths": [
                                "alice/cuttlefish/bin/cuttlefish"
                            ]
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            "release_type": "stable"
                        },

                        out={"VOICETECH_CUTTLEFISH_BINARY": 14},
                    ),

                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_MARKER_TESTS_EXE",
                        build_item="MARKER_TESTS_EXE",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 8 * (1024 ** 3),  # 8 Gb
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        ctx={
                            "build_type": "release",
                            "release_type": "stable",
                        },
                        out={"ALICE_MARKER_TESTS_EXE": 14},
                    ),
                    jg_build.JobGraphElementBuildBranched(
                        task_name="BUILD_MARKER_TESTS_CONFIG",
                        build_item="MARKER_TESTS_CONFIG",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 1 * (1024 ** 3),  # 1 Gb
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                            jg_arrows.ParamsData(
                                input_key="uniproxy_websocket_url",
                                transform=lambda x,
                                                 rm_config: "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-stable-{}/uni.ws".format(
                                    jg_utils.get_major_release_number(x, rm_config),
                                ),
                            ),
                        ),
                        out={"ALICE_MARKER_TESTS_CONFIG": 14},
                    ),
                    jg_build.JobGraphElementBuildBranched(
                        task_name="BUILD_MARKER_TESTS_DATA",
                        build_item="MARKER_TESTS_DATA",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 8 * (1024 ** 3),  # 8 Gb
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="ap_arcadia_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_arrows.ParamsData(
                                input_key="result_rd",
                                transform=jg_utils.construct_resource_description,
                            ),
                        ),
                        out={"ALICE_MARKER_TESTS_DATA": 14},
                    ),
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="uniproxy",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="EXPERIMENTS",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "uniproxy_experiments",
                                        # got from sandbox/projects/release_machine/components/configs/uniproxy.py
                                        "VOICETECH_UNIPROXY_EXPERIMENTS",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="PACKAGE",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "uniproxy",
                                        # got from sandbox/projects/release_machine/components/configs/uniproxy.py
                                        "VOICETECH_UNIPROXY_PACKAGE",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CUTTLEFISH",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cuttlefish",
                                        "VOICETECH_CUTTLEFISH_BINARY",
                                    ),
                                ),
                            ),

                            jg_arrows.ParamsData(
                                input_key="patch_name",
                                transform=lambda x, rm_config: str(jg_utils.get_major_release_number(x, rm_config)),
                            ),
                        ),
                        ctx={"beta_name_source": "STRING"},
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="CIT_TEST_VOICE_SERVER",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "job_name_parameter": "FUNC_ACCEPTANCE",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="uniproxy",
                            ),
                            jg_arrows.ParamsData(
                                input_key="uniproxy_websocket_url",
                                transform=lambda x,
                                                 rm_config: "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-stable-{}/uni.ws".format(
                                    jg_utils.get_major_release_number(x, rm_config),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                        ),
                        ctx={
                            # "uniproxy_websocket_url" : <set later>
                            "tests_group": "all",
                            # "release_number": <set later>
                        }
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ALICE_EVO_CONTEXT_FAST",
                        job_params={
                            "job_name_parameter": "ALICE_EVO_CONTEXT_FAST",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="evo_in_tag_num",
                                        output_key="new_tag_number",
                                    ),
                                )
                            ),
                            jg_arrows.ParamsData("evo_in_branch_num", jg_utils.get_major_release_number)
                        ),
                        ctx={"evo_in_component_name": "uniproxy"}
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name='ALICE_EVO_UE_2_E',
                        job_params={
                            "job_name_parameter": "UNIPROXY_UE2E",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None)
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="ALICE_EVO_CONTEXT_FAST",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="branch_num",
                                        output_key="evo_ctx_branch_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="tag_num",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="release_ticket",
                                        output_key="evo_ctx_release_ticket",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="uniproxy",
                            ),
                            jg_arrows.ParamsData(
                                input_key="uniproxy_url",
                                transform=lambda x,
                                                 rm_config: "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-stable-{}/uni.ws".format(
                                    jg_utils.get_major_release_number(x, rm_config),
                                ),
                            ),
                        ),
                        ctx={
                            "abc_id": 1732,
                            "component_name": "uniproxy"
                        }
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                        job_params={
                            "job_name_parameter": "ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="uniproxy_url",
                                transform=lambda x,
                                                 rm_config: "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-stable-{}/uni.ws".format(
                                    jg_utils.get_major_release_number(x, rm_config),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="beta_name",
                                transform=lambda x,
                                                 rm_config: "alice-uniproxy-stable-{}".format(
                                    jg_utils.get_major_release_number(x, rm_config))
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="uniproxy",
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="ALICE_EVO_CONTEXT_FAST",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="launch_type",
                                        output_key="evo_ctx_component_name",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="branch_number",
                                        output_key="evo_ctx_branch_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="tag_number",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_ticket",
                                        output_key="evo_out_release_ticket",
                                    ),
                                )
                            ),
                        ),
                        ctx={"fail_threshold": 300}
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ALICE_EVO_INTEGRATION_TESTS_WRAPPER",
                        job_params={
                            "job_name_parameter": "ALICE_EVO_INTEGRATION_TESTS_WRAPPER_HAMSTER",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="ALICE_EVO_CONTEXT_FAST",
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="launch_type",
                                        output_key="evo_ctx_component_name",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="branch_number",
                                        output_key="evo_ctx_branch_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="tag_number",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_ticket",
                                        output_key="evo_out_release_ticket",
                                    ),
                                )
                            ),
                        ),
                        ctx={"fail_threshold": 300,
                             "beta_name": "alice-uniproxy-hamster",
                             "uniproxy_url": "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-hamster/uni.ws"}
                    ),

                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="UNIPROXY_MARKER_TESTS",
                            ),
                        ),
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="UNIPROXY_PERF_TEST",
                        job_params={
                            "job_name_parameter": "PERF_TEST_QUASAR",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="process_id",
                                transform=lambda x, rm_config: "stable-{}".format(
                                    jg_utils.get_major_release_number(x, rm_config),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="uniproxy_websocket_url",
                                transform=lambda x,
                                                 rm_config: "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-stable-{}/uni.ws".format(
                                    jg_utils.get_major_release_number(x, rm_config),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="UNIPROXY_MARKER_TESTS",
                            ),
                        ),
                        ctx={
                            "app_type": "quasar",
                            "lang": "ru-RU",
                            "topic": "quasar-general",
                            "yt_table": "//home/voice/asr/buckets/feb2020/accept/ya.station",
                        },
                        out={
                            "VOICETECH_UNIPROXY_PERF_TESTER_RESULT": 40,
                        },
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="UNIPROXY_PERF_TEST_RESULTS_STAT",
                        job_params={
                            "job_name_parameter": "PERF_TEST_QUASAR_STAT",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="PERF_TEST_QUASAR",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        "perf_tester_results_resource",  # sandbox task parameter
                                        "VOICETECH_UNIPROXY_PERF_TESTER_RESULT",
                                    ),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="process_id",
                                transform=lambda x, rm_config: "stable-{}".format(
                                    jg_utils.get_major_release_number(x, rm_config),
                                ),
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                        ),
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="UNIPROXY_MARKER_TESTS",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "job_name_parameter": "UNIPROXY_MARKER_TESTS",
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 8 * (1024 ** 3),  # 8 Gb
                                },
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="MARKER_TESTS_CONFIG",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        "config_resource_id",  # sandbox task parameter
                                        "ALICE_MARKER_TESTS_CONFIG",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="MARKER_TESTS_EXE",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        "exe_resource_id",  # sandbox task parameter
                                        "ALICE_MARKER_TESTS_EXE",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="MARKER_TESTS_DATA",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        "data_resource_id",  # sandbox task parameter
                                        "ALICE_MARKER_TESTS_DATA",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="FUNC_ACCEPTANCE",
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                        ),
                        ctx={
                            "vins_url": "http://vins.alice.yandex.net/speechkit/app/pa/",
                            "checks_limit": 60,
                            "checks_period": 600,  # seconds
                        },
                    ),
                ]
                return default_branch_part + branch_part

            @property
            def _release(self):
                default_release_part = super(self.__class__, self)._release
                all_build_job_arrows = (
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="EXPERIMENTS",
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                "uniproxy_experiments",
                                "VOICETECH_UNIPROXY_EXPERIMENTS",
                            ),
                        ),
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="PACKAGE",
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                "uniproxy",
                                "VOICETECH_UNIPROXY_PACKAGE",
                            ),
                        ),
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="CUTTLEFISH_PACKAGE",
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                "cuttlefish_package",
                                "VOICETECH_CUTTLEFISH_PACKAGE",
                            ),
                        ),
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="CUTTLEFISH",
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                "cuttlefish",
                                "VOICETECH_CUTTLEFISH_BINARY"
                            ),
                        ),
                    ),
                )
                all_builds_releases = []
                for release_to in [
                    rm_const.ReleaseStatus.stable,
                    rm_const.ReleaseStatus.prestable,
                ]:
                    all_builds_releases.extend([
                        jg_release.JobGraphElementReleaseBranched(
                            task_name="RELEASE_RM_COMPONENT_2",
                            release_to=release_to,
                            job_params={
                                "ctx": {
                                    "deploy_system": rm_const.DeploySystem.nanny.name,
                                }
                            },
                            job_arrows=all_build_job_arrows + (
                                jg_job_triggers.JobTriggerNewTag([
                                    jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                                    jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                                ]),
                            )
                        ),
                        jg_release.JobGraphElementActionReleaseBranched(release_to=release_to),
                    ])
                stable_releases_separate_builds = [
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="EXPERIMENTS",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="EXPERIMENTS",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "uniproxy_experiments",
                                        "VOICETECH_UNIPROXY_EXPERIMENTS",
                                    ),
                                ),
                            ),
                        ),
                    ),
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="EXPERIMENTS",
                    ),
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="PACKAGE",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="PACKAGE",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "uniproxy",
                                        "VOICETECH_UNIPROXY_PACKAGE",
                                    ),
                                ),
                            ),
                        ),
                    ),
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="PACKAGE",
                    ),
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="CUTTLEFISH_PACKAGE",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CUTTLEFISH_PACKAGE",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cuttlefish_package",
                                        "VOICETECH_CUTTLEFISH_PACKAGE",
                                    ),
                                ),
                            ),
                        ),
                    ),
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="CUTTLEFISH_PACKAGE",
                    ),
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="CUTTLEFISH",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CUTTLEFISH",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "cuttlefish",
                                        "VOICETECH_CUTTLEFISH_BINARY",
                                    ),
                                ),
                            ),
                        ),
                    ),
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="CUTTLEFISH",
                    ),
                ]

                return default_release_part + stable_releases_separate_builds + all_builds_releases

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):

            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                uppercase_name = self.name.upper()
                tests.update({
                    "_{}_UMBRELLA_ACCEPT_SCHEDULED".format(uppercase_name): rm_const.TestFrequencies.FORTY_TWO_TEST,
                })
                return tests

            @property
            def ignore_match(self):
                common_ignored = super(self.__class__, self).ignore_match
                return common_ignored + [
                    "_TEST__UNIPROXY__TESTPALM",  # this test need only for prerelease
                    "_BUILD__UNIPROXY__BINARY_TRUNK",
                    "_BUILD__UNIPROXY__DOCKER_TRUNK",
                    "_BUILD__UNIPROXY__EXPERIMENTS_TRUNK",
                    "_BUILD__UNIPROXY__PACKAGE_TRUNK",
                    "_BUILD__UNIPROXY__CUTTLEFISH_TRUNK",
                    "_BUILD__UNIPROXY__BUILD_MARKER_TESTS_CONFIG_NIGHTLY",
                    "_BUILD__UNIPROXY__BUILD_MARKER_TESTS_DATA_NIGHTLY",
                    "_BUILD__UNIPROXY__BUILD_MARKER_TESTS_EXE_NIGHTLY",
                    "_TEST__UNIPROXY__PERF_TEST_QUASAR_NIGHTLY",
                    "_TEST__UNIPROXY__MARKER_TESTS_NIGHTLY",
                    "_TEST__UNIPROXY__FUNC_ACCEPTANCE_NIGHTLY",
                    "_TEST__UNIPROXY__FUNC_ACCEPTANCE_TRUNK",
                    "_TEST__UNIPROXY__UNIPROXY_MARKER_TESTS_NIGHTLY",
                    "_RELEASE__UNIPROXY__TESTING",
                    "_RELEASE__UNIPROXY__UNSTABLE",
                ]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):  # type: () -> List[ri.AbstractReleasableItem]
            deploy_infos = [
                ri.single_nanny_service("wsproxy-man"),
                ri.single_nanny_service("uniproxy2_man", stage=rm_const.ReleaseStatus.prestable),
                ri.single_nanny_service("voice-proxy-trunk", stage=rm_const.ReleaseStatus.testing),
                ri.single_nanny_service("voice-cuttlefish-trunk", stage=rm_const.ReleaseStatus.unstable),
            ]
            return [
                ri.ReleasableItem(
                    name="uniproxy",
                    data=ri.SandboxResourceData("VOICETECH_UNIPROXY_PACKAGE"),
                    deploy_infos=deploy_infos,
                ),
                ri.ReleasableItem(
                    name="uniproxy_experiments",
                    data=ri.SandboxResourceData("VOICETECH_UNIPROXY_EXPERIMENTS"),
                    deploy_infos=deploy_infos,
                ),
                ri.ReleasableItem(
                    name="cuttlefish",
                    data=ri.SandboxResourceData("VOICETECH_CUTTLEFISH_BINARY"),
                    deploy_infos=deploy_infos,
                ),
                ri.ReleasableItem(
                    name="cuttlefish_package",
                    data=ri.SandboxResourceData("VOICETECH_CUTTLEFISH_PACKAGE"),
                    deploy_infos=deploy_infos,
                ),
                ri.DynamicReleasableItem(
                    name="scheduled_bundle",
                    data=ri.SandboxResourceData("SCHEDULED_RM_RELEASE_DATA"),
                    deploy_infos=[ri.NannyDeployInfo(), ri.NannyDeployInfo(stage=rm_const.ReleaseStatus.prestable)],
                )
            ]

        wait_for_deploy_time_sec = 3600
        allow_robots_to_release_stable = True

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = [
                "lyalchenko@yandex-team.ru",
                "paxakor@yandex-team.ru",
            ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = []
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = "https://t.me/joinchat/CRF5o0R2B0P1wyfBKvK-Kw"

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "paxakor"
            queue = "ALICERELEASE"
            dev_queue = "ALICE"
            summary_template = u"Релиз uniproxy и/или его конфигурационных данных (экспериментов) {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "av-kotikov",
                "lyalchenko",
                "paxakor",
            ]
            add_commiters_as_followers = False
            deadline = 7
            hide_commits_under_cut = True
            important_changes_limit = 100

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "voicetechnology/dev/serverinfr/uniproxy/releases/"
        dirs = [
            "arcadia/alice/cuttlefish/library/cuttlefish/common",
            "arcadia/alice/cuttlefish/library/protos",
            "arcadia/alice/cuttlefish/library/proto_converters",
            "arcadia/alice/cuttlefish/library/surface_mapper",
            "arcadia/alice/uniproxy",
            "arcadia/voicetech/library/proto_api",
            "arcadia/voicetech/library/uniproxy2",
            "arcadia/voicetech/library/uniproxy2/resources",
            "arcadia/voicetech/library/ws_server",
            "arcadia/voicetech/uniproxy2",
        ]

        # TODO: wheh will be availabale review_group *uniproxy* - place here
        # review_groups = [
        #    "voicetech-infra",
        # ]

        @staticmethod
        def calculate_importance(changelog_entry):
            if changelog_entry.st_tickets():
                return 5

            return changelog_entry.mark

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "voicetech/uniproxy"

    class Yappy(yappy_cfg.YappyBaseCfg):
        betas = {
            "uniproxy": yappy_cfg.YappyTemplateCfg(
                template_name="alice-uniproxy-stable",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="alice-uniproxy-stable",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(
                                param_name="uniproxy",
                                local_path="uniproxy-package",
                            ),
                            yappy_cfg.YappyParametrizedResource(
                                param_name="uniproxy_experiments",
                                local_path="exps_pack"
                            ),
                            yappy_cfg.YappyParametrizedResource(
                                param_name="cuttlefish",
                                local_path="cuttlefish"
                            ),
                            yappy_cfg.YappyStaticResource(
                                manage_type="NO_MANAGE",
                                local_path="app_host"
                            )
                        ],
                        parent_service="wsproxy-vla",
                        ignore_instance_spec=False,
                    )
                ],
            ),
        }
