# -*- coding: utf-8 -*-

import itertools
import re
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.common.constants import constants as common_const
import sandbox.projects.release_machine.components.config_core.statistics_page as statistics_page
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.helpers.arcadia as rm_arc
from sandbox.projects.release_machine.components.configs.report import SearchIntergationTestBundle


_NOT_GRANTED_MESSAGE_FORMAT = """Merger {} is not allowed to merge into {}.
Please, visit noapacheupper releases chat: https://t.me/joinchat/CaUODkNMIpwWNzLu4pngow
or contact current release engineer on duty: https://abc.yandex-team.ru/services/upper/duty/
"""


class UpperCfg(configs.ReferenceBranchedConfig):
    name = "upper"
    responsible = "elshiko"

    class MergesCfg(configs.ReferenceBranchedConfig.MergesCfg):
        permissions = configs.MergePermissions(
            permission_type=rm_const.PermissionType.ALLOWED,
            people_groups=configs.PeopleGroups(
                staff_groups=None,
                # https://abc.yandex-team.ru/services/upper/
                abc_services=[configs.Abc(component_id=1609, role_id=None)],
                logins=None,
            ),
        )

        def not_granted_message(self, author, responsible):
            return _NOT_GRANTED_MESSAGE_FORMAT.format(
                author, self.name,
            )

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "noapacheupper-trunk"
        trunk_task_owner = "NOAPACHEUPPER-ROBOT"
        branch_db_template = "noapacheupper-stable-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            default_yamake_task = 'KOSHER_YA_MAKE'

            @property
            def _trunk_part(self):
                trunk_part = super(self.__class__, self)._trunk_part

                trunk_part.append(
                    jg_test.JobGraphElementRunIceFlame(
                        config_file="noapache_80.conf",
                        job_params={
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                            "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                time_interval_list=[(7, 9)],
                                seconds_from_last_run_commit=1.5 * 3600,
                                once_a_day=True,
                            ),
                        },
                        ctx={
                            "release_item_name": "upper",
                            "service_name": "production_noapache_sas_web_yp",
                            "fail_if_service_version_cannot_be_determined": True,
                            "check_production_version": True,
                        },
                    ),
                )

                return trunk_part

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name='BUILD_NOAPACHE_UPPER',
                        job_params={
                            "job_name_parameter": "RELEASE",
                        },
                        ctx={
                            'search_projects': 'build_all',
                            common_const.CLEAR_BUILD_KEY: False,
                            common_const.BUILD_SYSTEM_KEY: common_const.DISTBUILD_BUILD_SYSTEM,
                            common_const.USE_ARC_INSTEAD_OF_AAPI: True,
                            common_const.USE_AAPI_FUSE: True,
                            common_const.ALLOW_AAPI_FALLBACK: True,
                            common_const.BUILD_TYPE_KEY: common_const.RELEASE_BUILD_TYPE,
                            common_const.THINLTO: True,
                            common_const.DEFINITION_FLAGS_KEY: '-DNN_NO_OPENMP',
                        },
                        out={
                            'NOAPACHE_UPPER': 90,
                            'EVLOGDUMP_EXECUTABLE': 90,
                        },
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name='BUILD_REARRANGE',
                        job_params={
                            "job_name_parameter": "RELEASE_REARRANGE",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                jg_job_data.ParentDataCtx(
                                    input_key="rearrange_svn_url",
                                    output_key="result_path",
                                    transform=(
                                        lambda x, params:
                                        x.split("@")[0].strip("/") + "/arcadia/search/web/rearrs_upper/rearrange"
                                    ),
                                ),
                            )
                        ),
                        ctx={
                            'trivial_naming': True,
                            common_const.USE_AAPI_FUSE: True,
                            common_const.ALLOW_AAPI_FALLBACK: True,
                        },
                        out={
                            "REARRANGE_DATA": 90,
                        },
                    )
                )

                class ChartParams(jg_utils.ChartParams):
                    ChartLines = ('directory_size', 'bundle_size')
                    ChartYAxes = 'MiB'
                    UsePointComments = False

                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name='BUILD_REARRANGE_DYNAMIC',
                        job_params={
                            "job_name_parameter": "RELEASE_REARRANGE_DYNAMIC_FOR_NOAPACHE",
                            "get_data_from_task": lambda params: {
                                'directory_size': (
                                    int(params.ctx['directory_size']) if 'directory_size' in params.ctx else None
                                ),
                                'bundle_size': int(params.ctx['bundle_size']) if 'bundle_size' in params.ctx else None,
                            },
                            "chart": ChartParams(),
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        ctx={
                            'use_archiver': True,
                            common_const.USE_AAPI_FUSE: True,
                            common_const.ALLOW_AAPI_FALLBACK: True,
                        },
                        out={
                            'REARRANGE_DYNAMIC_DATA': 90,
                        },
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name='BUILD_UPPER_CONFIG',
                        job_params={
                            "job_name_parameter": "RELEASE_RTCC_BUNDLE",
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        ctx={
                            'config_type': 'noapache',
                            'bundle_type': 'production_noapache',
                        },
                        out={
                            'RTCC_BUNDLE_NOAPACHE': 90,
                            'RTCC_CACHE_NOAPACHE': 90,
                        },
                    )
                )
                for vertical in ["web", "video"]:
                    branch_part.append(
                        jg_build.JobGraphElementBuildBranched(
                            task_name='BUILD_REARRANGE_DATA_FAST',
                            job_params={
                                "job_name_parameter": "RELEASE_REARRANGE_DATA_FAST_{}".format(vertical.upper()),
                                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            },
                            job_arrows=(
                                jg_arrows.ParentsData(
                                    input_key="arcadia_url",
                                    triggers=(
                                        jg_job_triggers.JobTriggerNewTag(
                                            parent_job_data=(
                                                jg_job_data.ParentDataCtx(
                                                    input_key="arc_url",
                                                    output_key="result_path",
                                                ),
                                                jg_job_data.ParentDataCtx(
                                                    input_key="arc_rev",
                                                    output_key="result_revision",
                                                ),
                                            ),
                                        ),
                                    ),
                                    transform=rm_arc.construct_arc_url,
                                )
                            ),
                            ctx={
                                'vertical': vertical,
                                'arc_token_owner': 'NOAPACHEUPPER-ROBOT'
                            },
                            out={
                                'REARRANGE_DATA_FAST': 90,
                            },
                        )
                    )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="BUILD_CONFIGS_WITH_PVA",
                        job_params={
                            "job_name_parameter": "RELEASE_NOAPACHE_INSTANCECTL_CONF",
                        },
                        job_arrows=(
                            jg_arrows.GlobalResourceData(
                                input_key="pva",
                                resource_name="APPHOST_PVA",
                            ),
                            jg_arrows.ParentsData(
                                input_key="arcadia_url",
                                triggers=(
                                    jg_job_triggers.JobTriggerNewTag(
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="arc_url",
                                                output_key="result_path",
                                            ),
                                            jg_job_data.ParentDataCtx(
                                                input_key="arc_rev",
                                                output_key="result_revision",
                                            ),
                                        ),
                                    ),
                                ),
                                transform=rm_arc.construct_arc_url,
                            ),
                        ),
                        out={
                            'NOAPACHE_PRODUCTION_INSTANCECTL_CONF': 90,
                            'NOAPACHE_HAMSTER_INSTANCECTL_CONF': 90,
                            'NOAPACHE_PUSHCLIENT_PRODUCTION_CONF': 90,
                            'NOAPACHE_PUSHCLIENT_HAMSTER_CONF': 90,
                        },
                        ctx={
                            "config_types": '\n'.join([
                                'NOAPACHE_PRODUCTION_INSTANCECTL_CONF',
                                'NOAPACHE_HAMSTER_INSTANCECTL_CONF',
                                'NOAPACHE_PUSHCLIENT_PRODUCTION_CONF',
                                'NOAPACHE_PUSHCLIENT_HAMSTER_CONF',
                            ]),
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="COMPARE_MIDDLESEARCH_BINARIES",
                        job_params={
                            "job_name_parameter": "BLENDER_PERFORMANCE_VS_PROD",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="noapacheupper_2_executable_resource_id",
                                        resource_name="NOAPACHE_UPPER",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="noapacheupper_2_evlogdump_resource_id",
                                        resource_name="EVLOGDUMP_EXECUTABLE",
                                    ),
                                ),
                                job_name_parameter="RELEASE",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="rearrange_data_2_resource_id",
                                        resource_name="REARRANGE_DATA",
                                    ),
                                ),
                                job_name_parameter="RELEASE_REARRANGE",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="rearrange_data_fast_2_resource_id",
                                        resource_name="REARRANGE_DATA_FAST",
                                    ),
                                ),
                                job_name_parameter="RELEASE_REARRANGE_DATA_FAST_WEB",
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                            jg_arrows.GlobalResourceData(
                                input_key="noapacheupper_1_evlogdump_resource_id",
                                resource_name="NOAPACHEUPPER_EVLOGDUMP",
                            ),
                            jg_arrows.GlobalResourceData(
                                input_key="noapache_1_requests_resource_id",
                                resource_name="BLENDER_GRPC_CLIENT_PLAN",
                            ),
                            jg_arrows.GlobalResourceData(
                                input_key="noapache_2_requests_resource_id",
                                resource_name="BLENDER_GRPC_CLIENT_PLAN",
                            ),
                        ),
                        ctx={
                            "launch_mode": self.name,
                            "req_count": 15000,
                            "calc_mode": "blender",
                            "auto_mode": "nocache",
                            "enable_auto_diffs": False,
                            "fail_on_slowdown_threshold_q95": 500,
                            "fail_on_rearrange_slowdown_threshold_q95": 500,
                            "nruns": 6,  # temp lightweight
                            "noapacheupper_1_max_calc_relev_queue_size": 0,
                            "noapacheupper_2_max_calc_relev_queue_size": 0,
                            "use_new_calc_eventlog_stats": True,
                            "noapacheupper_1_config_debug_options": "LogPrepareSourcesTimings=true",
                            "noapacheupper_2_config_debug_options": "LogPrepareSourcesTimings=true",
                        },
                    )
                )
                UPPER_METRICS_SLA_PROJECT = "aa8286386850f0df016874cc48882d44"
                for vertical in ["web", "video", "images"]:
                    branch_part.append(
                        jg_test.JobGraphElementYappyBetaGeneratorBranched(
                            beta_conf_type=vertical,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "noapacheupper_executable_res_id",
                                            "NOAPACHE_UPPER",
                                        ),
                                    ),
                                    job_name_parameter="RELEASE",
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "rearrange_data_res_id",
                                            "REARRANGE_DATA",
                                        ),
                                    ),
                                    job_name_parameter="RELEASE_REARRANGE",
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "noapache_rtcc_bundle_res_id",
                                            "RTCC_BUNDLE_NOAPACHE",
                                        ),
                                    ),
                                    job_name_parameter="RELEASE_RTCC_BUNDLE",
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            'hamster_instancectl_conf_res_id',
                                            'NOAPACHE_HAMSTER_INSTANCECTL_CONF',
                                        ),
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            'pushclient_hamster_conf_res_id',
                                            'NOAPACHE_PUSHCLIENT_HAMSTER_CONF',
                                        ),
                                    ),
                                    job_name_parameter="RELEASE_NOAPACHE_INSTANCECTL_CONF",
                                )
                            ),
                            ctx={'get_beta_names_from_state': True}
                        )
                    )
                    branch_part.append(
                        jg_test.JobGraphElementLaunchMetrics(
                            search_subtype=vertical,
                            job_arrows=(
                                jg_job_triggers.JobTriggerGenerateBeta(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key='checked_beta',
                                            output_key='new_beta_url',
                                        ),
                                    ),
                                    job_name_parameter=vertical.upper(),
                                ),
                            ),
                            ctx={
                                "sample_beta": "upper-hamster.hamster",
                                "sla_project": UPPER_METRICS_SLA_PROJECT,
                                "scraper_over_yt_pool": "upper_{}_priemka".format(vertical),
                                "run_findurl": vertical != "web",
                            },
                        )
                    )
                branch_part.append(
                    jg_test.JobGraphElementTestTdiAndRearr(
                        job_params={
                            "ctx": {
                                'launch_tdi': False,
                                'launch_RA_for_blender_and_video': False,
                                'launch_new_RA_for_blender_and_video': True,
                                'launch_new_RA_for_blender_on_touch': True,
                                'launch_new_RA_for_blender_on_pad': True,
                                'launch_personalization_RA2': True,
                                'launch_new_RA': True,
                                'launch_images_RA': True,
                                'launch_video_RA': True,
                                "launch_videoserp_RA": True,
                                'seek_queries': True,
                                'seek_two_contexts_features_diff_binary': True,
                                'sample_beta': 'upper-hamster.hamster',
                                'notify_if_failed': '',  # we use telegram notifications ans release tickets
                                'ignore_passed': True,
                                "scraper_over_yt_pool": "upper_web_priemka",
                                "max_exec_children": 15,  # Experimental thing to speed up scraper parallel working
                            },
                            "job_name_parameter": "SCRAPER",
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestXMLSearch(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="beta_url",
                                        output_key="new_beta_url",
                                        transform=lambda x, params: "https://{}.ru".format(x),
                                    ),
                                ),
                                job_name_parameter="WEB",
                            )
                        ),
                        ctx={
                            "shoots_number": 50,
                            "validate_beta": True,
                            "add_cgi": "&numdoc=10",
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="SITESEARCH_INTEGRATION_TEST",
                        job_params={
                            "job_name_parameter": "SITESEARCH_INTEGRATION_TEST",
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                            "apiargs": {
                                "kill_timeout": 3 * 3600,
                                "requirements": {
                                    "platform": "linux_ubuntu_16.04_xenial",
                                },
                            },
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="WEB",
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="serp_host",
                                        output_key="new_beta_url",
                                        transform=lambda x, params: "{}.ru".format(x),
                                    )
                                )
                            )
                        ),
                        ctx={
                            "description": (
                                "SitesearchTests for upper release. "
                                "Responsible from fronend sitesearch: https://abc.yandex-team.ru/services/wmcon/duty/"
                            ),
                            "git_reference": "dev"
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementE2ETest(
                        job_arrows=jg_job_triggers.JobTriggerGenerateBeta(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key="hermionee2e_base_url",
                                    output_key="new_beta_url",
                                    transform=lambda x, params: "https://{}.ru".format(x),
                                ),
                            ),
                            job_name_parameter="WEB",
                        ),
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name='BUILD_RTCC_DIFF',
                        job_params={
                            "job_name_parameter": "RTCC_DIFF",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="production_noapache_new_bundle",
                                        resource_name="RTCC_BUNDLE_NOAPACHE",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="production_noapache_new_cache",
                                        resource_name="RTCC_CACHE_NOAPACHE",
                                    ),
                                ),
                                job_name_parameter="RELEASE_RTCC_BUNDLE",
                            ),
                            jg_job_triggers.JobTriggerNewTag(
                                jg_job_data.ParentDataCtx(
                                    input_key=common_const.ARCADIA_URL_KEY,
                                    output_key="result_path",
                                    transform=lambda x, params: '{}/arcadia'.format(x),
                                )
                            ),
                        ),
                        ctx={
                            "ctype": "production_noapache",
                        },
                        out={'RTCC_BRIEF_DIFF': 90},
                    )
                )
                RD_TEST_PATHS = [
                    "search/web/rearrs_upper/tests",
                    "search/web/rearrs_upper/rearrange.dynamic",
                ]
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name='YA_MAKE',
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "ram": 24 * 1024 ** 3,  # in Bytes
                                    "disk_space": 80 * (1024 ** 3),  # in Bytes
                                },
                            },
                            "job_name_parameter": "LAUNCH_ARCADIA_TESTS_FOR_REARRANGE_DYNAMIC",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key=common_const.ARCADIA_URL_KEY,
                                        output_key="result_path",
                                        transform=lambda x, params: '{}/arcadia'.format(x),
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            'targets': ';'.join(RD_TEST_PATHS),
                            'clear_build': False,
                            'test': True,
                            'report_tests_only': True,
                            'disable_test_timeout': True,
                            'cache_test_results': False,
                            'tests_retries': 2,
                        },
                    )
                )
                for t in SearchIntergationTestBundle.get_bundle("web", soy_pool="upper_web_priemka"):
                    branch_part.append(t)

                umbrella_parents = (
                    jg_job_triggers.JobTriggerLaunchMetrics(
                        job_name_parameter="WEB",
                    ),
                    jg_job_triggers.JobTriggerLaunchMetrics(
                        job_name_parameter="IMAGES",
                    ),
                    jg_job_triggers.JobTriggerLaunchMetrics(
                        job_name_parameter="VIDEO",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type=rm_const.JobTypes.TEST_ALL_RA2,
                        job_name_parameter="SCRAPER",
                    ),
                    jg_job_triggers.JobTriggerTestXMLSearch(),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="SITESEARCH_INTEGRATION_TEST",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type=rm_const.JobTypes.TEST_E2E,
                        job_name_parameter="WEB",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="LAUNCH_ARCADIA_TESTS_FOR_REARRANGE_DYNAMIC",
                    ),
                    jg_job_triggers.JobTriggerMetaTest(
                        job_name_parameter="SEARCH_INTEGRATION_TEST",
                    ),
                    jg_job_triggers.JobTriggerMetaTest(
                        job_name_parameter="SEARCH_INTEGRATION_TEST_SOY_HTTP",
                    ),
                    jg_job_triggers.JobTriggerMetaTest(
                        job_name_parameter="SEARCH_INTEGRATION_TEST_STYLE",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="EMPTY_EXPERIMENT",
                    ),
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=umbrella_parents,
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                        job_arrows=umbrella_parents,
                    )
                )

                # In stable noapacheupper branches we need binary and rearrange data
                # built from branch tag and last stable released rearrange.dynamic
                # (i.e. taken from production).
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="CHECK_UPPER_EXP",
                        job_params={
                            "job_name_parameter": "EMPTY_EXPERIMENT",
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="rearrange_data",
                                        resource_name="REARRANGE_DATA",
                                    ),
                                ),
                                job_name_parameter="RELEASE_REARRANGE",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="executable",
                                        resource_name="NOAPACHE_UPPER"
                                    )
                                ),
                                job_name_parameter="RELEASE",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="WEB",
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="config_source",
                                        output_key="new_beta_url",
                                        transform=lambda x, params: "{}.ru".format(x),
                                    )
                                )
                            ),
                        ),
                        ctx={
                            "new_cgi_param": "",
                            "scraper_pool": "upper_web_priemka",
                        },
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                build_resources_info = (
                    jg_job_triggers.JobTriggerBuild(
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                'noapacheupper_executable_res_id',
                                'NOAPACHE_UPPER',
                            ),
                        ),
                        job_name_parameter="RELEASE",
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                'rearrange_data_res_id',
                                'REARRANGE_DATA',
                            ),
                        ),
                        job_name_parameter="RELEASE_REARRANGE",
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                'component_resources',
                                'rearrange_data_fast_web_res_id',
                                'REARRANGE_DATA_FAST',
                            ),
                        ),
                        job_name_parameter="RELEASE_REARRANGE_DATA_FAST_WEB",
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                'rearrange_data_fast_video_res_id',
                                'REARRANGE_DATA_FAST',
                            ),
                        ),
                        job_name_parameter="RELEASE_REARRANGE_DATA_FAST_VIDEO",
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                'production_instancectl_conf_res_id',
                                'NOAPACHE_PRODUCTION_INSTANCECTL_CONF',
                            ),
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                'hamster_instancectl_conf_res_id',
                                'NOAPACHE_HAMSTER_INSTANCECTL_CONF',
                            ),
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                'pushclient_production_conf_res_id',
                                'NOAPACHE_PUSHCLIENT_PRODUCTION_CONF',
                            ),
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                'pushclient_hamster_conf_res_id',
                                'NOAPACHE_PUSHCLIENT_HAMSTER_CONF',
                            ),
                        ),
                        job_name_parameter="RELEASE_NOAPACHE_INSTANCECTL_CONF",
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        parent_job_data=(
                            jg_job_data.ParentDataDict(
                                "component_resources",
                                "noapache_rtcc_bundle_res_id",
                                "RTCC_BUNDLE_NOAPACHE",
                            ),
                        ),
                        job_name_parameter="RELEASE_RTCC_BUNDLE",
                    ),
                )
                for release_stage in [rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable]:
                    release_part.extend([
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            wait_for_deploy=release_stage == rm_const.ReleaseStatus.stable,
                            job_arrows=build_resources_info,
                        ),
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                        ),
                    ])
                release_part.append(
                    jg_release.JobGraphElementReleasePushBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=build_resources_info,
                        wait_for_deploy=True,
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionPushReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        'noapacheupper_executable_res_id',
                                        'NOAPACHE_UPPER',
                                    ),
                                ),
                                job_name_parameter="RELEASE",
                            ),
                        ),
                        wait_for_deploy=True,
                        release_item="ONLY_SEARCH",
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="ONLY_SEARCH",
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="SEARCH_AND_REARRANGE",
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        'noapacheupper_executable_res_id',
                                        'NOAPACHE_UPPER',
                                    ),
                                ),
                                job_name_parameter="RELEASE",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        'rearrange_data_res_id',
                                        'REARRANGE_DATA',
                                    ),
                                ),
                                job_name_parameter="RELEASE_REARRANGE",
                            ),
                        ),
                        wait_for_deploy=True,
                        release_item="SEARCH_AND_REARRANGE",
                    )
                )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_BUILD_RELEASE_REARRANGE": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_REARRANGE_DYNAMIC": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BUILD_RELEASE_NOAPACHE_INSTANCECTL_CONF": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_BLENDER_PERFORMANCE_VS_PROD": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_UPPER_RTCC_DIFF": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_LAUNCH_ARCADIA_TESTS_FOR_REARRANGE_DYNAMIC": rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

            @property
            def ignore_match(self):
                common_ignored = super(self.__class__, self).ignore_match
                return common_ignored + [
                    "UPDATE_NOAPACHEUPPER2_RESOURCES",
                    "UPDATE_NOAPACHEUPPER_YASM_PANELS",
                    "PREPARE_WEB_BLENDER_FUZZY_INPUT",
                    "WEB_BLENDER_FUZZY",
                    "BUILD_BLENDER_FUZZER",
                    "BLENDER_PERFORMANCE",
                    "CHECK_NOAPACHEUPPER_EMPTY_EXPERIMENT",
                    "BLENDER_PERFORMANCE_VS_PROD",
                ]

    class ReleaseViewer(configs.ReferenceConfig.ReleaseViewer):
        kpi_alert = 4
        kpi_alert_skip_weekends = True
        statistics_page_charts = statistics_page.DEFAULT + [statistics_page.PredefinedCharts.TIMESPECTRE_METRICS]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        allow_robots_to_release_stable = True
        allow_old_releases = True

        @property
        def releasable_items(self):
            services = [
                (
                    "web",
                    (
                        [
                            ri.DeployService("production_noapache_{}_web_yp".format(i), ["prod", "web"])
                            for i in rm_const.MAIN_LOCATIONS
                        ] + [
                            ri.DeployService("hamster_noapache_{}_web_yp".format(i), ["hamster", "web"])
                            for i in rm_const.MAIN_LOCATIONS
                        ] + [
                            ri.DeployService("prestable_noapache_sas_web_yp", ["prestable", "web"])
                        ]
                    ),
                ),
                (
                    "imgs",
                    (
                        [
                            ri.DeployService("production_noapache_{}_imgs_rkub".format(i), ["prod", "imgs"])
                            for i in rm_const.MAIN_LOCATIONS
                        ] + [
                            ri.DeployService("hamster_noapache_{}_imgs".format(i), ["hamster", "imgs"])
                            for i in rm_const.MAIN_LOCATIONS
                        ]
                    )
                ),
                (
                    "video",
                    (
                        [
                            ri.DeployService("production_noapache_{}_video_rkub".format(i), ["prod", "video"])
                            for i in ["sas", "vla"]
                        ] + [
                            ri.DeployService("production_noapache_{}_video_yp".format(i), ["prod", "video"])
                            for i in rm_const.MAIN_LOCATIONS
                        ] + [
                            ri.DeployService("hamster_noapache_{}_video_yp".format(i), ["hamster", "video"])
                            for i in rm_const.MAIN_LOCATIONS
                        ]
                    )
                )
            ]
            deploy_infos = [
                ri.NannyDeployInfo(
                    services=list(itertools.chain.from_iterable(i[1] for i in services))
                )
            ]

            return [
                ri.ReleasableItem(
                    name="upper",
                    data=ri.SandboxResourceData("NOAPACHE_UPPER"),
                    deploy_infos=deploy_infos
                ),
                ri.ReleasableItem(
                    name="rearrange_data",
                    data=ri.SandboxResourceData("REARRANGE_DATA", build_ctx_key="rearrange_svn_url"),
                    deploy_infos=deploy_infos
                ),
                ri.ReleasableItem(
                    name="rtcc_bundle",
                    data=ri.SandboxResourceData("RTCC_BUNDLE_NOAPACHE"),
                    deploy_infos=deploy_infos
                )
            ]

        resources_info = [
            configs.ReleasedResourceInfo(
                name="upper",
                resource_type="NOAPACHE_UPPER",
                resource_name="noapacheupper_executable_res_id",
                deploy=[
                    configs.DeployServicesInfo(
                        services=[],
                        dashboards=["noapache_web", "noapache_imgs", "noapache_video"]
                    ),
                ],
            ),
            configs.ReleasedResourceInfo(
                name="rearrange_data",
                resource_type="REARRANGE_DATA",
                resource_name="rearrange_data_res_id",
                build_ctx_key="rearrange_svn_url",
                deploy=[
                    configs.DeployServicesInfo(
                        services=[],
                        dashboards=["noapache_web", "noapache_imgs", "noapache_video"]
                    ),
                ],
            ),
            configs.ReleasedResourceInfo(
                name="rtcc_bundle",
                resource_type="RTCC_BUNDLE_NOAPACHE",
                resource_name="noapache_rtcc_bundle_res_id",
            ),
            configs.ReleasedResourceInfo(
                name="rearrange_dynamic_data_res_id",
                resource_type="REARRANGE_DYNAMIC_DATA",
            ),
            configs.ReleasedResourceInfo(
                name="rearrange_data_fast_web_res_id",
                resource_type="REARRANGE_DATA_FAST",
                build_ctx_key="arcadia_url",
            ),
            configs.ReleasedResourceInfo(
                name="rearrange_data_fast_video_res_id",
                resource_type="REARRANGE_DATA_FAST",
                build_ctx_key="arcadia_url",
            ),
            configs.ReleasedResourceInfo(
                name="production_instancectl_conf",
                resource_type="NOAPACHE_PRODUCTION_INSTANCECTL_CONF",
                resource_name="production_instancectl_conf_res_id",
                build_ctx_key="arcadia_url",
            ),
            configs.ReleasedResourceInfo(
                name="hamster_instancectl_conf",
                resource_type="NOAPACHE_HAMSTER_INSTANCECTL_CONF",
                resource_name="hamster_instancectl_conf_res_id",
                build_ctx_key="arcadia_url",
            ),
            configs.ReleasedResourceInfo(
                name="pushclient_production_conf",
                resource_type="NOAPACHE_PUSHCLIENT_PRODUCTION_CONF",
                resource_name="pushclient_production_conf_res_id",
                build_ctx_key="arcadia_url",
            ),
            configs.ReleasedResourceInfo(
                name="pushclient_hamster_conf",
                resource_type="NOAPACHE_PUSHCLIENT_HAMSTER_CONF",
                resource_name="pushclient_hamster_conf_res_id",
                build_ctx_key="arcadia_url",
            ),
        ]
        release_followers_permanent = [
            "images-releases",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        notifications = [
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="upper_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="upper_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="upper_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="upper_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="NewBetaGeneration",
                chat_name="upper_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="upper_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="upper_comp",
                message_template_file="notifications/upper/ticket_history.html",
                pin=True,
                conditions=rm_notifications.NotificationCondition(
                    conditions=[
                        rm_notifications.NotificationConditionItem(
                            field="ticket_history_data.ticket_history_latest_status",
                            operator="TEXT_EXACTLY_IS",
                            value="open",
                        ),
                    ],
                    join_strategy="OR",
                ),
            ),

            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="upper_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS + [
                        rm_notifications.NotificationConditionItem(
                            field="build_test_data.job_name",
                            operator="TEXT_EXACTLY_IS",
                            value="_TEST__UPPER__RTCC_DIFF",
                        ),
                    ],
                    join_strategy="AND",
                ),
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="upper_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["upper_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = u"https://t.me/joinchat/CaUODkNMIpwWNzLu4pngow"

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = configs.Responsible(abc=configs.Abc(component_id=1609), login="elshiko")
            queue = "UPREL"
            dev_queue = "SEARCH"
            summary_template = u"Приемка верхнего метапоиска noapacheupper {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "avitella",
                "dima-zakharov",
                "elshiko",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 4
            notify_on_deploy_to_feature_st_tickets = True

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "jandekspoisk/kachestvopoiska/noapacheupper/releases/"
        dirs = []
        markers = [
            (4, re.compile(r"\bdiff *: *upper", flags=re.IGNORECASE)),
            (2, re.compile(r"\bcrash *: *upper", flags=re.IGNORECASE)),
        ]
        review_groups = [
            "upper",
            "rearrange",
            "blender",
        ]

    class MetricsCfg(configs.ReferenceBranchedConfig.MetricsCfg):
        run_bisect = True

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "upple"
        tag_name = "upple"
        allow_autobranches = True

    class Yappy(yappy_cfg.YappyBaseCfg):
        _upper_common_beta_resources = [
            yappy_cfg.YappyParametrizedResource(
                local_path="httpsearch",
                param_name="noapacheupper_executable_res_id",
                checkconfig_name="upper-linux-bin-md5",
            ),
            yappy_cfg.YappyParametrizedResource(
                local_path="rearrange_data",
                param_name="rearrange_data_res_id",
            ),
            yappy_cfg.YappyParametrizedResource(
                local_path="BUNDLE",
                param_name="noapache_rtcc_bundle_res_id",
            ),
            yappy_cfg.YappyParametrizedResource(
                local_path="push-client.conf.tmpl",
                param_name="pushclient_hamster_conf_res_id",
            ),
        ]

        _upper_web_beta_resources = list(_upper_common_beta_resources)
        _upper_web_beta_resources.append(
            yappy_cfg.YappyParametrizedResource(
                local_path="instancectl.conf",
                param_name="hamster_instancectl_conf_res_id",
            ),
        )

        betas = {
            "web": yappy_cfg.YappyTemplateCfg(
                template_name="noapache-web",
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="noapache-web",
                        resources=_upper_web_beta_resources,
                        parent_service="hamster_noapache_vla_web_yp",
                        ignore_instance_spec=True,
                    ),
                ],
                new_yappy=True,
            ),
            "video": yappy_cfg.YappyTemplateCfg(
                template_name="noapache-video",
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="noapache-video",
                        resources=_upper_common_beta_resources,
                        parent_service="hamster_noapache_vla_video_yp",
                    ),
                ],
                new_yappy=True,
            ),
            "images": yappy_cfg.YappyTemplateCfg(
                template_name="noapache-images",
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="noapache-images",
                        resources=_upper_common_beta_resources,
                        parent_service="hamster_noapache_vla_imgs_yp",
                        ignore_instance_spec=True,
                    ),
                ],
                new_yappy=True,
            ),
        }

        working_betas_limit = 2
