# -*- coding: utf-8 -*-
import collections
import enum
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.common.constants import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows


class IntegrationTestNames(enum.Enum):
    MIDDLE = "MIDDLE"
    MISCDATA = "MISCDATA"
    USERDATA_CLEAN_EXPORTS_TEST = "USERDATA_CLEAN_EXPORTS_TEST"
    USERDATA_FAST = "USERDATA_FAST"
    USERDATA_MERGED_LONG = "USERDATA_MERGED_LONG"
    USERSTATS = "USERSTATS"
    USER_BROWSE_FAST = "USER_BROWSE_FAST"
    USER_BROWSE_LONG = "USER_BROWSE_LONG"
    USER_COUNTERS_LONG = "USER_COUNTERS_LONG"
    USER_SEARCH_FAST = "USER_SEARCH_FAST"
    USER_SEARCH_LONG = "USER_SEARCH_LONG"
    USER_SEARCH_PROC_PERIOD = "USER_SEARCH_PROC_PERIOD"
    USER_SEARCH_COLUMNAR_PROC_PERIOD = "USER_SEARCH_COLUMNAR_PROC_PERIOD"
    WEBMASTER_HOST2VEC = "WEBMASTER_HOST2VEC"


def make_extraction_job_name(parent_name):
    assert isinstance(parent_name, IntegrationTestNames)
    return "EXTRACT_{}".format(parent_name.value)


TRUNK_JOB_TAGS = [jg_utils.Tag.LIGHT_PRECOMMIT_CHECKS]


class UserfeatCfg(configs.ReferenceBranchedConfig):
    name = "userfeat"
    responsible = "tarum"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "userdata_full_trunk_yt"
        trunk_task_owner = "USERFEAT"
        branch_db_template = "userfeat-stable-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            class PeriodType(object):
                FAST = "fast"
                LONG = "long"

            class SysNames(object):
                BROWSE = "BROWSE"
                COUNTERS = "COUNTERS"
                MISC = "MISC"
                SEARCH = "SEARCH"
                UPDATER = "UPDATER"
                USERDATA = "USERDATA"
                COMMON = "COMMON"
                FAST_USERDATA = "FAST_USERDATA"
                WEBMASTER = "WEBMASTER"

            SysInfo = collections.namedtuple(
                "SysInfo",
                "binaries_type binaries_description cmpy_type period_types"
            )
            LONG_YT_RELEASE_ATTRIBUTE_PATH = "//userfeat/deploy_meta/@deploy/ready_for_deploy"
            FAST_YT_RELEASE_ATTRIBUTE_PATH = "//userfeat/deploy_meta/@deploy/fast_userdata_ready_for_deploy"

            GB = 1024 ** 3

            COMMON_TEST_OBSERVED_PATHS = [
                "quality/userdata/library/yuppie",
                "quality/userdata/test/common",
                "robot/jupiter/library/python/jupiter_util",
                "robot/library/yuppie",
            ]

            SYSTEMS = {
                SysNames.BROWSE: SysInfo(
                    binaries_type="USERFEAT_BROWSE_BINARIES_RELEASE",
                    binaries_description="quality/user_browse/package/yandex-search-user-browse.json",
                    cmpy_type="USERFEAT_CMPY_USER_BROWSE",
                    period_types=(PeriodType.LONG,),
                ),
                SysNames.COUNTERS: SysInfo(
                    binaries_type="USERFEAT_COUNTERS_BINARIES_RELEASE",
                    binaries_description="quality/user_counters/package/yandex-search-user-counters.json",
                    cmpy_type="USERFEAT_CMPY_USER_COUNTERS",
                    period_types=(PeriodType.LONG,),
                ),
                SysNames.MISC: SysInfo(
                    binaries_type="USERFEAT_MISC_BINARIES_RELEASE",
                    binaries_description="quality/miscdata/package/yandex-search-md-prep-runner.json",
                    cmpy_type="USERFEAT_CMPY_MISCDATA",
                    period_types=(PeriodType.LONG,),
                ),
                SysNames.SEARCH: SysInfo(
                    binaries_type="USERFEAT_SEARCH_BINARIES_RELEASE",
                    binaries_description="quality/user_search/package/yandex-search-user-search.json",
                    cmpy_type="USERFEAT_CMPY_USER_SEARCH",
                    period_types=(PeriodType.LONG,),
                ),
                SysNames.USERDATA: SysInfo(
                    binaries_type="USERFEAT_USERDATA_BINARIES_RELEASE",
                    binaries_description="quality/userdata/packages/prep/yandex-search-userdata-prep.json",
                    cmpy_type="USERFEAT_CMPY_USERDATA",
                    period_types=(PeriodType.LONG,),
                ),
                SysNames.COMMON: SysInfo(
                    binaries_type=None,
                    binaries_description=None,
                    cmpy_type="USERFEAT_CMPY_COMMON",
                    period_types=(PeriodType.FAST, PeriodType.LONG),
                ),
                SysNames.FAST_USERDATA: SysInfo(
                    binaries_type="USERFEAT_FAST_USERDATA_BINARIES_RELEASE",
                    binaries_description="quality/userdata/packages/fast_userdata/yandex-search-fast-userdata.json",
                    cmpy_type="USERFEAT_CMPY_FAST_USERDATA",
                    period_types=(PeriodType.FAST,),
                ),
                SysNames.WEBMASTER: SysInfo(
                    binaries_type="USERFEAT_WEBMASTER_BINARIES_RELEASE",
                    binaries_description="wmconsole/version3/processors/user_sessions/package/userfeat-webmaster-binaries.json",
                    cmpy_type="USERFEAT_CMPY_WEBMASTER",
                    period_types=(PeriodType.LONG,),
                )
            }

            """
            Released resource TTL is inf by default, but in practice we don't need that
            and 1 year is likely to be enough.
            Note that this setting doesn't cover YaPackage output yet, see RMDEV-2458
            """
            RELEASED_TTL_DAYS = "365"
            USERFEAT_YT_CLUSTER = "hahn.yt.yandex.net"
            USERFEAT_YT_VAULT_TOKEN = "robot-userfeat-yt-token"
            YT_TOKEN_YAV_SECRET = "sec-01cqpezcefkj1e8bkwm7hd47qb#yt.token"

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(
                        branching_mode=rm_const.BranchingMode.BRANCH_TO_BRANCH,
                    ),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_test.JobGraphElementTestTrunkCommon(
                        task_name="NEED_TO_BACKPORT",
                        job_params={
                            "job_name_parameter": "NEED_TO_BACKPORT",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerChangelog(),
                            jg_job_triggers.JobTriggerNewBranch(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="target_branch_path",
                                        output_key="result_path",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerStartrek(),
                        )
                    ),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="NEED_TO_BACKPORT",
                            ),
                        ),
                    ),
                ]

            def _build_mrapps_job(self):
                return jg_build.JobGraphElementBuildTrunk(
                    task_name="KOSHER_YA_MAKE",
                    build_item="MRAPPS",
                    job_params={
                        "apiargs": {
                            "requirements": {
                                "disk_space": 10 * self.GB,
                            },
                            "kill_timeout": 4200,
                        },
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.LAZY, None),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "tags": TRUNK_JOB_TAGS,
                    },
                    ctx={
                        "build_type": "release",
                        "check_return_code": True,
                        "checkout_mode": "auto",
                        "definition_flags": "-DDEBUGINFO_LINES_ONLY=yes",
                        "result_rt": "USERFEAT_MRAPPS_RELEASE",
                        "result_released_ttl": self.RELEASED_TTL_DAYS,
                        "arts": "quality/userdata/packages/mrapps_release",
                        "targets": "quality/userdata/packages/mrapps_release",
                        "clear_build": False,
                        sandbox_constants.YA_YT_TOKEN_YAV_SECRET: self.YT_TOKEN_YAV_SECRET,
                    },
                    out={"USERFEAT_MRAPPS_RELEASE": 30},  # days to store
                )

            def _build_cmpy_job(self, system_name, targets):
                return jg_build.JobGraphElementBuildTrunk(
                    task_name="KOSHER_YA_MAKE",
                    build_item="CMPY_{}".format(system_name),
                    job_params={
                        "apiargs": {
                            "requirements": {
                                "disk_space": 10 * self.GB,
                            },
                            "kill_timeout": 4200,
                        },
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.LAZY, None),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "tags": TRUNK_JOB_TAGS,
                    },
                    ctx={
                        "build_type": "release",
                        "check_return_code": True,
                        "checkout_mode": "auto",
                        "definition_flags": "-DDEBUGINFO_LINES_ONLY=yes",
                        "result_rt": self.SYSTEMS[system_name].cmpy_type,
                        "result_released_ttl": self.RELEASED_TTL_DAYS,
                        "arts": targets,
                        "targets": targets,
                        "clear_build": False,
                        sandbox_constants.YA_YT_TOKEN_YAV_SECRET: self.YT_TOKEN_YAV_SECRET,
                    },
                    out={self.SYSTEMS[system_name].cmpy_type: 30},  # days to store
                )

            def _build_update_job(self, build_item, packages, strip_binaries):
                return jg_build.JobGraphElementBuildTrunk(
                    task_name="KOSHER_YA_PACKAGE",
                    build_item="{}_UPDATE".format(build_item),
                    job_params={
                        "apiargs": {
                            "requirements": {
                                "disk_space": 100 * self.GB,
                            },
                            "kill_timeout": 10800,  # TODO(alexromanov): Decrease timemout after JUPITER-1700
                        },
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.LAZY, None),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "tags": TRUNK_JOB_TAGS,
                    },
                    ctx={
                        "build_type": "release",
                        "checkout_mode": "auto",
                        "package_type": "tarball",
                        "packages": packages,
                        "resource_type": "USERFEAT_{}_PACKAGE".format(build_item),
                        "publish_package": False,
                        "run_tests": False,
                        "save_build_output": False,
                        "strip_binaries": strip_binaries,
                        "separate_result_dirs": True,
                        "use_new_format": False,
                        "use_ya_dev": False,
                        "clear_build": False,
                        sandbox_constants.YA_YT_TOKEN_YAV_SECRET: self.YT_TOKEN_YAV_SECRET,
                        # "package_released_resource_ttl": self.RELEASED_TTL_DAYS,
                    },
                    out={
                        "USERFEAT_{}_PACKAGE".format(build_item): 30,  # days to store
                    },
                )

            def _join_long_job(self):
                return jg_test.JobGraphElementTestTagCommon(
                    task_name="COMBINE_MR_TABLES",
                    job_params={
                        "job_name_parameter": "JOIN_LONG",
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.LAZY, None),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "tags": TRUNK_JOB_TAGS,
                    },
                    job_arrows=(
                        jg_arrows.ParentsData(
                            input_key="result_name",
                            triggers=(
                                jg_arrows.GlobalResourceData(
                                    input_key="res_id",
                                    resource_name="USERDATA_EXTERNAL_INPUTS",
                                ),
                            ),
                            transform=lambda res_id: 'Joined tables for comp_long, originally from {}'.format(res_id),
                        ),
                        jg_arrows.ParentsData(
                            input_key="state_resource_ids",
                            triggers=(
                                jg_arrows.GlobalResourceData(
                                    input_key="userdata_external_id",
                                    resource_name="USERDATA_EXTERNAL_INPUTS",
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter=make_extraction_job_name(IntegrationTestNames.MISCDATA),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="miscdata_id",
                                            resource_name="USERDATA_TABLES_ARCHIVE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter=make_extraction_job_name(IntegrationTestNames.USER_SEARCH_LONG),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="user_search_id",
                                            resource_name="USERDATA_TABLES_ARCHIVE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter=make_extraction_job_name(IntegrationTestNames.USER_BROWSE_LONG),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="user_browse_id",
                                            resource_name="USERDATA_TABLES_ARCHIVE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter=make_extraction_job_name(IntegrationTestNames.USER_COUNTERS_LONG),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="user_counters_id",
                                            resource_name="USERDATA_TABLES_ARCHIVE",
                                        ),
                                    ),
                                ),
                            ),
                            transform=jg_utils.args_to_list,
                        ),
                    ),
                    ctx={
                        "merge_resource_attrs": True,
                        "extra_attrs": {"joined_by_testenv": 1},
                        "drop_attrs": [],
                    },
                    out={
                        "USERDATA_TABLES_ARCHIVE": 60,  # days to store
                    },
                )

            def _join_fast_job(self):
                return jg_test.JobGraphElementTestTagCommon(
                    task_name="COMBINE_MR_TABLES",
                    job_params={
                        "job_name_parameter": "JOIN_FAST",
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.LAZY, None),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "tags": TRUNK_JOB_TAGS,
                    },
                    job_arrows=(
                        jg_arrows.ParentsData(
                            input_key="result_name",
                            triggers=(
                                jg_arrows.GlobalResourceData(
                                    input_key="res_id",
                                    resource_name="USERDATA_MIRRORS_FOR_FAST",
                                ),
                            ),
                            transform=lambda res_id: 'Joined tables for comp_fast, originally from {}'.format(res_id),
                        ),
                        jg_arrows.ParentsData(
                            input_key="state_resource_ids",
                            triggers=(
                                jg_arrows.GlobalResourceData(
                                    input_key="userdata_mirrors_id",
                                    resource_name="USERDATA_MIRRORS_FOR_FAST",
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter=make_extraction_job_name(IntegrationTestNames.MISCDATA),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="miscdata_id",
                                            resource_name="USERDATA_TABLES_ARCHIVE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter=make_extraction_job_name(IntegrationTestNames.USER_SEARCH_FAST),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="user_search_id",
                                            resource_name="USERDATA_TABLES_ARCHIVE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_name_parameter=make_extraction_job_name(IntegrationTestNames.USER_BROWSE_FAST),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="user_browse_id",
                                            resource_name="USERDATA_TABLES_ARCHIVE",
                                        ),
                                    ),
                                ),
                            ),
                            transform=jg_utils.args_to_list,
                        ),
                    ),
                    ctx={
                        "merge_resource_attrs": True,
                        "extra_attrs": {"joined_by_testenv": 1},
                        "drop_attrs": [],
                    },
                    out={
                        "USERDATA_TABLES_ARCHIVE": 60,  # days to store
                    },
                )

            def _join_test_params(self, tables_id, geodata_id, ipreg_set_id, geoa_c2p_id, blockstat_id, rfl_id, clickdaemon_keys_id, **package_ids):
                test_params = [
                    ("tables_id", str(tables_id)),
                    ("geodata_id", str(geodata_id)),
                    ("ipreg_set_id", str(ipreg_set_id)),
                    ("geoa_c2p_id", str(geoa_c2p_id)),
                    ("blockstat_id", str(blockstat_id)),
                    ("rfl_id", str(rfl_id)),
                    ("clickdaemon_keys_id", str(clickdaemon_keys_id)),
                ]
                if package_ids:
                    test_params.append(("package_ids", ",".join([str(i) for i in package_ids.values()])))
                return " ".join(["{}={}".format(k, v) for k, v in test_params])

            global_resources = (
                jg_arrows.GlobalResourceData(
                    input_key="geodata_id",
                    resource_name="GEODATA5BIN_STABLE",
                ),
                jg_arrows.GlobalResourceData(
                    input_key="ipreg_set_id",
                    resource_name="IPREG_SET",
                ),
                jg_arrows.GlobalResourceData(
                    input_key="geoa_c2p_id",
                    resource_name="GEOA_C2P",
                ),
                jg_arrows.GlobalResourceData(
                    input_key="blockstat_id",
                    resource_name="USERDATA_BLOCKSTAT_DICT",
                ),
                jg_arrows.GlobalResourceData(
                    input_key="rfl_id",
                    resource_name="RFL_FOR_CANONIZER",
                ),
                jg_arrows.GlobalResourceData(
                    input_key="clickdaemon_keys_id",
                    resource_name="CLICKDAEMON_KEYS",
                ),
            )

            def _miscdata_comp_long_test_job(self):
                return jg_build.JobGraphElementBuildTrunk(
                    task_name="USERDATA_KOSHER_YA_MAKE_TEST_RUNNER",
                    build_item=IntegrationTestNames.MISCDATA.value,
                    job_params={
                        "observed_paths": [
                            "quality/miscdata",
                            "quality/userdata/cmpy/miscdata",
                        ] + self.COMMON_TEST_OBSERVED_PATHS,
                        "apiargs": {
                            "requirements": {
                                "disk_space": 300 * self.GB,
                                "ram": 56 * self.GB,
                            },
                        },
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 180),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "tags": TRUNK_JOB_TAGS,
                    },
                    job_arrows=(
                        jg_arrows.ParentsData(
                            input_key="test_params",
                            triggers=self.global_resources + (
                                jg_arrows.GlobalResourceData(
                                    input_key="tables_id",
                                    resource_name="MISCDATA_INPUT_TABLES",
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="MISCDATA_UPDATE",
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="miscdata_update_id",
                                            resource_name="USERFEAT_MISCDATA_PACKAGE",
                                        ),
                                    ),
                                ),
                            ),
                            transform=self._join_test_params,
                        ),
                    ),
                    ctx={
                        "targets": "quality/userdata/test/miscdata-diff",
                        "ram_drive_size": 32768,  # 32 * 1024 in megabytes
                        "test": True,
                        sandbox_constants.BUILD_SYSTEM_KEY: "semi_distbuild",
                        sandbox_constants.BUILD_TYPE_KEY: "release",
                        sandbox_constants.CLEAR_BUILD_KEY: False,
                        "definition_flags": "-DUSE_EAT_MY_DATA=yes -DDEBUGINFO_LINES_ONLY=yes -DNO_SVN_DEPENDS -DNO_BINARIES",
                        # without DEBUGINFO_LINES_ONLY distbuild will fail to build binaries package
                        sandbox_constants.USE_AAPI_FUSE: True,
                        "result_released_ttl": self.RELEASED_TTL_DAYS,
                    },
                    out={"BUILD_OUTPUT": 3},  # days to store
                )

            def _user_runner_job(self, integration_test_name, targets, job_arrows, requirements, observed_paths):
                assert isinstance(integration_test_name, IntegrationTestNames)
                assert isinstance(targets, str)
                assert isinstance(observed_paths, list)
                return jg_build.JobGraphElementBuildTrunk(
                    task_name="USERDATA_KOSHER_YA_MAKE_TEST_RUNNER",
                    build_item=integration_test_name.value,
                    job_params={
                        "apiargs": {
                            "requirements": requirements,
                        },
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 180),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "tags": TRUNK_JOB_TAGS,
                    },
                    job_arrows=(
                        jg_arrows.ParentsData(
                            input_key="test_params",
                            triggers=self.global_resources + job_arrows,
                            transform=self._join_test_params,
                        ) if integration_test_name != IntegrationTestNames.USERDATA_CLEAN_EXPORTS_TEST else None,
                    ),
                    filter_targets=observed_paths,
                    ctx={
                        "ram_drive_size": 32768,  # 32 * 1024 in megabytes
                        "test": True,
                        sandbox_constants.BUILD_SYSTEM_KEY: "semi_distbuild",
                        sandbox_constants.BUILD_TYPE_KEY: "release",
                        sandbox_constants.CLEAR_BUILD_KEY: False,
                        "targets": targets,
                        "definition_flags": "-DUSE_EAT_MY_DATA=yes -DDEBUGINFO_LINES_ONLY=yes -DNO_SVN_DEPENDS -DNO_BINARIES",
                        # without DEBUGINFO_LINES_ONLY distbuild will fail to build binaries package
                        sandbox_constants.USE_AAPI_FUSE: True,
                        "result_released_ttl": self.RELEASED_TTL_DAYS,
                    },
                    out={"BUILD_OUTPUT": 3},  # days to store
                )

            data_extractor_ctx = {
                "userdata_view_selector": "resource",
                "querydata_viewer_selector": "resource",
                "state_a_source_selector": "resource",
                "state_b_source_selector": "resource",
                "store_tables": True,
                "skip_unchanged_tables": True,
                "dump_to_text": True,
                "colored_diff": True,
                "mr_diff_selector": "resource",
                "mr_server": "yt_local",
            }

            def _construct_url_with_revision2(self, params, rm_config):
                return '{svn_ssh_url}/arcadia@{revision2}'.format(
                    svn_ssh_url=params.svn_ssh_url,
                    revision2=params.revision2,
                )

            def _extraction_job(self, integration_test_name, targets, requirements):
                assert isinstance(integration_test_name, IntegrationTestNames)
                return jg_test.JobGraphElementTestTagCommon(
                    task_name="USERDATA_TEST_DATA_EXTRACTOR",
                    job_params={
                        "job_name_parameter": make_extraction_job_name(integration_test_name),
                        "should_add_to_db": {
                            "trunk": jg_utils.should_add_to_db_trunk,
                            "branch": jg_utils.should_add_to_db_branch,
                        },
                        "frequency": {
                            "trunk": (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 180),
                            "branch": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        "apiargs": {
                            "kill_timeout": "10800",
                            "requirements": requirements,
                        },
                        "compare_task_name": "COMPARE_MR_TABLES",
                        "test_type": jg_utils.TestType.DIFF_TEST,
                        "diff_resource_type": {
                            "USERDATA_TABLES_DIFF_DESCR": 14,
                            "USERDATA_TABLES_ARCHIVE": 14,
                        },
                        "has_diff": lambda ctx: ctx.get("has_diff"),
                        "tags": TRUNK_JOB_TAGS,
                        "diff_url": "https://sandbox.yandex-team.ru/task/{diff_task_id}",
                        "compare_job_triggers": {
                            jg_arrows.GlobalResourceData(
                                input_key="querydata_viewer_resid",
                                resource_name="QUERYDATA_VIEWER_EXECUTABLE",
                            ),
                            jg_arrows.GlobalResourceData(
                                input_key="userdata_view_resid",
                                resource_name="USERDATA_VIEW_EXECUTABLE",
                            ),
                            jg_arrows.GlobalResourceData(
                                input_key="mr_diff_resid",
                                resource_name="MR_DIFF",
                            ),
                            jg_arrows.GlobalResourceData(
                                input_key="yt_local_resource_id",
                                resource_name="YT_LOCAL",
                            ),
                            jg_arrows.CompareTaskDataResource(
                                input_key="state_a_resource_id",
                                resource_name="USERDATA_TABLES_ARCHIVE",
                                task_number=1,
                            ),
                            jg_arrows.CompareTaskDataResource(
                                input_key="state_b_resource_id",
                                resource_name="USERDATA_TABLES_ARCHIVE",
                                task_number=2,
                            ),
                            jg_arrows.ParamsData(
                                input_key=sandbox_constants.ARCADIA_URL_KEY,
                                transform=self._construct_url_with_revision2,
                            ),
                        },
                        "compare_ctx": self.data_extractor_ctx,
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter=integration_test_name.value,
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="build_output",
                                    resource_name="BUILD_OUTPUT",
                                ),
                            )
                        )
                    ),
                    ctx={
                        "target": targets,
                    },
                    out={"USERDATA_TABLES_ARCHIVE": 30},  # days to store
                )

            @property
            def _trunk_part(self):
                default_trunk_part = super(self.__class__, self)._trunk_part
                build_mrapps = self._build_mrapps_job()
                build_cmpy_common = self._build_cmpy_job(self.SysNames.COMMON, "quality/userdata/packages/cmpy_common")
                build_cmpy_misc = self._build_cmpy_job(self.SysNames.MISC, "quality/userdata/packages/cmpy_miscdata")
                build_cmpy_userdata = self._build_cmpy_job(self.SysNames.USERDATA, "quality/userdata/packages/cmpy_userdata")
                build_cmpy_search = self._build_cmpy_job(self.SysNames.SEARCH, "quality/userdata/packages/cmpy_user_search")
                build_cmpy_browse = self._build_cmpy_job(self.SysNames.BROWSE, "quality/userdata/packages/cmpy_user_browse")
                build_cmpy_counters = self._build_cmpy_job(self.SysNames.COUNTERS, "quality/userdata/packages/cmpy_user_counters")
                build_cmpy_fast_userdata = self._build_cmpy_job(self.SysNames.FAST_USERDATA, "quality/userdata/packages/cmpy_fast_userdata")
                build_cmpy_webmaster = self._build_cmpy_job(self.SysNames.WEBMASTER, "quality/userdata/packages/cmpy_webmaster")

                build_update_miscdata = self._build_update_job(
                    "MISCDATA",
                    "quality/miscdata/package/yandex-search-md-prep-runner.json",
                    False,
                )
                build_update_userdata = self._build_update_job(
                    "USERDATA",
                    "quality/userdata/packages/prep/yandex-search-userdata-prep.json",
                    False,
                )
                build_update_user_browse = self._build_update_job(
                    "USER_BROWSE",
                    "quality/user_browse/package/yandex-search-user-browse.json",
                    False,
                )
                build_update_user_counters = self._build_update_job(
                    "USER_COUNTERS",
                    "quality/user_counters/package/yandex-search-user-counters.json",
                    False,
                )
                build_update_user_search = self._build_update_job(
                    "USER_SEARCH",
                    "quality/user_search/package/yandex-search-user-search.json",
                    True,
                )

                miscdata_comp_long_test = self._miscdata_comp_long_test_job()
                join_long_job = self._join_long_job()
                join_fast_job = self._join_fast_job()

                user_runner_middle_test = self._user_runner_job(
                    IntegrationTestNames.MIDDLE,
                    "quality/userdata/test/middle",
                    (
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="USERDATA_UPDATE",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="package_id",
                                    resource_name="USERFEAT_USERDATA_PACKAGE",
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter="JOIN_LONG",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="tables_id",
                                    resource_name="USERDATA_TABLES_ARCHIVE",
                                ),
                            )
                        ),
                    ),
                    {"disk_space": 300 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "quality/userdata/cmpy/library",
                        "quality/userdata/cmpy/userdata/config",
                        "quality/userdata/cmpy/userdata/library",
                        "quality/userdata/cmpy/userdata/middle",
                        "quality/userdata/test/middle",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_search_long_test = self._user_runner_job(
                    IntegrationTestNames.USER_SEARCH_LONG,
                    "quality/userdata/test/user-search-long-diff",
                    (
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="USER_SEARCH_UPDATE",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="update_id",
                                    resource_name="USERFEAT_USER_SEARCH_PACKAGE",
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="MRAPPS",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="mrapps_id",
                                    resource_name="USERFEAT_MRAPPS_RELEASE",
                                )
                            )
                        ),
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="USER_SEARCH_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 600 * self.GB, "ram": 256 * self.GB},
                    observed_paths=[
                        "robot/jupiter/library",
                        "quality/mr_quality_lib",
                        "quality/user_search",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/cmpy/user_search",
                        "quality/userdata/test/user-search-long-diff",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_counters_long_test = self._user_runner_job(
                    IntegrationTestNames.USER_COUNTERS_LONG,
                    "quality/userdata/test/user-counters-long-diff",
                    (
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="USER_COUNTERS_UPDATE",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="counters_id",
                                    resource_name="USERFEAT_USER_COUNTERS_PACKAGE",
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="MRAPPS",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="mrapps_id",
                                    resource_name="USERFEAT_MRAPPS_RELEASE",
                                )
                            )
                        ),
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="USER_COUNTERS_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 300 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "robot/jupiter/library",
                        "quality/mr_quality_lib",
                        "quality/user_counters",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/cmpy/user_counters",
                        "quality/userdata/test/user-counters-long-diff",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_userstats_test = self._user_runner_job(
                    IntegrationTestNames.USERSTATS,
                    "quality/userdata/test/userstats-diff",
                    (
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="USER_SEARCH_UPDATE",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="package_id",
                                    resource_name="USERFEAT_USER_SEARCH_PACKAGE",
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="MRAPPS",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="mrapps_id",
                                    resource_name="USERFEAT_MRAPPS_RELEASE",
                                )
                            )
                        ),
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="USER_SEARCH_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 300 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "robot/jupiter/library",
                        "quality/mr_quality_lib",
                        "quality/user_search",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/cmpy/user_search",
                        "quality/userdata/test/userstats-diff",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_browse_long_test = self._user_runner_job(
                    IntegrationTestNames.USER_BROWSE_LONG,
                    "quality/userdata/test/user-browse-long-diff",
                    (
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="USER_BROWSE_UPDATE",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="browse_update_id",
                                    resource_name="USERFEAT_USER_BROWSE_PACKAGE",
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="MRAPPS",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="mrapps_id",
                                    resource_name="USERFEAT_MRAPPS_RELEASE",
                                )
                            )
                        ),
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="USER_BROWSE_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 300 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "robot/jupiter/library",
                        "quality/mr_quality_lib",
                        "quality/user_browse",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/cmpy/user_browse",
                        "quality/userdata/test/user-browse-long-diff",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_fast_userdata_browse_period_test = self._user_runner_job(
                    IntegrationTestNames.USER_BROWSE_FAST,
                    "quality/userdata/test/fast_userdata/user_browse_proc_period",
                    (
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="USER_BROWSE_FAST_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 300 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "quality/user_browse",
                        "quality/userdata/cmpy/fast_userdata",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/test/fast_userdata/user_browse_proc_period",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_fast_userdata_search_period_test = self._user_runner_job(
                    IntegrationTestNames.USER_SEARCH_FAST,
                    "quality/userdata/test/fast_userdata/user_search_proc_period",
                    (
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="USER_SEARCH_FAST_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 600 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "robot/jupiter/library",
                        "quality/mr_quality_lib",
                        "quality/user_search",
                        "quality/userdata/cmpy/fast_userdata",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/test/fast_userdata/user_search_proc_period",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_fast_userdata_fast_test = self._user_runner_job(
                    IntegrationTestNames.USERDATA_FAST,
                    "quality/userdata/test/fast_userdata/comp_fast",
                    (
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter="JOIN_FAST",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="tables_id",
                                    resource_name="USERDATA_TABLES_ARCHIVE"
                                )
                            )
                        ),
                    ),
                    {"disk_space": 300 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "quality/userdata/cmpy/fast_userdata",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/test/fast_userdata/comp_fast",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_userdata_merged_long_test = self._user_runner_job(
                    IntegrationTestNames.USERDATA_MERGED_LONG,
                    "quality/userdata/test/userdata-merged-long",
                    (
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="USERDATA_UPDATE",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="update_id",
                                    resource_name="USERFEAT_USERDATA_PACKAGE",
                                ),
                            ),
                        ),
                        jg_job_triggers.JobTriggerTestBranchCommon(
                            job_name_parameter="JOIN_LONG",
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="tables_id",
                                    resource_name="USERDATA_TABLES_ARCHIVE",
                                )
                            )
                        ),
                    ),
                    {"disk_space": 300 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "quality/userdata/cmpy/library",
                        "quality/userdata/cmpy/userdata/library",
                        "quality/userdata/cmpy/userdata/userdata_merged_long",
                        "quality/userdata/test/userdata-merged-long",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_userdata_clean_exports_test = self._user_runner_job(
                    IntegrationTestNames.USERDATA_CLEAN_EXPORTS_TEST,
                    "quality/userdata/test/cleanup-exports",
                    (),
                    {"disk_space": 70 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "quality/userdata/cmpy/library",
                        "quality/userdata/cmpy/userdata/cleanup_exports",
                        "quality/userdata/test/cleanup-exports",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_userdata_search_proc_period_test = self._user_runner_job(
                    IntegrationTestNames.USER_SEARCH_PROC_PERIOD,
                    "quality/userdata/test/user_search/proc_period",
                    (
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="USER_SEARCH_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 600 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "robot/jupiter/library",
                        "quality/mr_quality_lib",
                        "quality/user_search/proc_period_mr",
                        "quality/userdata/cmpy/user_search/user_search_long_proc_period",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/test/user_search/proc_period",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_userdata_search_columnar_proc_period_test = self._user_runner_job(
                    IntegrationTestNames.USER_SEARCH_COLUMNAR_PROC_PERIOD,
                    "quality/userdata/test/user_search/columnar_proc_period",
                    (
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="USER_SEARCH_COLUMNAR_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 600 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "robot/jupiter/library",
                        "quality/mr_quality_lib",
                        "quality/user_search/proc_period_mr",
                        "quality/userdata/cmpy/user_search/user_search_long_proc_period",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/test/user_search/columnar_proc_period",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )
                user_runner_webmaster_host2vec_test = self._user_runner_job(
                    IntegrationTestNames.WEBMASTER_HOST2VEC,
                    "quality/userdata/test/webmaster/host2vec",
                    (
                        jg_arrows.GlobalResourceData(
                            input_key="tables_id",
                            resource_name="WEBMASTER_HOST2VEC_INPUT_TABLES",
                        ),
                    ),
                    {"disk_space": 50 * self.GB, "ram": 56 * self.GB},
                    observed_paths=[
                        "robot/jupiter/library",
                        "wmconsole/version3/processors/tools/host2vec/model",
                        "quality/userdata/cmpy/webmaster/host2vec_datasets",
                        "quality/userdata/cmpy/library",
                        "quality/userdata/test/webmaster/host2vec",
                    ] + self.COMMON_TEST_OBSERVED_PATHS,
                )

                diff_middle_test = self._extraction_job(
                    IntegrationTestNames.MIDDLE,
                    "quality/userdata/test/middle",
                    {"disk_space": 70 * self.GB},
                )
                diff_miscdata_comp_long_test = self._extraction_job(
                    IntegrationTestNames.MISCDATA,
                    "quality/userdata/test/miscdata-diff",
                    {"disk_space": 60 * self.GB},
                )
                diff_user_search_long_test = self._extraction_job(
                    IntegrationTestNames.USER_SEARCH_LONG,
                    "quality/userdata/test/user-search-long-diff",
                    {"disk_space": 100 * self.GB},
                )
                diff_user_counters_long_test = self._extraction_job(
                    IntegrationTestNames.USER_COUNTERS_LONG,
                    "quality/userdata/test/user-counters-long-diff",
                    {"disk_space": 60 * self.GB},
                )
                diff_userstats_test = self._extraction_job(
                    IntegrationTestNames.USERSTATS,
                    "quality/userdata/test/userstats-diff",
                    {"disk_space": 60 * self.GB},
                )
                diff_user_browse_long_test = self._extraction_job(
                    IntegrationTestNames.USER_BROWSE_LONG,
                    "quality/userdata/test/user-browse-long-diff",
                    {"disk_space": 120 * self.GB},
                )
                diff_fast_browse_proc_period_test = self._extraction_job(
                    IntegrationTestNames.USER_BROWSE_FAST,
                    "quality/userdata/test/fast_userdata/user_browse_proc_period",
                    {"disk_space": 50 * self.GB},
                )
                diff_fast_search_proc_period_test = self._extraction_job(
                    IntegrationTestNames.USER_SEARCH_FAST,
                    "quality/userdata/test/fast_userdata/user_search_proc_period",
                    {"disk_space": 50 * self.GB},
                )
                diff_fast_userdata_comp_fast_test = self._extraction_job(
                    IntegrationTestNames.USERDATA_FAST,
                    "quality/userdata/test/fast_userdata/comp_fast",
                    {"disk_space": 60 * self.GB},
                )
                diff_userdata_merged_long_test = self._extraction_job(
                    IntegrationTestNames.USERDATA_MERGED_LONG,
                    "quality/userdata/test/userdata-merged-long",
                    {"disk_space": 70 * self.GB},
                )
                diff_userdata_search_proc_period_test = self._extraction_job(
                    IntegrationTestNames.USER_SEARCH_PROC_PERIOD,
                    "quality/userdata/test/user_search/proc_period",
                    {"disk_space": 50 * self.GB},
                )
                diff_userdata_search_columnar_proc_period_test = self._extraction_job(
                    IntegrationTestNames.USER_SEARCH_COLUMNAR_PROC_PERIOD,
                    "quality/userdata/test/user_search/columnar_proc_period",
                    {"disk_space": 50 * self.GB},
                )
                diff_webmaster_host2vec_test = self._extraction_job(
                    IntegrationTestNames.WEBMASTER_HOST2VEC,
                    "quality/userdata/test/webmaster/host2vec",
                    {"disk_space": 50 * self.GB},
                )

                trunk_part = [
                    build_mrapps,
                    build_cmpy_common,
                    build_cmpy_misc,
                    build_cmpy_userdata,
                    build_cmpy_search,
                    build_cmpy_browse,
                    build_cmpy_counters,
                    build_cmpy_fast_userdata,
                    build_cmpy_webmaster,
                    build_update_miscdata,
                    build_update_userdata,
                    build_update_user_browse,
                    build_update_user_counters,
                    build_update_user_search,
                    miscdata_comp_long_test,
                    join_long_job,
                    join_fast_job,
                    user_runner_middle_test,
                    user_runner_search_long_test,
                    user_runner_counters_long_test,
                    user_runner_userstats_test,
                    user_runner_browse_long_test,
                    user_runner_fast_userdata_browse_period_test,
                    user_runner_fast_userdata_search_period_test,
                    user_runner_fast_userdata_fast_test,
                    user_runner_userdata_merged_long_test,
                    user_runner_userdata_clean_exports_test,
                    user_runner_userdata_search_proc_period_test,
                    user_runner_userdata_search_columnar_proc_period_test,
                    user_runner_webmaster_host2vec_test,
                    diff_middle_test,
                    diff_miscdata_comp_long_test,
                    diff_user_search_long_test,
                    diff_user_counters_long_test,
                    diff_userstats_test,
                    diff_user_browse_long_test,
                    diff_fast_browse_proc_period_test,
                    diff_fast_search_proc_period_test,
                    diff_fast_userdata_comp_fast_test,
                    diff_userdata_merged_long_test,
                    diff_userdata_search_proc_period_test,
                    diff_userdata_search_columnar_proc_period_test,
                    diff_webmaster_host2vec_test,
                ]
                return default_trunk_part + trunk_part

            def _build_binaries_job(self, sysname):
                resource_type = self.SYSTEMS[sysname].binaries_type
                return jg_build.JobGraphElementYaMakeBuildBranched(
                    task_name="KOSHER_YA_PACKAGE",
                    build_item="{}_BINARIES".format(sysname),
                    job_params={
                        "apiargs": {
                            "requirements": {
                                "disk_space": 100 * 1024,
                            },
                        },
                        "frequency": (jg_utils.TestFrequency.LAZY, None),
                    },
                    ctx={
                        "build_type": "release",
                        "checkout_mode": "auto",
                        "custom_version": "UNSET",
                        "package_type": "tarball",
                        "raw_package": True,
                        "packages": self.SYSTEMS[sysname].binaries_description,
                        "publish_package": False,
                        "resource_type": resource_type,
                        "run_tests": False,
                        "save_build_output": False,
                        "sloppy_debian": True,
                        "strip_binaries": False,
                        sandbox_constants.YA_YT_TOKEN_YAV_SECRET: self.YT_TOKEN_YAV_SECRET,
                        # TODO(ovknp)(JUPITER-1700): uncomment this (and 1 more) when RMDEV-2458 is finished
                        # "package_released_resource_ttl": self.RELEASED_TTL_DAYS,
                    },
                    out={
                        resource_type: 90,  # days to store
                    }
                )

            def _prepare_data_job(self):
                return jg_build.JobGraphElementYaMakeBuildBranched(
                    task_name="USERFEAT_PREPARE_DATA_RELEASE",
                    build_item="DATA",
                    job_params={
                        "frequency": (jg_utils.TestFrequency.LAZY, None),
                    },
                    job_arrows=(
                        jg_arrows.ParentsData(
                            input_key="result_description",
                            triggers=(
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="branch_number",
                                            output_key="branch_number_for_tag",
                                        ),
                                        jg_job_data.ParentDataOutput(
                                            input_key="new_tag_number",
                                            output_key="new_tag_number",
                                        ),
                                    )
                                ),
                            ),
                            transform=lambda branch_number, new_tag_number: "Release userfeat/stable-{}-{}".format(
                                branch_number,
                                new_tag_number,
                            ),
                        ),
                    ),
                    ctx={
                        "result_released_ttl": self.RELEASED_TTL_DAYS,
                    },
                    out={
                        # Longer TTL since we will most certainly need it later
                        "USERFEAT_DATA_RELEASE": 90  # days to store
                    },
                )

            @property
            def _branch_part(self):
                default_branch_part = super(self.__class__, self)._branch_part
                build_browse = self._build_binaries_job(self.SysNames.BROWSE)
                build_counters = self._build_binaries_job(self.SysNames.COUNTERS)
                build_misc = self._build_binaries_job(self.SysNames.MISC)
                build_search = self._build_binaries_job(self.SysNames.SEARCH)
                build_userdata = self._build_binaries_job(self.SysNames.USERDATA)
                build_fast_userdata = self._build_binaries_job(self.SysNames.FAST_USERDATA)
                build_webmaster = self._build_binaries_job(self.SysNames.WEBMASTER)

                prepare_data_release = self._prepare_data_job()
                acceptance_arrows = [
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="DATA",
                    )
                ]
                for sysname in self.SYSTEMS.keys():
                    acceptance_arrows.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="{}_BINARIES".format(sysname),
                        ) if sysname != self.SysNames.COMMON else None,
                    )
                    acceptance_arrows.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="CMPY_{}".format(sysname),
                        )
                    )
                acceptance_test = jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                    job_arrows=acceptance_arrows,
                )

                branch_part = [
                    build_browse,
                    build_counters,
                    build_misc,
                    build_search,
                    build_userdata,
                    build_fast_userdata,
                    build_webmaster,
                    prepare_data_release,
                    acceptance_test,
                ]

                return default_branch_part + branch_part

            def _get_yt_attribute_path(self, period_type):
                if period_type == self.PeriodType.FAST:
                    return self.FAST_YT_RELEASE_ATTRIBUTE_PATH
                if period_type == self.PeriodType.LONG:
                    return self.LONG_YT_RELEASE_ATTRIBUTE_PATH
                raise Exception("unknown period type {}".format(period_type))

            def _release_job(self, period_type, policy):
                triggers = []
                if period_type == self.PeriodType.LONG:
                    for sysname in self.SYSTEMS.keys():
                        if sysname == self.SysNames.FAST_USERDATA:
                            continue
                        bin_resource_type = self.SYSTEMS[sysname].binaries_type
                        cmpy_resource_type = self.SYSTEMS[sysname].cmpy_type
                        if bin_resource_type:
                            triggers.append(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="{}_BINARIES".format(sysname),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="{}_bin_id".format(sysname),
                                            resource_name=bin_resource_type,
                                        ),
                                    ),
                                )
                            )
                        if cmpy_resource_type:
                            triggers.append(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="CMPY_{}".format(sysname),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="{}_cmpy_id".format(sysname),
                                            resource_name=cmpy_resource_type,
                                        )
                                    )
                                )
                            )
                else:
                    triggers.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="{}_BINARIES".format("FAST_USERDATA"),
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="{}_bin_id".format("FAST_USERDATA"),
                                    resource_name="USERFEAT_FAST_USERDATA_BINARIES_RELEASE",
                                ),
                            ),
                        )
                    )
                    triggers.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="CMPY_{}".format("FAST_USERDATA"),
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="{}_cmpy_id".format("FAST_USERDATA"),
                                    resource_name="USERFEAT_CMPY_FAST_USERDATA",
                                )
                            )
                        )
                    )
                    triggers.append(
                        jg_job_triggers.JobTriggerBuild(
                            job_name_parameter="CMPY_{}".format("COMMON"),
                            parent_job_data=(
                                jg_job_data.ParentDataResource(
                                    input_key="{}_cmpy_id".format("COMMON"),
                                    resource_name="USERFEAT_CMPY_COMMON",
                                )
                            )
                        )
                    )
                return jg_release.JobGraphElementReleaseBranched(
                    task_name="USERFEAT_PREPARE_FOR_DEPLOY",
                    release_to=rm_const.ReleaseStatus.stable,
                    release_item="{}_{}".format(period_type.upper(), policy.upper()),
                    job_params={
                        "ctx": {
                            "yt_cluster": self.USERFEAT_YT_CLUSTER,
                            "yt_attribute_path": self._get_yt_attribute_path(period_type),
                            "vault_yt_token": self.USERFEAT_YT_VAULT_TOKEN,
                            "mark_resources_as_released": True,
                            "need_send_st": True,
                            "policy": policy,
                        },
                    },
                    job_arrows=(
                        jg_arrows.ParamsData(
                            input_key="revision_number",
                            transform=lambda x, rm_config: x.revision,
                        ),
                        jg_arrows.ParentsData(
                            input_key="minor_release_number",
                            triggers=(
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="new_tag_number",
                                            output_key="new_tag_number",
                                        ),
                                    )
                                ),
                            ),
                            transform=lambda new_tag_number: new_tag_number,
                        ),
                        jg_arrows.ParentsData(
                            input_key="resources",
                            triggers=[
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="DATA",
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="data_res",
                                            resource_name="USERFEAT_DATA_RELEASE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="MRAPPS",
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="mrapps_id",
                                            resource_name="USERFEAT_MRAPPS_RELEASE",
                                        ),
                                    ),
                                ),
                            ] + triggers,
                            transform=jg_utils.args_to_list,
                        )
                    ),
                )

            def _action_release_job(self, period_type, policy):
                return jg_release.JobGraphElementActionReleaseBranched(
                    release_to=rm_const.ReleaseStatus.stable,
                    release_item="{}_{}".format(period_type.upper(), policy.upper()),
                )

            @property
            def _release(self):
                default_release_part = super(self.__class__, self)._release

                release_to_long_forced = self._release_job(self.PeriodType.LONG, "forced")
                action_release_to_long_forced = self._action_release_job(self.PeriodType.LONG, "forced")
                release_to_long_regular = self._release_job(self.PeriodType.LONG, "regular")
                action_release_to_long_regular = self._action_release_job(self.PeriodType.LONG, "regular")
                release_to_fast_forced = self._release_job(self.PeriodType.FAST, "forced")
                action_release_to_fast_forced = self._action_release_job(self.PeriodType.FAST, "forced")
                release_to_fast_regular = self._release_job(self.PeriodType.FAST, "regular")
                action_release_to_fast_regular = self._action_release_job(self.PeriodType.FAST, "regular")

                release_part = [
                    release_to_long_forced,
                    action_release_to_long_forced,
                    release_to_long_regular,
                    action_release_to_long_regular,
                    release_to_fast_forced,
                    action_release_to_fast_forced,
                    release_to_fast_regular,
                    action_release_to_fast_regular,
                ]
                return default_release_part + release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                result = dict()
                have_no_extraction_job = (
                    IntegrationTestNames.USERDATA_CLEAN_EXPORTS_TEST,
                    IntegrationTestNames.WEBMASTER_HOST2VEC,
                )
                for integration_test_name in IntegrationTestNames:
                    if integration_test_name in have_no_extraction_job:
                        continue
                    test_name = rm_const.JobTypes.rm_job_name(
                        job_type=rm_const.JobTypes.TEST,
                        component_name=uppercase_name,
                        parametrized=make_extraction_job_name(integration_test_name),
                    )
                    result[test_name] = rm_const.TestFrequencies.EACH_REV_TEST

                clean_exports_test_name = rm_const.JobTypes.rm_job_name(
                    job_type=rm_const.JobTypes.BUILD,
                    component_name=uppercase_name,
                    parametrized=IntegrationTestNames.USERDATA_CLEAN_EXPORTS_TEST.value,
                )
                result[clean_exports_test_name] = rm_const.TestFrequencies.EACH_REV_TEST

                host2vec_test_name = rm_const.JobTypes.rm_job_name(
                    job_type=rm_const.JobTypes.BUILD,
                    component_name=uppercase_name,
                    parametrized=IntegrationTestNames.WEBMASTER_HOST2VEC.value,
                )
                result[host2vec_test_name] = rm_const.TestFrequencies.EACH_REV_TEST

                return result

            @property
            def ignore_match(self):
                common_ignored = super(self.__class__, self).ignore_match
                uppercase_name = self.name.upper()
                return common_ignored + [
                    "{}_BUILD_CLUSTERMASTER_CONFIGS".format(uppercase_name),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.TEST, self.name, "NEED_TO_BACKPORT"),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.BUILD, rm_const.RMNames.USERFEAT_VIEWERS),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.RELEASE, rm_const.RMNames.USERFEAT_VIEWERS, rm_const.ReleaseStatus.stable),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.ACTION_RELEASE, rm_const.RMNames.USERFEAT_VIEWERS, rm_const.ReleaseStatus.stable),
                ]

    class Releases(configs.ReferenceBranchedConfig.Releases):
        """Releases configuration"""
        """ keep it in sync with the USERFEAT_PREPARE_FOR_DEPLOY input resources """
        resources_info = [
            configs.ReleasedResourceInfo(
                name="userfeat_cmpy_common",
                resource_type="USERFEAT_CMPY_COMMON",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_cmpy_userdata",
                resource_type="USERFEAT_CMPY_USERDATA",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_cmpy_user_search",
                resource_type="USERFEAT_CMPY_USER_SEARCH",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_cmpy_user_browse",
                resource_type="USERFEAT_CMPY_USER_BROWSE",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_data_release",
                resource_type="USERFEAT_DATA_RELEASE",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_misc_binaries_release",
                resource_type="USERFEAT_MISC_BINARIES_RELEASE",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_browse_binaries_release",
                resource_type="USERFEAT_BROWSE_BINARIES_RELEASE",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_counters_binaries_release",
                resource_type="USERFEAT_COUNTERS_BINARIES_RELEASE",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_search_binaries_release",
                resource_type="USERFEAT_SEARCH_BINARIES_RELEASE",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_userdata_binaries_release",
                resource_type="USERFEAT_USERDATA_BINARIES_RELEASE",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_cmpy_user_counters",
                resource_type="USERFEAT_CMPY_USER_COUNTERS",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_cmpy_miscdata",
                resource_type="USERFEAT_CMPY_MISCDATA",
            ),
            configs.ReleasedResourceInfo(
                name="userfeat_mrapps_release",
                resource_type="USERFEAT_MRAPPS_RELEASE",
            ),
        ]
        release_followers_permanent = [
            "tarum",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        """Notifications configuration"""

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            """Telegram notifications configuration"""
            chats = ["userfeat_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            """Startrek notifications configuration"""
            assignee = "tarum"
            queue = "USERFEATRELEASE"
            dev_queue = "USERFEAT"
            summary_template = u"Userfeat release {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "niknik",
                "tarum",
            ]
            add_commiters_as_followers = False
            deadline = 14

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        """Changelog configuration"""
        wiki_page = "JandeksPoisk/KachestvoPoiska/userfeatures/changelogs/"
        dirs = [
            "arcadia/quality/click_machine/",
            "arcadia/quality/traffic/",
            "arcadia/quality/relev_tools/xfactor/",
            "arcadia/quality/factors/",
            "arcadia/quality/lib/query_norm/",
            "arcadia/quality/miscdata/",
            "arcadia/quality/seinfo_monitor/",
            "arcadia/quality/user_search/",
            "arcadia/quality/user_browse/",
            "arcadia/quality/user_counters/",
            "arcadia/quality/userdata/",
        ]
        review_groups = [
            "userfeat",
        ]
