# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class AliceEvoJobTypes(object):
    ALICE_EVO_CONTEXT_FAST = "ALICE_EVO_CONTEXT_FAST"
    ALICE_EVO_CONTEXT_PACKAGE = "ALICE_EVO_CONTEXT_PACKAGE"
    ALICE_EVO_CONTEXT_SERVICE = "ALICE_EVO_CONTEXT_SERVICE"
    ALICE_EVO_DIFF_CONFIG = "ALICE_EVO_DIFF_CONFIG"
    ALICE_EVO_INTEGRATION_TESTS = "ALICE_EVO_INTEGRATION_TESTS_WRAPPER"
    ALICE_EVO_MARKER_TESTS = "ALICE_EVO_MARKER_TESTS_WRAPPER"
    ALICE_EVO_UE_2_E = "ALICE_EVO_UE_2_E"
    ALICE_GOODWIN_E2E_TESTS = "SANDBOX_CI_MEGAMIND_GOODWIN_E2E"
    ALICE_EVO_MEASURE_QUALITY = "ALICE_EVO_MEASURE_QUALITY"


class VinsCfg(configs.ReferenceBranchedConfig):
    name = "vins"
    responsible = "zhigan"

    class MergesCfg(configs.ReferenceBranchedConfig.MergesCfg):
        permissions = configs.MergePermissions(
            permission_type=rm_const.PermissionType.ALLOWED,
            people_groups=configs.PeopleGroups(
                abc_services=[],
                logins={"shuster", "avitella", "ispetrukhin", "sparkle", "zhigan", "alexanderplat", "nkodosov", "yagafarov"},
                staff_groups=[],
            ),
        )

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "VINS"
        branch_task_owner = "BASS"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            vins_build_arrow = jg_job_triggers.JobTriggerBuild(
                parent_job_data=jg_job_data.ParentDataDict(
                    "component_resources",
                    "vins_package",
                    "VINS_PACKAGE",
                ),
            )
            megamind_build_arrow = jg_job_triggers.JobTriggerBuild(
                parent_job_data=jg_job_data.ParentDataDict(
                    "component_resources",
                    "megamind_package",
                    "MEGAMIND_PACKAGE",
                ),
                job_name_parameter="MEGAMIND_STANDALONE",
            )

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(
                        ctx={"use_previous_branch_as_baseline": True}
                    ),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_test.JobGraphElementTestTagCommon(
                        task_name="BASS_REGRESS_TESTS_TASK",
                        job_params={
                            "job_name_parameter": "TESTPALM",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerStartrek(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="target_ticket",
                                        output_key="startrek_issue",
                                    ),
                                )
                            )
                        ),
                        ctx={
                            "target_service": "megamind",
                            "testsuite": "alice",
                        }
                    ),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="TESTPALM",
                            )
                        ),
                        next_revision_custom_schedule=jg_utils.CustomSchedule(
                            time_interval_list=[(2, 2)],
                            weekday_list=[3, 6],
                            once_a_day=True
                        )
                    ),
                ]

            @property
            def _trunk_part(self):
                trunk_part = super(self.__class__, self)._trunk_part
                trunk_part.append(jg_build.JobGraphElementBuildTrunk(
                    task_name="BUILD_MEGAMIND_STANDALONE_CI",
                    job_params={
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        "observed_paths": ["alice/megamind"],
                    },
                    build_item="MEGAMIND_STANDALONE_TRUNK",
                ))
                return trunk_part

            @property
            def _branch_part(self):
                default_branch_part = super(self.__class__, self)._branch_part

                class ChartParams(jg_utils.ChartParams):
                    ChartLines = ("avg", "q50", "q75", "q90", "q95", "q99", "q999", "rate")
                    ChartYAxes = (
                        jg_utils.ChartParams.YAxis(
                            "Begemot answer time (milliseconds)",
                            ("avg", "q50", "q75", "q90", "q95", "q99", "q999",),
                        ),
                        jg_utils.ChartParams.YAxis("Perf requests success rate (percents)", ("rate",))
                    )

                def GetDataFromTask(params):
                    begemot_metrics = params.ctx.get("begemot_metrics", {})
                    return {
                        "avg": begemot_metrics.get("begemot_score", 0.0),
                        "q50": begemot_metrics.get("q50", 0.0),
                        "q75": begemot_metrics.get("q75", 0.0),
                        "q90": begemot_metrics.get("q90", 0.0),
                        "q95": begemot_metrics.get("q95", 0.0),
                        "q99": begemot_metrics.get("q99", 0.0),
                        "q999": begemot_metrics.get("q99.9", 0.0),
                        "rate": begemot_metrics.get("success_rate", 0.0),
                    }

                branch_part = [
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_VINS_PACKAGE",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "client_tags": "SSD",
                                    "cores": 56,
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        ctx={
                            "force_vcs_info_update": True,
                            "packages": "alice/vins/packages/vins_package.json",
                            "raw_package": True,
                            "resource_type": "VINS_PACKAGE",
                            "ya_yt_store": True,
                        },
                        out={"VINS_PACKAGE": 90},
                    ),
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_MEGAMIND_PACKAGE",
                        build_item="MEGAMIND_STANDALONE",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "client_tags": "SSD",
                                    "cores": 56,
                                },
                            },
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        ctx={
                            "force_vcs_info_update": True,
                            "packages": "alice/megamind/deploy/packages/megamind_standalone.json",
                            "raw_package": False,
                            "resource_type": "MEGAMIND_PACKAGE",
                            "ya_yt_dir": "//home/alice/vins_yamake_cache",
                            "ya_yt_proxy": "hahn.yt.yandex.net",
                            "ya_yt_put": True,
                            "ya_yt_store": True,
                            "ya_yt_token_vault_name": "robot-voiceint_yt_token",
                            "ya_yt_token_vault_owner": "VINS",
                        },
                        out={"MEGAMIND_PACKAGE": 90},
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ALICE_BEGEMOT_MEGAMIND_PERF_TEST",
                        job_params={
                            "job_name_parameter": "ALICE_BEGEMOT_MEGAMIND_PERFORMANCE",
                            "get_data_from_task": GetDataFromTask,
                            "chart": ChartParams(),
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="test_pkg",
                                        resource_name="VINS_PACKAGE",
                                    )
                                )
                            )
                        ),
                        ctx={
                            "reqs_pkg": "1003270617",
                        },
                    ),
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="acceptance",
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            self.vins_build_arrow,
                            self.megamind_build_arrow,
                        ),
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name=AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="evo_in_tag_num",
                                        output_key="new_tag_number",
                                    ),
                                )
                            ),
                            jg_arrows.ParamsData("evo_in_branch_num", jg_utils.get_major_release_number),
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name=AliceEvoJobTypes.ALICE_EVO_CONTEXT_PACKAGE,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_EVO_CONTEXT_PACKAGE,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="evo_in_vins_package",
                                        resource_name="VINS_PACKAGE",
                                    )
                                )
                            )
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name=AliceEvoJobTypes.ALICE_EVO_CONTEXT_SERVICE,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_EVO_CONTEXT_SERVICE,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="acceptance"
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="evo_in_branch_num",
                                        output_key="evo_ctx_branch_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="evo_in_tag_num",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                )
                            ),
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        ctx={
                            "component_name": "vins",
                        },
                        task_name=AliceEvoJobTypes.ALICE_EVO_DIFF_CONFIG,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_EVO_DIFF_CONFIG,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="branch_num",
                                        output_key="evo_ctx_branch_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="tag_num",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_ticket",
                                        output_key="evo_out_release_ticket",
                                    ),
                                )
                            )
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name=AliceEvoJobTypes.ALICE_EVO_INTEGRATION_TESTS,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_EVO_INTEGRATION_TESTS,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        ctx={
                            "repeat_failed_test": True,
                            "run_call_owner_subtask": True,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_SERVICE,
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="beta_name",
                                        output_key="evo_out_beta_name",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="megamind_url",
                                        output_key="evo_out_megamind_url",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="launch_type",
                                        output_key="evo_ctx_component_name",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="branch_number",
                                        output_key="evo_ctx_branch_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="tag_number",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_ticket",
                                        output_key="evo_out_release_ticket",
                                    ),
                                )
                            ),
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name=AliceEvoJobTypes.ALICE_EVO_INTEGRATION_TESTS,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_EVO_MARKER_TESTS,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        ctx={
                            "run_call_owner_subtask": True,
                            "subtask_type": "ALICE_EVO_MARKER_TESTS",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_SERVICE,
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="beta_name",
                                        output_key="evo_out_beta_name",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="megamind_url",
                                        output_key="evo_out_megamind_url",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="launch_type",
                                        output_key="evo_ctx_component_name",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="branch_number",
                                        output_key="evo_ctx_branch_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="tag_number",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_ticket",
                                        output_key="evo_out_release_ticket",
                                    ),
                                )
                            ),
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name=AliceEvoJobTypes.ALICE_GOODWIN_E2E_TESTS,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_GOODWIN_E2E_TESTS,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_SERVICE,
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="megamind_url",
                                        output_key="evo_out_megamind_url",
                                    )
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_ticket",
                                        output_key="evo_out_release_ticket",
                                    )
                                )
                            ),
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name=AliceEvoJobTypes.ALICE_EVO_UE_2_E,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_EVO_UE_2_E,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="component_name",
                                        output_key="evo_ctx_component_name",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="branch_num",
                                        output_key="evo_ctx_branch_num",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="tag_num",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_ticket",
                                        output_key="evo_out_release_ticket",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_SERVICE,
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="megamind_url",
                                        output_key="evo_out_megamind_url",
                                    ),
                                )
                            ),
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name=AliceEvoJobTypes.ALICE_EVO_MEASURE_QUALITY,
                        job_params={
                            "job_name_parameter": AliceEvoJobTypes.ALICE_EVO_MEASURE_QUALITY,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter=AliceEvoJobTypes.ALICE_EVO_CONTEXT_FAST,
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="evo_in_release_ticket",
                                        output_key="evo_out_release_ticket",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="evo_in_stable_svn_path",
                                        output_key="evo_out_sandbox_stable_svn_path",
                                    ),
                                    jg_job_data.ParentDataOutput(
                                        input_key="evo_in_release_svn_path",
                                        output_key="evo_out_sandbox_svn_path",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="evo_in_tag_num",
                                        output_key="evo_ctx_tag_num",
                                    ),
                                )
                            ),
                        )
                    ),
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="acceptance",
                            )
                        )
                    )
                ]
                return default_branch_part + branch_part

            @property
            def _release(self):
                default_release_part = [
                    jg_release.JobGraphElementNewTag(),
                    jg_release.JobGraphElementChangelogFinal(ctx={"use_previous_branch_as_baseline": True}),
                ]
                release_part = []
                for release_stage in [
                    rm_const.ReleaseStatus.stable,
                    rm_const.ReleaseStatus.testing,
                ]:
                    release_stage_job_params = None if release_stage != rm_const.ReleaseStatus.testing else {
                        "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                    }
                    release_part.extend([
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                self.vins_build_arrow,
                                self.megamind_build_arrow,
                                jg_job_triggers.JobTriggerNewTag([
                                    jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
                                ]),
                            )
                        ),
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                            job_params=release_stage_job_params
                        ),
                        jg_release.JobGraphElementReleasePushBranched(
                            task_name="RELEASE_RM_COMPONENT_2",
                            release_to=release_stage,
                            job_params={
                                "ctx": {
                                    "deploy_system": rm_const.DeploySystem.nanny_push.name,
                                }
                            },
                            job_arrows=(
                                self.vins_build_arrow,
                                self.megamind_build_arrow,
                                jg_job_triggers.JobTriggerNewTag([
                                    jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                                    jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                                ]),
                            )
                        ),
                        jg_release.JobGraphElementActionPushReleaseBranched(release_to=release_stage),
                    ])
                return default_release_part + release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                freq_patch = {
                    "_ALICE_BEGEMOT_MEGAMIND_PERFORMANCE_TEST": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_GENERATE_YAPPY_BETA_VINS": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_VINS_PERFORMANCE_TEST": rm_const.TestFrequencies.LAZY_TEST,
                    "_TEST__VINS__PERFORMANCE": rm_const.TestFrequencies.LAZY_TEST,
                    "VINS_PERFORMANCE_TEST": rm_const.TestFrequencies.LAZY_TEST,
                }
                tests.update(freq_patch)
                return tests

            @property
            def ignore_match(self):
                common_ignored = super(self.__class__, self).ignore_match
                vins_ignored = [
                    "ALICE_EVO_CHECKPOINT_FAST_TE",
                    "ALICE_EVO_CHECKPOINT_PACKAGE_TE",
                    "ALICE_EVO_INTEGRATION_TESTS_WRAPPER_TE",
                    "ALICE_EVO_PERF_WRAPPER_TE",
                    "ALICE_EVO_TRIGGER_ALFA",
                    "ALICE_EVO_TRIGGER_BRAVO",
                    "ALICE_EVO_TRIGGER_CHARLIE",
                    "ALICE_EVO_UE_2_E_TE",
                    "ALICE_EVO_YAPPY_WRAPPER_TE",
                    "ALICE_PERFORMANCE_TEST",
                    "BUILD_MEGAMIND_PACKAGE",
                    "BUILD_VINS_PACKAGE",
                    "MEGAMIND_DIFF_TEST",
                    "MEGAMIND_INTEGRATION_TEST",
                    "MEGAMIND_PERFORMANCE_TEST",
                    "MEGAMIND_PERF_TEST",
                    "VINS_PERFORMANCE_TEST",
                    "_TEST__VINS__PERFORMANCE",
                    "_TEST__VINS__TESTPALM",
                    "_VINS_UMBRELLA_ACCEPT_SCHEDULED",
                ]
                return common_ignored + vins_ignored

    class ReleaseViewer(configs.ReferenceBranchedConfig.ReleaseViewer):

        RELEASE_SAW_GRAPH = (
            "https://solomon.yandex-team.ru/?"
            "project=release_machine&cluster=default&service=release_statistics&description=Release+Saw&"
            "l.sensor=release_saw*&"
            "l.component_name=vins&"
            "graph=auto&stack=false&"
            "min=0&b=180d&e=&aggr=max&downsampling=byInterval&grid=1d&downsamplingAggr=max&graphOnly=y"
        )

        statistics_page_charts = [
            RELEASE_SAW_GRAPH,
        ] + configs.ReferenceBranchedConfig.ReleaseViewer.statistics_page_charts

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="vins_package",
                    data=ri.SandboxResourceData("VINS_PACKAGE"),
                    deploy_infos=[
                        ri.NannyDeployInfo(
                            [
                                ri.DeployService("megamind_{}".format(i), tags=["prod"])
                                for i in rm_const.MAIN_LOCATIONS
                            ] + [
                                ri.DeployService("megamind_hamster", tags=["hamster"])
                            ]
                        ),
                        ri.NannyDeployInfo(
                            ri.DeployService("megamind_rc", tags=["rc"]),
                            stage=rm_const.ReleaseStatus.testing
                        )
                    ],
                ),
                ri.ReleasableItem(
                    name="megamind_package",
                    data=ri.SandboxResourceData("MEGAMIND_PACKAGE"),  # TODO: add deploy_infos
                )
            ]

        release_followers_permanent = [
            "zubchick",
            "sparkle",
            "zhigan",
        ]
        allow_robots_to_release_stable = True
        allow_old_releases = {
            rm_const.ReleaseStatus.testing: False,
            rm_const.ReleaseStatus.stable: True,
        }

    class Notify(configs.ReferenceBranchedConfig.Notify):

        notifications = [
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="[vins] Alice Releases 2.0",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="[vins] Alice Releases 2.0",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
        ]

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = [
                "vins-megamind-releases@yandex-team.ru",
            ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["bass_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = "https://t.me/joinchat/AAAAAEUzgJcHCRaSP_o0cw"

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = configs.Responsible(abc=configs.Abc(component_id=2651), login="zhigan")
            queue = "ALICERELEASE"
            dev_queue = "ALICE"
            summary_template = u"VINS {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "zubchick",
                "okras",
                "sparkle",
                "ran1s",
                "zhigan",
            ]
            add_commiters_as_followers = False
            deadline = 0
            hide_commits_under_cut = True
            important_changes_limit = 1000
            write_merges_in_table = False
            nanny_reports = False
            use_task_author_as_assignee = False

            @property
            def ticket_description_prefix(self):
                custom_description = [
                    '<{Памятка дежурному релиз-менеджеру',
                    '',
                    '((https://wiki.yandex-team.ru/alice/vins/megamindreleasemanagement/ '
                    'Документация по релиз-менеджменту Megamind))',
                    'Релиз-менеджер заполняет таблицу ниже. В колонку **"Ссылка"** нужно дать'
                    ' ссылки на подробности соответствующих проверок, а в колонке'
                    ' **"ОК/НЕОК"** нужно оставить что-то одно из двух, причем решение'
                    ' принимает **"Ответственный"**. Где !!(зел)ОК!! – можно катить в'
                    ' продакшен, а !!НЕОК!! – нельзя катить в продакшен. В продакшн'
                    ' выкатывает дежурный по рантайму Megamind',
                    '',
                    'Чаты:',
                    '* ((https://t.me/joinchat/BBPaeU754t2iM8ROyTY1LQ Чат Дежурства по Алисе))',
                    '* ((https://t.me/joinchat/EHnbnBCKkUbWHGpJHw5GVQ Чат тестирования))',
                    '}>',
                    '',
                    '===CHECKS',
                    '#|',
                    '|| **Проверка** | **Ответственный** | '
                    '**ОК/НЕОК** | **Cсылка** ||',
                ]
                table = [
                    ['Интеграционные тесты EVO',
                     'Релиз-менеджер',
                     'Cсылка на комментарий.'],

                    ['Качество',
                     'кто:olegator',
                     'Ссылка на комментарий с результатами измерения качества модели на валидационных корзинках.'],

                    ['Ручное тестирование',
                     'Дежурный тестировщик',
                     'Ссылка на комментарий с результатами ручного тестирования.'],

                    ['UE2E',
                     'Релиз-менеджер',
                     'Ссылка на комментарий с вердиктом по UE2E. ((https://wiki.yandex-team.ru/Alice/VINS/Vykatka-reliza-VINS/#rezultatyocenivaetreliz-inzhenertak Как оценить))'],

                    ['Валидация включенных сценариев',
                     'кто:zubchick кто:yagafarov',
                     'Ссылка на дифф.'],
                ]
                custom_description.extend(
                    '|| {} ||'.format(' | '.join([check, responsible, '!!(зел)ОК!!/!!НЕОК!!?', link]))
                    for check, responsible, link in table
                )
                custom_description.append('|#')
                return '\n'.join(custom_description)

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "Alice/VINS/releasechangelogs/"
        dirs = [
            "arcadia/alice/bass",
            "arcadia/alice/library",
            "arcadia/alice/megamind",
            "arcadia/alice/nlg",
            "arcadia/alice/vins",
        ]
        svn_paths_filter = configs.ChangelogPathsFilter(
            rm_const.PermissionType.BANNED,
            [
                "arcadia/alice/vins/api/crm_bot",
                "arcadia/alice/vins/api/vins_api/webim",
                "arcadia/alice/vins/apps/crm_bot",
                "arcadia/alice/vins/cit_configs/components/webim-beru-crm",
                "arcadia/alice/vins/resources/crm_bot",
                "arcadia/alice/library/java",
                "arcadia/alice/kronstadt",
            ]
        )

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_prefix = "vins"

    class Yappy(yappy_cfg.YappyBaseCfg):
        betas = {
            "acceptance": yappy_cfg.YappyTemplateCfg(
                template_name="megamindvinsbass-hamster",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="vinsbass",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(param_name="vins_package", local_path="package"),
                        ],
                        parent_service="megamind_hamster",
                    ),
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="megamind",
                        resources=[
                            yappy_cfg.YappyParametrizedResource(param_name="megamind_package", local_path="megamind_standalone.tar.gz"),
                        ],
                        parent_service="megamind_standalone_hamster",
                    ),
                ]
            ),
        }
        working_betas_limit = 5
