# -*- coding: utf-8 -*-
"""
Release machine for Xurma Taxi graph
"""
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.xurma.rm as xurma_rm


class XurmaTaxiCfg(configs.ReferenceBranchedConfig):
    name = "xurma_taxi"
    responsible = xurma_rm.RESPONSIBLE

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        dirs = [
            "arcadia/quality/antifraud/xurma",
        ]
        wiki_page = "JandeksPoisk/Antirobots/Xurma/releases/releases-taxi/"
        wiki_page_owner = xurma_rm.WIKI_PAGE_OWNER

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="xurma_taxi_data",
                    data=ri.SandboxResourceData("XURMA_TAXI_DATA"),
                    deploy_infos=[
                        xurma_rm.ya_deploy_stage("xurma-taxi-prod"),
                        xurma_rm.ya_deploy_stage("xurma-taxi-preprod", stage=rm_const.ReleaseStatus.prestable),
                    ],
                )
            ]

        allow_robots_to_release_stable = True

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = xurma_rm.TRUNK_TASK_OWNER
        testenv_db_owners = xurma_rm.TESTENV_DB_OWNERS

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                """
                    Prerelease part of job graph for Testenv with arrows and dependencies
                    :return: list with jobs
                """
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    xurma_rm.JobGraphElementChangelogProcessor(ctx={
                        "graph_name": "taxi",
                    }),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrekBase(job_arrows=(
                        jg_arrows.JobTrigger(
                            "CHANGELOG_PROCESSOR",
                            jg_data.ParentDataResource("changelog", "RELEASE_MACHINE_CHANGELOG"),
                        ),
                    )),
                    jg_prerelease.JobGraphElementWikiBase(job_arrows=(
                        jg_arrows.JobTrigger(
                            "CHANGELOG_PROCESSOR",
                            jg_data.ParentDataResource("changelog", "RELEASE_MACHINE_CHANGELOG"),
                        ),
                        jg_triggers.JobTriggerStartrek(),
                    )),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(),
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="XURMA_BUILD_TAXI",
                        out={
                            "XURMA_TAXI_DATA": 30,
                            "XURMA_DEBUG": 30,
                        },
                        ctx={
                            "use_aapi_fuse": True,
                            "packages": "quality/antifraud/xurma/deploy/xurma/graphs/taxi.json;quality/antifraud/xurma/deploy/xurma/debug.json",
                            "resource_type": "XURMA_TAXI_DATA;XURMA_DEBUG",
                        },
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                for release_to in (rm_const.ReleaseStatus.stable, rm_const.ReleaseStatus.prestable):
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_to,
                    ))
                    release_part.append(jg_release.JobGraphElementReleaseBranched(
                        release_to=release_to,
                        job_arrows=[
                            jg_triggers.JobTriggerBuild(
                                parent_job_data=[
                                    jg_data.ParentDataDict(
                                        input_key="component_resources",
                                        dict_key="xurma_taxi_data",
                                        resource_name="XURMA_TAXI_DATA",
                                    ),
                                ],
                            ),
                        ]
                    ))
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def ignore_prefix(self):
                return super(self.__class__, self).ignore_prefix + [
                    "_CHANGELOG_PROCESSOR",
                ]

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "xurma/graphs/taxi"
        tag_name = "xurma/graphs/taxi"

    class Notify(configs.ReferenceBranchedConfig.Notify):

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            queue = xurma_rm.STARTREK_QUEUE
            assignee = xurma_rm.STARTREK_ASSIGNEE
            ticket_type = "Release"
            summary_template = u"Xurma Taxi graph {}"
            add_commiters_as_followers = False
            followers = []
            workflow = xurma_rm.STARTREK_WORKFLOW
