# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import os

from sandbox.projects.common.constants import constants as common_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.job_graph.presets import ci_presets
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


PLATFORMS = ["linux", "darwin"]


class YaToolReleasedComponentCfg(configs.ReferenceTaggedConfig):
    component_group = "ya_tool_based"

    class Releases(configs.ReferenceTaggedConfig.Releases):
        allow_robots_to_release_stable = True
        deploy_system = rm_const.DeploySystem.ya_tool

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name=self.name,
                    data=ri.SandboxResourceData("ARCADIA_PROJECT_TGZ", ttl=30),
                    deploy_infos=[ri.YaToolDeployInfo(ri.DeployService(
                        "bottles.{}.formula.sandbox_id".format(self.name.lower())
                    ))]
                ),
            ]

    class Testenv(configs.ReferenceTaggedConfig.Testenv):
        trunk_task_owner = "SEARCH-RELEASERS"

        class JobGraph(configs.ReferenceTaggedConfig.Testenv.JobGraph):
            @property
            def _target(self):
                raise NotImplementedError

            @property
            def _binary_name(self):
                raise NotImplementedError

            @property
            def _tag_part(self):
                tag_part = super(YaToolReleasedComponentCfg.Testenv.JobGraph, self)._tag_part
                tag_part.append(jg_build.JobGraphElementBuildTagged(
                    task_name="BUILD_ARCADIA_PROJECT_FOR_ALL",
                    ctx={
                        "project": os.path.join(self._target, self._binary_name),
                        "platforms_list": PLATFORMS,
                        "strip_binaries": True,
                        "backup_to_mds": False,
                        common_const.USE_AAPI_FUSE: True,
                        common_const.USE_ARC_INSTEAD_OF_AAPI: True,
                        common_const.BUILD_SYSTEM_KEY: common_const.SEMI_DISTBUILD_BUILD_SYSTEM,
                    },
                    out={
                        "ARCADIA_PROJECT_TGZ": 30,
                        "PLATFORM_MAPPING": 30,
                    },
                    filter_targets=[self._target]
                ))
                return tag_part

            @property
            def _release(self):
                release_part = super(YaToolReleasedComponentCfg.Testenv.JobGraph, self)._release
                release_part.append(jg_release.JobGraphElementNewTagTagged())
                release_part.append(jg_prerelease.JobGraphElementPreliminaryChangelogTagged())
                release_part.append(jg_release.JobGraphElementReleaseTagged(
                    task_name="RELEASE_RM_COMPONENT_2",
                    release_to=rm_const.ReleaseStatus.stable,
                    job_params={
                        "ctx": {
                            "deploy_system": rm_const.DeploySystem.ya_tool.name,
                            "change_json_path": "bottles.{}.formula.sandbox_id".format(self.name.lower()),
                        },
                        "filter_targets": [self._target],
                    },
                    job_arrows=(
                        jg_job_triggers.JobTriggerBuild(parent_job_data=jg_job_data.ParentDataId("build_task_id")),
                        jg_job_triggers.JobTriggerNewTag(
                            jg_job_data.ParentDataOutput("major_release_num", "new_tag_number")
                        ),
                    ),
                ))
                release_part.append(jg_release.JobGraphElementActionReleaseTagged(
                    release_to=rm_const.ReleaseStatus.stable,
                    job_params={
                        "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                        "next_revision_custom_schedule": jg_utils.CustomSchedule(
                            time_interval_list=[(4, 5)],
                            last_run_finished=False,
                            once_a_day=True,
                        ),
                        "filter_targets": [self._target],
                    },
                ))
                return release_part

    class Notify(configs.ReferenceTaggedConfig.Notify):
        use_startrek = False

    class ChangelogCfg(configs.ReferenceTaggedConfig.ChangelogCfg):
        wiki_page = ""

    def __init__(self):
        super(YaToolReleasedComponentCfg, self).__init__()
        self.changelog_cfg = self.ChangelogCfg(self, self.svn_cfg.main_url, self.responsible)


class YaToolReleasedComponentCiCfg(configs.ReferenceCIConfig):
    component_group = "ya_tool_based_ci"

    class Releases(configs.ReferenceCIConfig.Releases):
        allow_robots_to_release_stable = True
        deploy_system = rm_const.DeploySystem.ya_tool

        @property
        def _ya_tool_bottle_name(self):
            return self.name.lower()

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="{}_{}".format(self.name, platform),
                    data=ri.SandboxResourceData("ARCADIA_PROJECT_TGZ", ttl=30, attributes={"platform": platform}),
                    deploy_infos=[ri.YaToolDeployInfo(ri.DeployService(
                        "bottles.{}.formula.sandbox_id".format(self._ya_tool_bottle_name)
                    ))]
                )
                for platform in PLATFORMS
            ] + [
                ri.ReleasableItem(
                    name="{}_platform_mapping".format(self.name),
                    data=ri.SandboxResourceData("PLATFORM_MAPPING", ttl=30),
                ),
            ]

    class Notify(configs.ReferenceCIConfig.Notify):
        use_startrek = False

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = ""

    class CI(configs.ReferenceCIConfig.CI):
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "SEARCH-RELEASERS"

        class JobGraph(configs.ReferenceCIConfig.CI.JobGraph):
            @property
            def _target(self):
                raise NotImplementedError

            @property
            def _binary_name(self):
                raise NotImplementedError

            @property
            def _stages(self):
                """A list of `jg_utils.StageReleaseFrequency` objects"""
                return [
                    jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.stable, jg_utils.CheckEachCommit()),
                ]

            @property
            def _preset(self):
                path_to_task_id = self.releasable_items[0].deploy_infos[0].services[0].name
                return ci_presets.SingleBuildCiPreset(
                    build_task_name="BUILD_ARCADIA_PROJECT_FOR_ALL",
                    build_ctx={
                        "project": os.path.join(self._target, self._binary_name),
                        "platforms_list": PLATFORMS,
                        "strip_binaries": True,
                        "backup_to_mds": False,
                    },
                    release_ctx={
                        "change_json_path": path_to_task_id
                    },
                    deploy_system=rm_const.DeploySystem.ya_tool.name,
                    stages=self._stages,
                )
