# -*- coding: utf-8 -*-
from datetime import timedelta
from collections import namedtuple

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


CHAT_NAME = u"Релизные будни"
RELEASE_CONFIRM_PEOPLE = ["vstromov", "gvastash", "lelby"]


class YabsServerJobTypes(object):
    SETUP = "SETUP"
    NOTIFY = "NOTIFY"
    ANALYZE_PREPRODUCTION = "ANALYZE_PREPRODUCTION"
    REPORT_BUILD = "REPORT_BUILD"
    REPORT_FT_RESULTS = "REPORT_FT_RESULTS"
    REPORT_B2B_DIFFS = "REPORT_B2B_DIFFS"
    NOTIFY_RELEASE_TICKET_CHANGED = "NOTIFY_RELEASE_TICKET_CHANGED"
    REPORT_PERFORMANCE_DELTA = "REPORT_PERFORMANCE_DELTA"
    MOVE_STARTREK_ISSUE = "MOVE_STARTREK_ISSUE"
    GET_LAST_GOOD_REVISION = "GET_LAST_GOOD_REVISION"


YDeployService = namedtuple(
    'YDeployService',
    [
        'name',
        'resource_name',
        'resource_type',
        'package_name',
        'testing_service_ids',
        'stable_service_ids',
        'dashboard_id',
        'recipe_id',
        'transform_resource_id',
    ]
)


YDEPLOY_SERVICES = [
    YDeployService(
        name='GOALNET',
        resource_name='yabs-goalnet-binary',
        resource_type='GOALNET_RELEASE_TAR',
        package_name='yabs/server/goalnet/package.json',
        testing_service_ids=['goalnet-preproduction-1-resources'],
        stable_service_ids=['goalnet-production-1-resources'],
        dashboard_id='deploy-goalnet-testing',
        recipe_id='test_deploy_goalnet.yaml',
        transform_resource_id=lambda goalnet_release_tar_resource_id: {
            'goalnet-binary': goalnet_release_tar_resource_id,
        },
    ),
    YDeployService(
        name='WORD_NET',
        resource_name='yabs-word-net-server-layer',
        resource_type='WORD_NET_RELEASE_TAR',
        package_name='yabs/infra/deploy/box/nr-microservices/word_net/server_layer/package.json',
        testing_service_ids=['word_net_preprod_resources'],
        stable_service_ids=['word_net_prod_resources'],
        dashboard_id='deploy_word_net',
        recipe_id='deploy_word_net.yaml',
        transform_resource_id=lambda word_net_release_tar_resource_id: {
            'word-net-server': word_net_release_tar_resource_id,
        },
    ),
    YDeployService(
        name='OFFER_MATCH',
        resource_name='offers-matching-service-binary',
        resource_type='OFFER_MATCH_RELEASE_TAR',
        package_name='yabs/infra/deploy/box/nr-microservices/offer_match/package_service.json',
        testing_service_ids=[],
        stable_service_ids=[
            'offer-match-production-resources',
            'yabs-offer-match-production-sas-1-resources',
            'yabs-offer-match-production-iva-1-resources',
            'yabs-offer-match-production-vla-1-resources'
        ],
        dashboard_id='deploy-offer-match-production',
        recipe_id='deploy_offer_match.yaml',
        transform_resource_id=lambda offer_match_release_tar_resource_id: {
            'offers-matching-service-binary': offer_match_release_tar_resource_id,
        },
    ),
]


class YabsServerCfg(configs.ReferenceBranchedConfig):
    name = "yabs_server"
    responsible = configs.Responsible(abc=configs.Abc(component_id=179, schedule_slug='yabs_frontend_duty_first'), login="igorock")

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "yabs-server-trunk"
        trunk_task_owner = "YABS_SERVER_SANDBOX_TESTS"
        branch_db_template = "yabs-server-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_test.JobGraphElementTestTagCommon(
                        task_name="YABS_SERVER_GET_FT_RESULTS",
                        job_params={
                            "job_type": YabsServerJobTypes.REPORT_FT_RESULTS,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewBranch(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_number",
                                        output_key="new_branch_number",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerStartrek(),
                        ),
                    ),
                    jg_test.JobGraphElementTestTagCommon(
                        task_name="YABS_SERVER_GET_B2B_DIFFS",
                        job_params={
                            "job_type": YabsServerJobTypes.REPORT_B2B_DIFFS,
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="final_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_job_triggers.JobTriggerNewBranch(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_number",
                                        output_key="new_branch_number",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerStartrek(),
                        ),
                        ctx={
                            "database_name": "yabs-2.0",
                            "notification_is_enabled": True,
                            "wait_on_diffs": True,
                            "release_confirm_people": RELEASE_CONFIRM_PEOPLE
                        },
                    ),
                    jg_test.JobGraphElementTestTagCommon(
                        task_name="YABS_SERVER_NOTIFY_RELEASE_TICKET_CHANGED",
                        job_params={
                            "job_type": YabsServerJobTypes.NOTIFY_RELEASE_TICKET_CHANGED,
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="final_revision",
                                transform=lambda x, rm_config: x.revision,
                            ),
                            jg_job_triggers.JobTriggerNewBranch(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="release_number",
                                        output_key="new_branch_number",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerStartrek(),
                        ),
                        ctx={
                            "ticket_filed": "status.key",
                            "ticket_filed_value": "readyToDeploy",
                            "wait_until_change": True,
                            "release_confirm_people": RELEASE_CONFIRM_PEOPLE
                        },
                    ),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(
                        use_arc=True,
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type=YabsServerJobTypes.REPORT_FT_RESULTS,
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type=YabsServerJobTypes.REPORT_B2B_DIFFS,
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type=YabsServerJobTypes.NOTIFY_RELEASE_TICKET_CHANGED,
                            ),
                        )
                    ),
                ]

            @property
            def _branch_part(self):
                default_branch_part = super(self.__class__, self)._branch_part
                branch_part = [
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_YABS_SERVER",
                        ctx={
                            "build_types": ["release", "profile"],
                            "wait_for_build_type": "release",
                            "build_system": "ya",
                            "yt_proxy": "hahn",
                            "yt_dir": "//home/yabs-cs-sandbox/build-cache",
                            "compute_md5": True,
                            "compute_version": True,
                            "resource_attributes": {
                                "component_name": self.name,
                            },
                        },
                        out={
                            "BS_RELEASE_TAR": 30,
                            "BS_RELEASE_YT": 30,
                        },
                    ),
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_YABS_SERVER",
                        build_item="DEBUG",
                        ctx={
                            "build_types": ["debug"],
                            "wait_for_build_type": "debug",
                            "build_type": "debug",
                            "packages_to_build": {},
                            "tools_to_build": {},
                            "yt_store": False,
                            "resource_attributes": {
                                "component_name": self.name,
                            },
                        },
                        out={
                            "BS_RELEASE_TAR": 30,
                        },
                    ),
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="YA_MAKE_2",
                        build_item="YABS_UDF",
                        ctx={
                            "targets": "yql/udfs/yabs/module",
                            "arts": "yql/udfs/yabs/module/libyabs_udf.so",
                            "build_type": "release",
                            "result_rt": "YABS_UDF",
                            "result_single_file": True,
                        },
                        out={
                            "YABS_UDF": 30,
                        },
                    ),
                ]
                branch_part.extend(
                    jg_build.JobGraphElementBuildPackageBranched(
                        task_name="YA_PACKAGE_2",
                        package_names=service.package_name,
                        resource_names=service.resource_type,
                        job_params={
                            "job_name_parameter": service.name.upper(),
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        out={
                            service.resource_type: 30,
                        }
                    )
                    for service in YDEPLOY_SERVICES
                )
                branch_part += [
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="YABS_SERVER_SETUP_YA_MAKE",
                        job_params={
                            "job_type": YabsServerJobTypes.SETUP,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="server_resource",
                                        resource_name="BS_RELEASE_TAR",
                                    ),
                                    jg_job_data.ParentDataResource(
                                        input_key="cs_resource",
                                        resource_name="BS_RELEASE_YT",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "create_mkdb_info": True,
                        },
                        out={
                            "MKDB_INFO_RESOURCE": 30,
                        },
                    ),
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="YABS_SERVER_REPORT_BUILD",
                        build_item=YabsServerJobTypes.REPORT_BUILD,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "yabs_server_bundle",
                                        "BS_RELEASE_TAR",
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "yabs_server_cs_package",
                                        "BS_RELEASE_YT",
                                    ),
                                )
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type=YabsServerJobTypes.SETUP,
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "yabs_mkdb_info",
                                        "MKDB_INFO_RESOURCE",
                                    ),
                                ),
                            ),
                        ),
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        ctx={
                            "post_st_comment": True,
                            "send_tg_message": True,
                            "binary_executor_release_type": "none",
                        },
                    )
                ]
                return default_branch_part + branch_part

            @property
            def _release(self):
                default_release_part = super(self.__class__, self)._release
                release_part = [
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="YABS_SERVER_CHECK_RELEASE_AUDIT_STATUS",
                        job_params={
                            "job_type": "CHECK_AUDIT",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="build_task_id",
                                    ),
                                ),
                            ),
                        ),
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="DEPLOY_SAMOGON_SERVANTS",
                        job_params={
                            "job_type": rm_const.JobTypes.DEPLOY,
                            "job_name_parameter": "CS_{}".format(rm_const.ReleaseStatus.stable),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="resource",
                                        resource_name="BS_RELEASE_YT",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type="CHECK_AUDIT",
                            ),
                        ),
                        ctx={
                            "project": "yabscs",
                            "namespace": "5",
                            "auto": False,
                            "samogon_oauth_token_vault_name": "YABS_CS_SAMOGON_TOKEN",
                        },
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="DEPLOY_SAMOGON_SERVANTS",
                        job_params={
                            "job_type": rm_const.JobTypes.DEPLOY,
                            "job_name_parameter": "CS_{}_META".format(rm_const.ReleaseStatus.stable),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="resource",
                                        resource_name="BS_RELEASE_YT",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type="CHECK_AUDIT",
                            ),
                        ),
                        ctx={
                            "project": "yabscs",
                            "namespace": "6",
                            "auto": False,
                            "samogon_oauth_token_vault_name": "YABS_CS_SAMOGON_TOKEN",
                        },
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="DEPLOY_SAMOGON_SERVANTS",
                        job_params={
                            "job_type": rm_const.JobTypes.DEPLOY,
                            "job_name_parameter": "PI_TESTING_CS_{}".format(rm_const.ReleaseStatus.stable),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="resource",
                                        resource_name="BS_RELEASE_YT",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "project": "yabscs",
                            "namespace": "7",
                            "auto": False,
                            "samogon_oauth_token_vault_name": "YABS_CS_SAMOGON_TOKEN",
                        },
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="DEPLOY_YABS_CS",
                        job_params={
                            "job_type": rm_const.JobTypes.DEPLOY,
                            "job_name_parameter": "CS_{}".format(rm_const.ReleaseStatus.testing),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="resource",
                                        resource_name="BS_RELEASE_YT",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "project": "yabscs",
                            "namespaces": ["1", "2"],
                            "samogon_oauth_token_vault_name": "YABS_CS_SAMOGON_TOKEN",
                            "wait_deploy": True,
                            "ignored_server_services": [
                                "preproduction_yabs_frontend_server_bs_dev1_meta",
                                "preproduction_yabs_frontend_server_bs_dev1_stat",
                            ],
                            "whitelist_nanny_labels": {
                                "ctype": "prod",
                            },
                        },
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="DEPLOY_YABS_CS",
                        job_params={
                            "job_type": rm_const.JobTypes.DEPLOY,
                            "job_name_parameter": "CS_{}_META".format(rm_const.ReleaseStatus.testing),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="resource",
                                        resource_name="BS_RELEASE_YT",
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "project": "yabscs",
                            "namespaces": ["3", "4"],
                            "samogon_oauth_token_vault_name": "YABS_CS_SAMOGON_TOKEN",
                            "wait_deploy": True,
                            "ignored_server_services": [
                                "production_yabs_frontend_server_bs_msk_iva_new_runtime_meta",
                                "production_yabs_frontend_server_bs_msk_iva_new_runtime_stat",
                            ],
                            "whitelist_nanny_labels": {
                                "ctype": "prod",
                                "server_mode": "meta",
                            },
                        },
                    ),
                ]

                for service in YDEPLOY_SERVICES:
                    release_mode_to_nanny_params = {
                        rm_const.ReleaseStatus.testing: {
                            "service_ids": service.testing_service_ids,
                            "comment": lambda major_version, minor_version, name=service.name.lower(): "{name} testing release r{major}-{minor}".format(
                                name=name, major=major_version, minor=minor_version
                            )
                        },
                        rm_const.ReleaseStatus.stable: {
                            "service_ids": service.stable_service_ids,
                            "comment": lambda major_version, minor_version, name=service.name.lower(): "{name} stable release r{major}-{minor}".format(
                                name=name, major=major_version, minor=minor_version
                            )
                        }
                    }
                    for release_stage, stage_params in release_mode_to_nanny_params.items():
                        if not stage_params['service_ids']:
                            continue
                        release_part.append(
                            jg_test.JobGraphElementTestBranchCommon(
                                task_name="CREATE_NANNY_SNAPSHOTS",
                                job_params={
                                    "job_type": rm_const.JobTypes.DEPLOY,
                                    "job_name_parameter": "PREPARE_{}_RUNTIME_SNAPHOTS_{}".format(service.name.upper(), release_stage),
                                },
                                job_arrows=(
                                    jg_arrows.ParentsData(
                                        input_key="sandbox_files_dict",
                                        triggers=(
                                            jg_job_triggers.JobTriggerBuild(
                                                job_name_parameter=service.name.upper(),
                                                parent_job_data=(
                                                    jg_job_data.ParentDataResource(
                                                        input_key="{}_release_tar_resource_id".format(service.name.lower()),
                                                        resource_name=service.resource_type,
                                                    ),
                                                ),
                                            ),
                                        ),
                                        transform=service.transform_resource_id
                                    ),
                                    jg_arrows.ParentsData(
                                        input_key="comment",
                                        triggers=(
                                            jg_arrows.ParamsData(
                                                input_key="major_version",
                                                transform=jg_utils.get_major_release_number,
                                            ),
                                            jg_job_triggers.JobTriggerNewTag(
                                                parent_job_data=(
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="minor_version",
                                                        output_key="new_tag_number",
                                                    ),
                                                ),
                                            ),
                                        ),
                                        transform=stage_params["comment"],
                                    ),
                                ),
                                ctx={
                                    "nanny_oauth_token_name": "YABS_RELEASER_NANNY_TOKEN",
                                    "release_type": release_stage,
                                    "dashboard_id": service.dashboard_id,
                                    "filter_type": "service_id",
                                    "service_ids": stage_params["service_ids"],
                                },
                            )
                        )
                        release_part.append(
                            jg_test.JobGraphElementTestBranchCommon(
                                task_name="YABS_SERVER_DEPLOY_NANNY_SNAPSHOTS",
                                job_params={
                                    "job_type": rm_const.JobTypes.DEPLOY,
                                    "job_name_parameter": "{}_RUNTIME_{}".format(service.name.upper(), release_stage),
                                },
                                job_arrows=(
                                    jg_arrows.ParentsData(
                                        input_key="snapshots_to_deploy",
                                        triggers=(
                                            jg_job_triggers.JobTriggerTestBranchCommon(
                                                job_type=rm_const.JobTypes.DEPLOY,
                                                job_name_parameter="PREPARE_{}_RUNTIME_SNAPHOTS_{}".format(service.name.upper(), release_stage),
                                                parent_job_data=(
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="runtime_snapshots",
                                                        output_key="created_snapshots",
                                                        transform=lambda x, params: x or dict(),
                                                    )
                                                )
                                            )
                                        ),
                                        transform=lambda runtime_snapshots: runtime_snapshots,
                                    ),
                                    jg_arrows.ParamsData(
                                        input_key="major_version",
                                        transform=jg_utils.get_major_release_number,
                                    ),
                                    jg_job_triggers.JobTriggerNewTag(
                                        parent_job_data=(
                                            jg_job_data.ParentDataOutput(
                                                input_key="minor_version",
                                                output_key="new_tag_number",
                                            ),
                                        ),
                                    ),
                                ),
                                ctx={
                                    "wait_deploy": False,
                                    "nanny_oauth_token_name": "YABS_RELEASER_NANNY_TOKEN",
                                    "release_type": release_stage,
                                    "dashboard_id": service.dashboard_id,
                                    "recipe_name": service.recipe_id,
                                    "notify": False,
                                },
                            )
                        )

                RELEASE_MODE_TO_NANNY_PARAMS = {
                    rm_const.ReleaseStatus.testing: {
                        "filter_services_by_labels": {"ctype": "prestable"},
                        "wait_deploy": True,
                        "use_negation": False,
                    },
                    rm_const.ReleaseStatus.stable: {
                        "filter_services_by_labels": {"ctype": "prodtest"},
                        "wait_deploy": False,
                        "use_negation": True,  # Remove all mirrors
                    },
                    "mirror": {
                        "filter_services_by_labels": {"ctype": "prodtest"},
                        "wait_deploy": False,
                        "use_negation": False,  # Add all mirrors
                    },
                }
                RECIPE_NAME = "bsfront_universal_by_segment.yaml"
                DASHBOARD_ID = "bsfront_production"
                for release_stage, comment_transform in [
                    (
                        rm_const.ReleaseStatus.stable,
                        lambda major_version, minor_version: "yabs_server stable release r{major}-{minor}".format(
                            major=major_version,
                            minor=minor_version,
                        ),
                    ),
                    (
                        rm_const.ReleaseStatus.testing,
                        lambda major_version, minor_version: "yabs_server testing release r{major}-{minor}".format(
                            major=major_version,
                            minor=minor_version,
                        ),
                    )
                ]:
                    release_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="CREATE_NANNY_SNAPSHOTS",
                            job_params={
                                "job_type": rm_const.JobTypes.DEPLOY,
                                "job_name_parameter": "PREPARE_RUNTIME_SNAPHOTS_{}".format(release_stage),
                            },
                            job_arrows=(
                                jg_arrows.ParentsData(
                                    input_key="sandbox_files_dict",
                                    triggers=(
                                        jg_job_triggers.JobTriggerBuild(
                                            parent_job_data=(
                                                jg_job_data.ParentDataOutput(
                                                    input_key="bs_release_tar_resource_id",
                                                    output_key="bs_release_tar_resource",
                                                    transform=lambda resource_id, params: resource_id,
                                                ),
                                            ),
                                        ),
                                    ),
                                    transform=lambda bs_release_tar_resource_id: {
                                        "yabs-server-bundle.tar.gz": bs_release_tar_resource_id
                                    },
                                ),
                                jg_arrows.ParentsData(
                                    input_key="comment",
                                    triggers=(
                                        jg_arrows.ParamsData(
                                            input_key="major_version",
                                            transform=jg_utils.get_major_release_number,
                                        ),
                                        jg_job_triggers.JobTriggerNewTag(
                                            parent_job_data=(
                                                jg_job_data.ParentDataOutput(
                                                    input_key="minor_version",
                                                    output_key="new_tag_number",
                                                ),
                                            ),
                                        ),
                                    ),
                                    transform=comment_transform,
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type="CHECK_AUDIT",
                                ) if release_stage == rm_const.ReleaseStatus.stable else None,
                            ),
                            ctx={
                                "nanny_oauth_token_name": "YABS_RELEASER_NANNY_TOKEN",
                                "release_type": release_stage,
                                "dashboard_id": DASHBOARD_ID,
                                "filter_type": "labels",
                                "filter_services_by_labels": RELEASE_MODE_TO_NANNY_PARAMS[release_stage]["filter_services_by_labels"],
                                "use_negation": RELEASE_MODE_TO_NANNY_PARAMS[release_stage]["use_negation"],
                            },
                        )
                    )
                    if release_stage == rm_const.ReleaseStatus.stable:
                        release_part.append(
                            jg_test.JobGraphElementTestBranchCommon(
                                task_name="CREATE_NANNY_SNAPSHOTS",
                                job_params={
                                    "job_type": rm_const.JobTypes.DEPLOY,
                                    "job_name_parameter": "PREPARE_RUNTIME_MIRROR_SNAPHOTS_{}".format(release_stage),
                                },
                                job_arrows=(
                                    jg_arrows.ParentsData(
                                        input_key="sandbox_files_dict",
                                        triggers=(
                                            jg_job_triggers.JobTriggerBuild(
                                                parent_job_data=(
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="bs_release_tar_resource_id",
                                                        output_key="bs_release_tar_resource",
                                                        transform=lambda resource_id, params: resource_id,
                                                    ),
                                                ),
                                            ),
                                            jg_job_triggers.JobTriggerBuild(
                                                job_name_parameter="DEBUG",
                                                parent_job_data=(
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="bs_release_tar_debug_resource_id",
                                                        output_key="bs_release_tar_resource",
                                                        transform=lambda resource_id, params: resource_id,
                                                    ),
                                                ),
                                            ),
                                        ),
                                        transform=lambda bs_release_tar_resource_id, bs_release_tar_debug_resource_id: {
                                            "yabs-server-bundle.tar.gz": bs_release_tar_resource_id,
                                            "yabs-server-bundle-debug.tar.gz": bs_release_tar_debug_resource_id,
                                        }
                                    ),
                                    jg_arrows.ParentsData(
                                        input_key="comment",
                                        triggers=(
                                            jg_arrows.ParamsData(
                                                input_key="major_version",
                                                transform=jg_utils.get_major_release_number,
                                            ),
                                            jg_job_triggers.JobTriggerNewTag(
                                                parent_job_data=(
                                                    jg_job_data.ParentDataOutput(
                                                        input_key="minor_version",
                                                        output_key="new_tag_number",
                                                    ),
                                                ),
                                            ),
                                        ),
                                        transform=comment_transform,
                                    ),
                                    jg_job_triggers.JobTriggerTestBranchCommon(
                                        job_type="CHECK_AUDIT",
                                    ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                ),
                                ctx={
                                    "nanny_oauth_token_name": "YABS_RELEASER_NANNY_TOKEN",
                                    "release_type": release_stage,
                                    "dashboard_id": DASHBOARD_ID,
                                    "filter_type": "labels",
                                    "filter_services_by_labels": RELEASE_MODE_TO_NANNY_PARAMS["mirror"]["filter_services_by_labels"],
                                    "use_negation": RELEASE_MODE_TO_NANNY_PARAMS["mirror"]["use_negation"],
                                },
                            )
                        )

                    release_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="YABS_SERVER_DEPLOY_NANNY_SNAPSHOTS",
                            job_params={
                                "job_type": rm_const.JobTypes.DEPLOY,
                                "job_name_parameter": "RUNTIME_{}".format(release_stage),
                            },
                            job_arrows=(
                                jg_arrows.ParentsData(
                                    input_key="snapshots_to_deploy",
                                    triggers=(
                                        jg_job_triggers.JobTriggerTestBranchCommon(
                                            job_type=rm_const.JobTypes.DEPLOY,
                                            job_name_parameter="PREPARE_BUMP_SNAPHOTS_{}".format(rm_const.ReleaseStatus.testing),
                                            parent_job_data=(
                                                jg_job_data.ParentDataOutput(
                                                    input_key="bump_snapshots",
                                                    output_key="created_snapshots",
                                                    transform=lambda x, params: x or dict(),
                                                )
                                            ),
                                        ),
                                        jg_job_triggers.JobTriggerTestBranchCommon(
                                            job_type=rm_const.JobTypes.DEPLOY,
                                            job_name_parameter="PREPARE_RUNTIME_SNAPHOTS_{}".format(rm_const.ReleaseStatus.testing),
                                            parent_job_data=(
                                                jg_job_data.ParentDataOutput(
                                                    input_key="runtime_snapshots",
                                                    output_key="created_snapshots",
                                                    transform=lambda x, params: x or dict(),
                                                )
                                            )
                                        )
                                    ),
                                    transform=lambda bump_snapshots, runtime_snapshots: jg_utils.union_dicts(
                                        bump_snapshots,
                                        runtime_snapshots,
                                    ),
                                ) if release_stage == rm_const.ReleaseStatus.testing else None,
                                jg_arrows.ParentsData(
                                    input_key="snapshots_to_deploy",
                                    triggers=(
                                        jg_job_triggers.JobTriggerTestBranchCommon(
                                            job_type=rm_const.JobTypes.DEPLOY,
                                            job_name_parameter="PREPARE_RUNTIME_SNAPHOTS_{}".format(rm_const.ReleaseStatus.stable),
                                            parent_job_data=(
                                                jg_job_data.ParentDataOutput(
                                                    input_key="runtime_snapshots",
                                                    output_key="created_snapshots",
                                                    transform=lambda x, params: x or dict(),
                                                )
                                            )
                                        ),
                                        jg_job_triggers.JobTriggerTestBranchCommon(
                                            job_type=rm_const.JobTypes.DEPLOY,
                                            job_name_parameter="PREPARE_RUNTIME_MIRROR_SNAPHOTS_{}".format(rm_const.ReleaseStatus.stable),
                                            parent_job_data=(
                                                jg_job_data.ParentDataOutput(
                                                    input_key="runtime_mirror_snapshots",
                                                    output_key="created_snapshots",
                                                    transform=lambda x, params: x or dict(),
                                                )
                                            )
                                        ),
                                    ),
                                    transform=lambda runtime_snapshots, runtime_mirror_snapshots: jg_utils.union_dicts(
                                        runtime_snapshots,
                                        runtime_mirror_snapshots,
                                    )
                                ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                jg_arrows.ParamsData(
                                    input_key="major_version",
                                    transform=jg_utils.get_major_release_number,
                                ),
                                jg_job_triggers.JobTriggerNewTag(
                                    parent_job_data=(
                                        jg_job_data.ParentDataOutput(
                                            input_key="minor_version",
                                            output_key="new_tag_number",
                                        ),
                                    ),
                                ),
                            ),
                            ctx={
                                "wait_deploy": RELEASE_MODE_TO_NANNY_PARAMS[release_stage]["wait_deploy"],
                                "nanny_oauth_token_name": "YABS_RELEASER_NANNY_TOKEN",
                                "release_type": release_stage,
                                "dashboard_id": DASHBOARD_ID,
                                "recipe_name": RECIPE_NAME,
                                "notify": True,
                                "notify_stages": [
                                    "deployment_precondition_failed",
                                    "deployment_rejected",
                                    "deployment_scheduled",
                                    "deployment_stalled",
                                    "deployment_succeeded",
                                    "deployment_timed_out",
                                ] if release_stage == rm_const.ReleaseStatus.stable else [
                                    "deployment_precondition_failed",
                                    "deployment_rejected",
                                    "deployment_scheduled",
                                    "deployment_stalled",
                                    "deployment_succeeded",
                                    "deployment_timed_out",
                                    "deployment_started",
                                ],
                            },
                        )
                    )

                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "yabs_server_bundle",
                                            "BS_RELEASE_TAR",
                                        ),
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "yabs_server_cs_package",
                                            "BS_RELEASE_YT",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "yabs_udf",
                                            "YABS_UDF",
                                        ),
                                    ),
                                    job_name_parameter="YABS_UDF",
                                ),
                            ) + tuple(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            service.resource_name,
                                            service.resource_type
                                        )
                                    ),
                                    job_name_parameter=service.name.upper()
                                )
                                for service in YDEPLOY_SERVICES
                            ) + (
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=YabsServerJobTypes.SETUP,
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            "component_resources",
                                            "yabs_mkdb_info",
                                            "MKDB_INFO_RESOURCE",
                                        ),
                                    ),
                                ),
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type="CHECK_AUDIT",
                                ) if release_stage == rm_const.ReleaseStatus.stable else None,
                            )
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=YabsServerJobTypes.MOVE_STARTREK_ISSUE,
                                    job_name_parameter=rm_const.ReleaseStatus.testing,
                                ) if release_stage == rm_const.ReleaseStatus.testing else None,
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=YabsServerJobTypes.NOTIFY,
                                    job_name_parameter="READY_FOR_STABLE",
                                ) if release_stage == rm_const.ReleaseStatus.testing else None,
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter="CS_{}".format(rm_const.ReleaseStatus.testing),
                                ) if release_stage == rm_const.ReleaseStatus.testing else None,
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter="CS_{}_META".format(rm_const.ReleaseStatus.testing),
                                ) if release_stage == rm_const.ReleaseStatus.testing else None
                            ) + tuple(
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter="{}_RUNTIME_{}".format(service.name.upper(), release_stage),
                                ) if release_stage == rm_const.ReleaseStatus.testing else None
                                for service in YDEPLOY_SERVICES if service.testing_service_ids
                            ) + (
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=YabsServerJobTypes.NOTIFY,
                                    job_name_parameter="RELEASED_TO_STABLE",
                                ) if release_stage == rm_const.ReleaseStatus.stable else None,
                            ) + tuple(
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter="{}_RUNTIME_{}".format(service.name.upper(), release_stage),
                                ) if release_stage == rm_const.ReleaseStatus.stable else None
                                for service in YDEPLOY_SERVICES if service.stable_service_ids
                            ) + (
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter="RUNTIME_{}".format(release_stage),
                                ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter="CS_{}".format(rm_const.ReleaseStatus.stable),
                                ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter="CS_{}_META".format(rm_const.ReleaseStatus.stable),
                                ) if release_stage == rm_const.ReleaseStatus.stable else None,
                                jg_job_triggers.JobTriggerTestBranchCommon(
                                    job_type=rm_const.JobTypes.DEPLOY,
                                    job_name_parameter="PI_TESTING_CS_{}".format(rm_const.ReleaseStatus.stable),
                                ) if release_stage == rm_const.ReleaseStatus.stable else None,
                            )
                        )
                    )
                release_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="MOVE_STARTREK_TICKET",
                        job_params={
                            "job_type": YabsServerJobTypes.MOVE_STARTREK_ISSUE,
                            "job_name_parameter": rm_const.ReleaseStatus.testing,
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="release_num",
                                transform=jg_utils.get_major_release_number,
                            ),
                        ),
                        ctx={
                            "transition": "passed",
                        },
                    )
                )
                release_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="CREATE_NANNY_SNAPSHOTS",
                        job_params={
                            "job_type": rm_const.JobTypes.DEPLOY,
                            "job_name_parameter": "PREPARE_BUMP_SNAPHOTS_{}".format(rm_const.ReleaseStatus.testing),
                        },
                        job_arrows=(
                            jg_arrows.ParentsData(
                                input_key="static_files_dict",
                                triggers=(
                                    jg_arrows.ParamsData(
                                        input_key="major_version",
                                        transform=jg_utils.get_major_release_number,
                                    ),
                                    jg_job_triggers.JobTriggerNewTag(
                                        parent_job_data=(
                                            jg_job_data.ParentDataOutput(
                                                input_key="minor_version",
                                                output_key="new_tag_number",
                                            ),
                                        ),
                                    ),
                                ),
                                transform=lambda major_version, minor_version: {
                                    "bump_file.txt": "Bumped automatically for r{major}-{minor}".format(
                                        major=major_version,
                                        minor=minor_version,
                                    ),
                                }
                            ),
                        ),
                        ctx={
                            "nanny_oauth_token_name": "YABS_RELEASER_NANNY_TOKEN",
                            "release_type": rm_const.ReleaseStatus.testing,
                            "dashboard_id": DASHBOARD_ID,
                            "filter_type": "service_id",
                            "service_ids": [
                                "stable_yabs_frontend_server_bs_msk_iva_meta",
                                "stable_yabs_frontend_server_bs_msk_iva_stat",
                                "stable_yabs_frontend_server_yabs_sas_meta",
                                "stable_yabs_frontend_server_yabs_sas_stat",
                                "stable_yabs_frontend_server_metadsp_vla_meta",
                                "stable_yabs_frontend_server_metadsp_vla_stat"
                            ],
                        },
                    )
                )
                release_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="YABS_SERVER_ANALYZE_PRODUCTION_WRAPPER",
                        job_params={
                            "job_type": YabsServerJobTypes.ANALYZE_PREPRODUCTION,
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type=rm_const.JobTypes.DEPLOY,
                                job_name_parameter="RUNTIME_{}".format(rm_const.ReleaseStatus.testing),
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="start_timestamp",
                                        output_key="deploy_end_time",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                        ),
                        ctx={
                            "initial_analysis_duration": timedelta(hours=4).total_seconds(),
                            "max_analysis_duration": timedelta(hours=9).total_seconds(),
                            "time_to_wait": timedelta(minutes=30).total_seconds(),
                            "report_resource_ttl": 550,  # 1.5 year
                            "validate_timeouts": False,
                            "bs_stable": "stable_yabs_frontend_server_bs_msk_iva",
                            "bs_testing": "prestable_yabs_frontend_server_bs_msk_iva",
                            "yabs_stable": "stable_yabs_frontend_server_yabs_sas",
                            "yabs_testing": "prestable_yabs_frontend_server_yabs_sas",
                        }
                    )
                )
                release_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="YABS_SERVER_READY_FOR_STABLE_NOTIFY",
                        job_params={
                            "job_type": YabsServerJobTypes.NOTIFY,
                            "job_name_parameter": "READY_FOR_STABLE",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_type=YabsServerJobTypes.ANALYZE_PREPRODUCTION,
                            ),
                        ),
                        ctx={
                            "chat_name": CHAT_NAME,
                            "release_confirm_people": RELEASE_CONFIRM_PEOPLE,
                        },
                    )
                )
                release_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="YABS_SERVER_STABLE_RELEASE_NOTIFY",
                        job_params={
                            "job_type": YabsServerJobTypes.NOTIFY,
                            "job_name_parameter": "RELEASED_TO_STABLE",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerChangelogFinal(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="changelog",
                                        output_key="changelog",
                                        transform=lambda x, params: str(x),
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter=rm_const.ReleaseStatus.stable,
                            ),
                        ),
                        ctx={
                            "recipients": ["yabs-dev"],
                            "filter_finalized_issues_by_statuses": ["rc", "awaitingDeployment", "deployed"],
                            "filter_wip_issues_by_statuses": ["inProgress", "needInfo"],
                        },
                    )
                )
                return default_release_part + release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            """TestEnv Job Patch"""

            @property
            def ignore_match(self):
                additional_ignored = [
                    rm_const.JobTypes.rm_job_name(YabsServerJobTypes.NOTIFY_RELEASE_TICKET_CHANGED, self.name),
                    rm_const.JobTypes.rm_job_name(YabsServerJobTypes.REPORT_B2B_DIFFS, self.name),
                    rm_const.JobTypes.rm_job_name(YabsServerJobTypes.REPORT_FT_RESULTS, self.name),
                    rm_const.JobTypes.rm_job_name(YabsServerJobTypes.REPORT_PERFORMANCE_DELTA, self.name),
                ]
                return super(self.__class__, self).ignore_match + additional_ignored

    class Releases(configs.ReferenceBranchedConfig.Releases):
        resources_info = [
            configs.ReleasedResourceInfo(
                name="yabs_server_bundle",
                resource_type="BS_RELEASE_TAR",
            ),
            configs.ReleasedResourceInfo(
                name="yabs_server_cs_package",
                resource_type="BS_RELEASE_YT",
            ),
            configs.ReleasedResourceInfo(
                name="yabs_mkdb_info",
                resource_type="MKDB_INFO_RESOURCE",
            ),
            configs.ReleasedResourceInfo(
                name="yabs_udf",
                resource_type="YABS_UDF",
            ),
        ] + [
            configs.ReleasedResourceInfo(
                name=service.resource_name,
                resource_type=service.resource_type
            )
            for service in YDEPLOY_SERVICES
        ]
        allow_old_releases = True

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["yabs-server_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = configs.Responsible(abc=configs.Abc(component_id=179, schedule_slug='yabs_frontend_duty_first'), login="igorock")
            queue = "BSRELEASE"
            dev_queue = "YABS"
            summary_template = u"Релиз server r{}"
            components = ["release"]
            important_changes_limit = 1000
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "igorock",
            ]
            ticket_type = "Release-server"
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7
            commit_importance_threshold = 4
            abc_followers = configs.Abc(role_id=8, component_id=179)
            ticket_filter = rm_const.TicketFilter(
                queues={"BSSERVER", "BSDEV", "BIGB", "NANPU", "ADVMACHINE", "RMP", "DSP"},
                filter_type=rm_const.StartrekTicketFilterTypes.include
            )

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "BannernajaKrutilka/server/Relizy/changelog/"
        ya_make_targets = [
            "arcadia/yabs/server/daemons/bs_static",
            "arcadia/yabs/server/config",
            "arcadia/yabs/server/tools/mkdb",
            "arcadia/yabs/server/cs/tools/cs",
            "arcadia/yabs/server/cs/tools/dbtool",
            "arcadia/yabs/server/cs/package",
            "arcadia/yabs/server/cs/package/plugin",
        ]
        svn_paths_filter = configs.ChangelogPathsFilter(rm_const.PermissionType.CHANGED, ["arcadia/yabs"], 5)

        banned_authors = ["robot-upd-res"]

        def should_skip_revision(self, r_info):
            return super(self.__class__, self).should_skip_revision(r_info) or self._should_skip_revision(r_info)

        def _should_skip_revision(self, r_info):
            def get_normalized_path(original_path):
                return "".join(original_path.partition("arcadia/")[1:])

            def is_meta_stat_protocol_test_autoupdate_commit(r_info):
                if r_info["author"] != "robot-upd-res":
                    return False

                for _, path in r_info["paths"]:
                    path = get_normalized_path(path)
                    if path.startswith("arcadia/yabs/server/libs/server/protocol/tests"):
                        return True
                return False

            return is_meta_stat_protocol_test_autoupdate_commit(r_info)

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        branch_name = "yabs/server"
        branch_prefix = "r"
        branch_folder_template = r"{branch_prefix}{branch_num}"
        tag_prefix = "r"
        tag_folder_template = "{tag_prefix}{branch_num}-{tag_num}"
        allow_autobranches = True
        max_active_branches = 3
        use_arc = True
