# -*- coding: utf-8 -*-

from sandbox.projects.common.constants import constants as common_const
from sandbox.projects.release_machine.components import configs
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri

CLUSTERS = ("sas", "man", "vla", "msk")


class YpDnsCfg(configs.ReferenceBranchedConfig):
    name = "yp_dns"
    responsible = "elshiko"
    display_name = "YP DNS"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "SEARCH-RELEASERS"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _trunk_part(self):
                trunk_part = super(self.__class__, self)._trunk_part

                for cluster in CLUSTERS:
                    trunk_part.append(
                        jg_test.JobGraphElementRunIceFlame(
                            config_file="yp_dns.conf",
                            job_params={
                                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                                "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                                "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                    time_interval_list=[(13, 19)],
                                    weekday_list=[0, 1, 2, 3, 4],
                                    seconds_from_last_run_commit=1.5 * 3600,
                                    once_a_day=True,
                                ),
                                "job_name_parameter": "{}_RELEASE".format(cluster.upper()),
                            },
                            ctx={
                                "release_item_name": "yp_dns",
                                "service_name": "{}_yp_dns".format(cluster),
                                "fail_if_service_version_cannot_be_determined": True,
                                "check_production_version": True,
                                "iceflame_command_run_options_json": {
                                    "remote_collector_options": {
                                        "analyze_inplace": True,
                                        "build_flamegraph": True,
                                    },
                                    "apply_config_patch": "change_nanny_name",
                                },
                                "env_vars": "NANNY_SERVICE_NAME='{}_yp_dns'".format(cluster),
                            },
                        ),
                    )
                    trunk_part.append(
                        jg_test.JobGraphElementRunIceFlame(
                            config_file="yp_dns.conf",
                            job_params={
                                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                                "frequency": (jg_utils.TestFrequency.DEFINED_BY_CODE, None),
                                "next_revision_custom_schedule": jg_utils.CustomSchedule(
                                    time_interval_list=[(11, 12)],
                                    weekday_list=[0],
                                    seconds_from_last_run_start_time=7 * 24 * 3600,  # every week
                                    last_run_finished=True,
                                    once_a_day=True,
                                ),
                                "job_name_parameter": "{}_WEEKLY".format(cluster.upper()),
                            },
                            ctx={
                                "release_item_name": "yp_dns",
                                "service_name": "msk_yp_dns",
                                "iceflame_command_run_options_json": {
                                    "remote_collector_options": {
                                        "analyze_inplace": True,
                                        "build_flamegraph": True,
                                    },
                                    "apply_config_patch": "change_nanny_name",
                                },
                                "env_vars": "NANNY_SERVICE_NAME='{}_yp_dns'".format(cluster),
                            },
                        ),
                    )

                return trunk_part

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="YA_MAKE_RELEASE_TO_NANNY",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 1024 ** 2,
                                }
                            },
                            "job_name_parameter": "EXECUTABLE_RELEASE",
                        },
                        ctx={
                            "targets": "infra/yp_dns/deploy/yp_dns",
                            "arts": "infra/yp_dns/deploy/yp_dns/yp_dns",
                            "arts_source": "",
                            "result_rt": "YP_DNS",
                            "result_rd": "YP DNS executable",
                            "result_single_file": True,
                            common_const.DEFINITION_FLAGS_KEY: "",
                            common_const.USE_AAPI_FUSE: True,
                            common_const.ALLOW_AAPI_FALLBACK: True,
                            common_const.CHECKOUT_MODE: common_const.CHECKOUT_MODE_AUTO,
                            common_const.CHECKOUT: True,
                            common_const.BUILD_SYSTEM_KEY: common_const.YMAKE_BUILD_SYSTEM,
                            common_const.CLEAR_BUILD_KEY: False,
                            common_const.CHECK_RETURN_CODE: True,
                            common_const.TESTS_REQUESTED: False,
                        },
                        out={"YP_DNS": 30},
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="YA_MAKE_RELEASE_TO_NANNY",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 50 * 1024,
                                }
                            },
                            "job_name_parameter": "pdns_config",
                        },
                        ctx={
                            "targets": "infra/yp_dns/deploy/pdns_config",
                            "arts": "infra/yp_dns/deploy/pdns_config/pdns.conf",
                            "arts_source": "",
                            "result_rt": "YP_DNS_PDNS_CONF",
                            "result_rd": "pdns.conf for YP DNS",
                            "result_single_file": True,
                            common_const.DEFINITION_FLAGS_KEY: "",
                            common_const.USE_AAPI_FUSE: True,
                            common_const.ALLOW_AAPI_FALLBACK: True,
                            common_const.CHECKOUT_MODE: common_const.CHECKOUT_MODE_AUTO,
                            common_const.CHECKOUT: True,
                            common_const.BUILD_SYSTEM_KEY: common_const.YMAKE_BUILD_SYSTEM,
                            common_const.CLEAR_BUILD_KEY: False,
                            common_const.CHECK_RETURN_CODE: True,
                            common_const.TESTS_REQUESTED: False,
                        },
                        out={"YP_DNS_PDNS_CONF": 30},
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="YA_MAKE_RELEASE_TO_NANNY",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 50 * 1024,
                                }
                            },
                            "job_name_parameter": "instancectl_conf",
                        },
                        ctx={
                            "targets": "infra/yp_dns/deploy/instancectl_conf",
                            "arts": "infra/yp_dns/deploy/instancectl_conf/instancectl.conf",
                            "arts_source": "",
                            "result_rt": "YP_DNS_INSTANCECTL_CONF",
                            "result_rd": "instancectl.conf for YP DNS",
                            "result_single_file": True,
                            common_const.DEFINITION_FLAGS_KEY: "",
                            common_const.USE_AAPI_FUSE: True,
                            common_const.ALLOW_AAPI_FALLBACK: True,
                            common_const.CHECKOUT_MODE: common_const.CHECKOUT_MODE_AUTO,
                            common_const.CHECKOUT: True,
                            common_const.BUILD_SYSTEM_KEY: common_const.YMAKE_BUILD_SYSTEM,
                            common_const.CLEAR_BUILD_KEY: False,
                            common_const.CHECK_RETURN_CODE: True,
                            common_const.TESTS_REQUESTED: False,
                        },
                        out={"YP_DNS_INSTANCECTL_CONF": 30},
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="YA_MAKE_RELEASE_TO_NANNY",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 50 * 1024,
                                }
                            },
                            "job_name_parameter": "ua_config",
                        },
                        ctx={
                            "targets": "infra/yp_dns/deploy/unified_agent",
                            "arts": "infra/yp_dns/deploy/unified_agent/unified_agent.yml.tmpl",
                            "arts_source": "",
                            "result_rt": "YP_DNS_UNIFIED_AGENT_CONFIG",
                            "result_rd": "Unified Agent config for YP DNS",
                            "result_single_file": True,
                            common_const.DEFINITION_FLAGS_KEY: "",
                            common_const.USE_AAPI_FUSE: True,
                            common_const.ALLOW_AAPI_FALLBACK: True,
                            common_const.CHECKOUT_MODE: common_const.CHECKOUT_MODE_AUTO,
                            common_const.CHECKOUT: True,
                            common_const.BUILD_SYSTEM_KEY: common_const.YMAKE_BUILD_SYSTEM,
                            common_const.CLEAR_BUILD_KEY: False,
                            common_const.CHECK_RETURN_CODE: True,
                            common_const.TESTS_REQUESTED: False,
                        },
                        out={"YP_DNS_UNIFIED_AGENT_CONFIG": 30},
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="YA_MAKE",
                        job_params={
                            "job_name_parameter": "LAUNCH_ARCTESTS",
                            "apiargs": {
                                "requirements": {
                                    "ram": 50 * (1024 ** 3),
                                }
                            },
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(
                                parent_job_data=[
                                    jg_job_data.ParentDataOutput(
                                        input_key=common_const.ARCADIA_URL_KEY,
                                        output_key=common_const.ARCADIA_URL_KEY,
                                    ),
                                ]
                            ),
                        ),
                        ctx={
                            common_const.BUILD_SYSTEM_KEY: common_const.DISTBUILD_BUILD_SYSTEM,
                            common_const.BUILD_TYPE_KEY: common_const.RELEASE_BUILD_TYPE,
                            common_const.USE_AAPI_FUSE: True,
                            "targets": ";".join([
                                "infra/yp_dns/deploy/tests",
                            ]),
                            "test": True,
                            "report_tests_only": True,
                            "disable_test_timeout": False,
                            "cache_test_results": False,
                            "tests_retries": 2,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="YP_DNS_DIFF_TEST",
                        job_params={
                            "job_name_parameter": "DIFF_TEST_PRODUCTION_NAMESERVER",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="checked_yp_dns_executable",
                                        resource_name="YP_DNS",
                                    ),
                                ),
                                job_name_parameter="EXECUTABLE_RELEASE",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="checked_pdns_config",
                                        resource_name="YP_DNS_PDNS_CONF",
                                    ),
                                ),
                                job_name_parameter="pdns_config",
                            )
                        ),
                        ctx={
                            "sample_type": "nameserver_address",
                            "sample_dns_address": "ns1.yp-dns.yandex.net:53",
                            "queries_percentage": 0.2,
                            "number_of_queries": 50000,
                            "yp_token_owner": "robot-yp-dns-test",
                            "yt_token_owner": "robot-yp-dns-test",
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="YP_DNS_DIFF_TEST",
                        job_params={
                            "job_name_parameter": "DIFF_TEST_STABLE",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="checked_yp_dns_executable",
                                        resource_name="YP_DNS",
                                    ),
                                ),
                                job_name_parameter="EXECUTABLE_RELEASE",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="checked_pdns_config",
                                        resource_name="YP_DNS_PDNS_CONF",
                                    ),
                                ),
                                job_name_parameter="pdns_config",
                            )
                        ),
                        ctx={
                            "sample_type": "local_nameserver",
                            "queries_percentage": 0.2,
                            "number_of_queries": 50000,
                            "yp_token_owner": "robot-yp-dns-test",
                            "yt_token_owner": "robot-yp-dns-test",
                        }
                    )
                )
                umbrella_parents = (
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="EXECUTABLE_RELEASE",
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="pdns_config",
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="INSTANCECTL_CONF",
                    ),
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter="ua_config",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="LAUNCH_ARCTESTS",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="DIFF_TEST_PRODUCTION_NAMESERVER",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_name_parameter="DIFF_TEST_STABLE",
                    ),
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranched(
                        job_params={
                            "job_name_parameter": "UMBRELLA_ACCEPT",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=umbrella_parents,
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                release_part.append(
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "yp_dns",
                                        "YP_DNS",
                                    )
                                ),
                                job_name_parameter="EXECUTABLE_RELEASE",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "pdns_conf",
                                        "YP_DNS_PDNS_CONF",
                                    )
                                ),
                                job_name_parameter="pdns_config",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "instancectl_conf",
                                        "YP_DNS_INSTANCECTL_CONF",
                                    )
                                ),
                                job_name_parameter="INSTANCECTL_CONF",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "unified_agent_config",
                                        "YP_DNS_UNIFIED_AGENT_CONFIG",
                                    )
                                ),
                                job_name_parameter="UA_CONFIG",
                            ),
                        )
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                    )
                )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_LAUNCH_ARCTESTS_FOR_YP_DNS": rm_const.TestFrequencies.EACH_REV_TEST,
                    "_YP_DNS_UMBRELLA_ACCEPT": rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            deploy_infos = [
                ri.NannyDeployInfo([
                    ri.DeployService("sas_yp_dns"),
                    ri.DeployService("man_yp_dns"),
                    ri.DeployService("vla_yp_dns"),
                    ri.DeployService("msk_yp_dns"),
                ]),
            ]
            return [
                ri.ReleasableItem(
                    name="yp_dns",
                    data=ri.SandboxResourceData("YP_DNS"),
                    deploy_infos=deploy_infos,
                ),
                ri.ReleasableItem(
                    name="pdns_conf",
                    data=ri.SandboxResourceData("YP_DNS_PDNS_CONF"),
                ),
                ri.ReleasableItem(
                    name="instancectl_conf",
                    data=ri.SandboxResourceData("YP_DNS_INSTANCECTL_CONF"),
                ),
                ri.ReleasableItem(
                    name="unified_agent_config",
                    data=ri.SandboxResourceData("YP_DNS_UNIFIED_AGENT_CONFIG"),
                ),
            ]

        release_followers_permanent = [
            "elshiko",
        ]

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "ypdns/releases/"
        dirs = [
            "arcadia/infra/yp_dns",
            "arcadia/yp/cpp/yp",
            "arcadia/infra/libs/yp_replica",
        ]
        review_groups = [
            "yp-dns",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = []

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = []

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = configs.Responsible(abc=configs.Abc(component_id=3915, schedule_slug="yp-dns-duty"), login="elshiko")
            use_task_author_as_assignee = True
            queue = "YPDNSREL"
            dev_queue = "DISCOVERY"
            summary_template = u"Приемка YP DNS {}"
            add_commiters_as_followers = True
            deadline = 7
