# -*- coding: utf-8 -*-
from __future__ import unicode_literals


import re
import json
import logging
import datetime

import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
from sandbox.sdk2.vcs.svn import Arcadia


class ABTInfo(rm_comp.BranchedFromBranch, rm_comp.mixin.Changelogged, rm_comp.mixin.Startreked):

    retrieved_source_branch_url = None

    @staticmethod
    def get_source_branch_override():
        override_path = "arcadia:/arc/trunk/arcadia/quality/ab_testing/override_source_branch"

        override_text = rm_svn.SvnHelper.read_file(override_path)

        if not override_text:
            logging.info("No override text")
            return None

        override_data = json.loads(override_text)
        if not override_data:
            logging.info("No override json data")
            return None

        max_relevant_date = datetime.datetime.strptime(override_data["max_relevant_date"], "%Y-%m-%d").date()
        if max_relevant_date < datetime.datetime.now().date():
            logging.info("Override date is not relevant: %s", override_text)
            return None

        source_branch_number = int(override_data["source_branch_number"])
        logging.info("Got relevant override data: %s", override_data)

        return int(source_branch_number)

    @property
    def source_branch_url(self):
        if not self.retrieved_source_branch_url:
            logging.info("Looking for user_sessions branch to fork abt branch from..")
            user_sessions_branches_dir = "arcadia:/arc/branches/user_sessions"

            if self.last_branch_num != 0:
                # Get prev created abt branch
                last_abt = self.last_branch_path
                logging.info("last created abt release branch: %s", last_abt)

                # Get parent branch of the last created abt
                last_abt_parent = rm_svn.SvnHelper.get_parent_branch(last_abt)

                # Subversion prints folders with a trailing slash in "svn list", let's have it too
                if last_abt_parent[-1] != "/":
                    last_abt_parent += "/"
                logging.info("parent of last created abt: %s", last_abt_parent)

                # The first actual user_sessions release with RM is numbered '8',
                # let's target it by default
                prev_user_sessions_num = 7
                match = re.search(r"branches/user_sessions/stable-(\d+)/", last_abt_parent)
                if match:
                    prev_user_sessions_num = int(match.group(1))
                logging.info("using %d as prev user_sessions branch num", prev_user_sessions_num)
                next_user_sessions_num = prev_user_sessions_num + 1
            else:
                next_user_sessions_num = 17

            # Check if source branch for the next release is overridden in a special place in Subversion
            override_next_num = ABTInfo.get_source_branch_override()
            if override_next_num is not None:
                logging.info("overriding next source branch num with %d", override_next_num)
                next_user_sessions_num = override_next_num

            # Check the user_sessions branch exists
            user_sessions_branch_names = rm_svn.SvnHelper.list_files_with_pattern(
                user_sessions_branches_dir,
                "^stable-{}/$".format(next_user_sessions_num)
            )
            if len(user_sessions_branch_names) != 1:
                logging.error("detected %d user_sessions branches for num %d",
                              len(user_sessions_branch_names),
                              next_user_sessions_num)
                raise Exception("unable to find next source branch")

            self.retrieved_source_branch_url = user_sessions_branches_dir + "/" + user_sessions_branch_names[0]
            logging.info("source branch url is found: %s", self.retrieved_source_branch_url)

        return self.retrieved_source_branch_url

    def get_last_deploy(self, token=None, only_level=None):
        if only_level is not None and only_level != rm_const.ReleaseStatus.stable:
            return None
        return [rm_core.DeployedResource.from_released(i) for i in self.get_last_release()]

    def _get_branch_logs(self, branch_path):
        return Arcadia.log(branch_path, self.first_rev, 'HEAD')

    def st_description(self, release_num=None):
        # TODO:
        # * prev abt ticket
        # * prev user_sessions ticket
        # * relevant sandbox resources

        abt_branch_path = self.last_branch_path
        abt_branch_arcanum = rm_const.Urls.make_arcanum_url(abt_branch_path)

        user_sessions_branch_path = rm_svn.SvnHelper.get_parent_branch(abt_branch_path)
        user_sessions_branch_arcanum = rm_const.Urls.make_arcanum_url(user_sessions_branch_path)

        descr = "Stable abt branch: {}\n".format(abt_branch_arcanum)
        descr += "Parent user_sessions branch: {}\n".format(user_sessions_branch_arcanum)

        descr += super(ABTInfo, self).st_description(release_num)
        return descr
