# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import datetime
import logging
import os
import itertools

import sandbox.sdk2 as sdk2
from sandbox.projects import resource_types
import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.common.reactor as reactor
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
import sandbox.projects.release_machine.security as rm_sec
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import decorators
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common.search import findurl
from sandbox.projects.common.differ import printers
from sandbox.projects.common.differ import json_differ
from sandbox.projects.release_machine.client import RMClient


class BegemotInfo(rm_comp.ReferenceComponentMetricsed):
    # we need to correctly exclude holidays when counting to reduce this number below 7
    # otherwise we're notified each weekend which is useless.
    release_duty_wiki = 'https://wiki.yandex-team.ru/Begemot/release/duty/'
    reactor_namespace = '/release-machine/search/begemot'
    _begemot_responsibles = [
        "gluk47",
        "ageraab",
        "dmitryno",
    ]

    def _responses_gen(self, queries, reference_host, checked_host, metrics_cgis):
        """
        * generator of tuples (response_from_reference_host, response_from_checked_host, title)
        """
        json_dump_magic_cgi = "json_dump=search.app_host.sources.(_.name eq 'WIZARD')&dbgwzr=2"
        for query in queries:
            logging.debug('Printing query %s', query.text)
            query.text = query.text.encode('utf-8')
            result = []
            for host in [reference_host, checked_host]:
                qu = 'https://{}/search/?lr={}&text={}&{}&{}'.format(
                    host,
                    query.region_id,
                    query.text,
                    metrics_cgis,
                    json_dump_magic_cgi
                )
                logging.debug('query %s', qu)
                resp = requests_wrapper.get_r(qu)
                logging.debug('response %s', resp)
                result.append(resp.text)
            result.append('"<b>{}</b>". Metrics diff: {}'.format(query.text, query.metrics_diff))
            yield tuple(result)

    def _get_diff_resources(self, task, metrics_json_response):
        diff_resources = []
        max_queries_per_basket = 7
        token = sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME)
        try:
            for launch in metrics_json_response['launches']:
                regional_type = launch['regionalType']
                evaluation_type = launch['evaluationType']
                for query_group in launch['diffQueryGroups']:
                    reference_host = query_group['baselineServer']['host']
                    checked_host = query_group['server']['host']
                    reference_serpset_id = query_group['baselineSerpsetId']
                    checked_serpset_id = query_group['serpsetId']
                    queries_group_name = ' '.join((query_group['name'], query_group['filterName']))

                    logging.debug('Getting queries with high diff...')

                    queries = []
                    for positive in (False, True):
                        queries.extend(itertools.islice(findurl.get_top_queries_from_serpsets_diff(
                            reference_serpset_id,
                            checked_serpset_id,
                            regional_type,
                            evaluation_type,
                            token,
                            metric_name='judged-pfound-5',
                            positive=positive,
                        ), max_queries_per_basket))

                    if not queries:
                        logging.debug('No suitable queries found in %s' % queries_group_name)
                        continue

                    queries = [q.set_diff_url(
                        regional_type,
                        evaluation_type,
                        reference_serpset_id,
                        checked_serpset_id,
                        metric_name='judged-pfound-5',
                    ) for q in queries]

                    logging.debug('Comparing...')

                    output_dir = 'diff_{}'.format(queries_group_name)
                    os.mkdir(output_dir)
                    diff_resource = resource_types.METRICS_LAUNCH_DIFF(task, queries_group_name, output_dir, arch="any")

                    printer = printers.PrinterToHtml(
                        output_dir,
                        pair_head_template="diff #{obj_index}",
                        max_file_size=100 * 1024 * 1024
                    )
                    differ = json_differ.JsonDiffer(printer)

                    metrics_cgis = findurl.get_cgis_from_metrics(reference_serpset_id, token)

                    differ.compare_pairs(self._responses_gen(queries, reference_host, checked_host, metrics_cgis))
                    task.mark_resource_ready(diff_resource.id)
                    diff_resource.main_diff_html = next(
                        name for name in os.listdir(output_dir) if name.endswith('.html')
                    )
                    diff_resources.append(diff_resource)
        except Exception as e:
            eh.log_exception('Exception while trying to notify about unanswers', e)
        return diff_resources

    def launch_result_on_end(self, task, report, metrics_json_response, sla_response):
        diff_resources = self._get_diff_resources(task, metrics_json_response)
        message = ""
        if diff_resources:
            message = "Diffs in responses, per bucket: {}\n".format(", ".join(
                "(({}/{} {}))".format(
                    diff_resource.proxy_url,
                    diff_resource.main_diff_html,
                    diff_resource.description
                ) for diff_resource in diff_resources
            ))
        title, rest_message = super(BegemotInfo, self).launch_result_on_end(
            task, report, metrics_json_response, sla_response
        )
        return title, message + rest_message

    def custom_st_update(self, st_helper, issue=None):
        # Update current acceptance dashboard item
        query = (
            'Key: {0} OR ("Linked To": {0} AND (Queue: "BEGEMOT" OR Queue: "REQWIZARD"))'
            ' "Sort By": Queue, Priority DESC, Status'.format(issue.key)
        )
        st_helper.patch_filter(query, 25593)
        # Add tags for linked tickets
        linked_issues = st_helper.st_client.issues.find(
            query='"Linked To": {0} AND (Queue: "BEGEMOT" OR Queue: "REQWIZARD")'.format(issue.key)
        )
        release_num = issue.summary[len(self.st_summary("")):]
        st_helper.st_client.bulkchange.update(
            [t.key for t in linked_issues],
            tags={"add": "begemot-{}".format(release_num)}
        ).wait()
        # Update next acceptance dashboard item
        query = "(Queue: BEGEMOT or Queue: REQWIZARD) and Status: Commited and Tags: !begemot-{}".format(release_num)
        st_helper.patch_filter(query, 25753)
        return None

    @decorators.retries(3)
    def st_post_release(self, task, auth_token, release_num=None):
        # todo: remove after complete migration to 'after_release' method
        st_helper = rm_st.STHelper(token=auth_token)
        issue = st_helper.find_ticket_by_release_number(release_num, self)
        linked_issues = st_helper.st_client.issues.find(
            query=self.notify_cfg__st__on_release_close_tickets_by_query(issue.key)
        )
        for l_issue in linked_issues:
            l_issue.update(tags=l_issue.tags + ["released"])
            l_issue = st_helper.st_client.issues[l_issue.key]
            st_helper.close_issue(l_issue, "Close ticket, major release {} has been released".format(release_num))

        reactor.reactor_push_instance(auth_token, self.reactor_namespace, release_num)

    def after_release(
        self,
        release_results, release_stage,
        major_release_num, minor_release_num,
        st_helper=None, st_issue=None, task=None,
    ):
        super(BegemotInfo, self).after_release(
            release_results, release_stage,
            major_release_num, minor_release_num,
            st_helper=st_helper, st_issue=st_issue, task=task,
        )
        query = self.notify_cfg__st__on_release_close_tickets_by_query(st_issue.key)
        if query:
            found_issues = st_helper.st_client.issues.find(query=query)
            for i in found_issues:
                i.update(tags=i.tags + ["released"])

        reactor.reactor_push_instance(rm_sec.get_rm_token(None), self.reactor_namespace, major_release_num)

    def pin_message(self, st_ticket_link, scope_number=None):
        curr_version = RMClient().get_current_version(self.name) or {}
        return "В продакшне {}-{}, приемочный тикет {}, инвайт: {}, wiki про дежурство {}".format(
            curr_version.get("scope_number"),
            curr_version.get("tag_number"),
            st_ticket_link,
            self.notify_cfg__tg__invite_link,
            self.release_duty_wiki,
        )

    def new_release_trigger_activated(self, *args, **kwargs):
        return self.stable_release_initialized_for_the_latest_scope(*args, **kwargs)

    def new_release_requirements_satisfied(self, *args, **kwargs):
        current_time = datetime.datetime.now()
        return (0 <= current_time.weekday() <= 4) and (6 <= current_time.hour <= 8)
