# -*- encoding: utf-8 -*-
from __future__ import unicode_literals


import itertools
import logging

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
import sandbox.projects.abc.client as abc_client
import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.changelogs as rm_ch
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
import sandbox.projects.release_machine.security as rm_sec
from sandbox.projects.common import decorators


PODBORKI_ABC_SERVICE_ID = 1072
COLLECTIONS_BACKEND_ABC_SERVICE_ID = 2124
ROLE_DUTY = 413
ROLE_SCOPE_TESTING = 'testing'
ROLE_SCOPE_MANAGEMENT = 'services_management'


class CollectionsBackendInfo(rm_comp.ReferenceComponent):
    FLAG_MARKERS = [
        '[ITS]',
    ]

    def create_flag_ticket(self, st_helper, main_ticket, title, change):
        revision = change['revision']
        summary = change['summary']
        author = change['commit_author']
        st_helper.st_client.issues.create(
            queue=self.notify_cfg__st__dev_queue,
            assignee=author,
            followers='abakharev',
            summary='Apply backend flag {}'.format(title),
            description='{rev} from {ticket}\n\n{summary}'.format(
                rev=lb.revision_link(revision, link_type=lb.LinkType.wiki),
                ticket=main_ticket.key,
                summary=summary,
            ),
            tags='backend_flag',
            components='Backend',
        )

    @decorators.retries(3)
    def st_post_release(self, task, auth_token, release_num=None):
        if not release_num:
            return

        st_helper = rm_st.STHelper(token=auth_token)
        try:
            changelog_resource = rm_ch.get_changelog_resource(
                self.name, release_num, task.Context.max_minor_release_num
            )
            if changelog_resource:
                main_ticket = st_helper.find_ticket_by_release_number(release_num, self)
                changelog = rm_ch.get_rm_changelog(changelog_resource)
                flags_tickets_created_for_revs = set()
                for all_changes in changelog.get("all_changes", []):
                    for change in all_changes.get("changes", []):
                        if change['revision'] in flags_tickets_created_for_revs:
                            continue
                        title = change.get("summary").splitlines()[0]
                        logging.debug("Title of the change: %s", title)
                        if any([marker in title for marker in self.FLAG_MARKERS]):
                            self.create_flag_ticket(st_helper, main_ticket, title, change)
                            flags_tickets_created_for_revs.add(change['revision'])
        except Exception as e:
            eh.log_exception("Couldn't create tickets for flag enabling from changelog", e)

    @decorators.retries(3)
    def custom_st_update(self, st_helper, issue=None):
        transition = 'RC'
        linked_issues = st_helper.st_client.issues.find(
            query='"Linked To": {0} AND Status: "Tested"'.format(issue.key)
        )
        logging.debug("Collect Tested issues: %s", " ".join([str(l_issue.key) for l_issue in linked_issues]))
        for l_issue in linked_issues:
            if l_issue.status.key == transition:
                continue
            available_transitions = [tr.id for tr in l_issue.transitions.get_all()]
            if transition in available_transitions:
                logging.info("Execute transition '%s'.", transition)
                issue.transitions[transition].execute()

    @decorators.retries(3)
    def change_ticket_on_post_deploy(self, major_release, st_helper, task):
        if not self.notify_cfg__st__workflow:
            # didn't specify workflow, nothing to do here
            return

        issue = st_helper.find_ticket_by_release_number(major_release, self)
        linked_issues = st_helper.st_client.issues.find(
            query='"Linked To": {0} AND (Status: "RC" OR Status: "Tested") AND (Queue: "PODB")'.format(issue.key)
        )
        logging.debug("Collect issues: %s", " ".join([str(l_issue.key) for l_issue in linked_issues]))
        for l_issue in itertools.chain(linked_issues, [issue]):
            st_helper.close_issue(
                l_issue, "Close ticket. Collections Backend release {} has been released".format(major_release)
            )

    def get_stable_release_label(self, release_item, release_num):
        if release_num:
            return super(CollectionsBackendInfo, self).get_stable_release_label(
                release_item, release_num
            )
        else:
            return release_item.name

    @property
    def notify_cfg__st__followers(self):
        abc_api = abc_client.AbcClient(rm_sec.get_rm_token(None))

        testers = abc_api.get_people_from_service(PODBORKI_ABC_SERVICE_ID, role_scope=ROLE_SCOPE_TESTING)
        managers = abc_api.get_people_from_service(PODBORKI_ABC_SERVICE_ID, role_scope=ROLE_SCOPE_MANAGEMENT)
        team = abc_api.get_people_from_service(COLLECTIONS_BACKEND_ABC_SERVICE_ID)

        return testers + managers + team
