# -*- coding: utf-8 -*-
from __future__ import unicode_literals


import logging
import calendar

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.components_info as rm_comp

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine.core import ReleasedItem
from sandbox.projects.release_machine.core import ReleasedResource


class QuasarInfo(rm_comp.ReferenceComponent):
    def get_mail_subject(self, **kwargs):
        return '[{name}] Стабильная ветка {name}/{branch_prefix}-{major_release_num} отведена от trunk@{revision}'.format(**kwargs)

    def notify_cfg__mail__get_start_letter(self, st_issue_key, **kwargs):
        subject = self.get_mail_subject(name=self.name, branch_prefix=self.svn_cfg__branch_prefix, **kwargs)
        content = (
            '{subject}\n\n'
            'Процесс приемки проходит в тикете: {st_base_url}{st_key}\n\n'
        ).format(
            subject=subject,
            st_base_url=rm_const.Urls.STARTREK,
            st_key=st_issue_key,
        )
        return {
            "recipients": self.notify_cfg__mail__mailing_list,
            "subject": subject,
            "body": content,
        }

    @staticmethod
    def get_release_reminder():
        return (
            '((https://wiki.yandex-team.ru/quasar/station-dev-release-flow/ Релизный процесс Quasar))\n\n'
            'Релиз инженер получает ОК/НЕОК от требуемых людей в комментариях и заполняет таблицу ниже. '
            'В поле **"Комментарий"** нужно дать ссылки или подробности соответствующих проверок, '
            'а в поле **"ОК/НЕОК"** нужно оставить что-то одно из двух, причем решение принимает **"Ответственный"**. '
            'Где !!(зел)ОК!! – можно катить в продакшен, а !!НЕОК!! – нельзя катить в продакшен.\n\n'
            'После получения ОК в CHECKS необходимо:\n'
            '- Выложить прошивку в Бету\n'
            '- Сообщить об этом в чат Станция: OPS center\n'
            '- Перевести этот Тикет в статус ВЫКЛАДЫВАЕТСЯ\n'
            '- После успешного периода в Бете выложить в Прод\n'
            '- Сообщить об этом в чат Станция: OPS center\n'
            '- Перевести тикет в статус ЗАКРЫТ\n'
        )

    @staticmethod
    def get_checks_table():
        return (
            '#| '
            '|| **Проверка** | **Ответственный** | **ОК/НЕОК** |**Комментарий** || '
            '|| Пройден pre-smoke | | {is_ok} |  || '
            '|| Все новые изменения протестированы | | {is_ok} | || '
            '|| Ручной регресс ран пройден | кто:osennikovak кто:cubovaya кто:kaplya | {is_ok} | Ссылка на тестран с результатами ручного тестирования || '
            '|| Тикет на заказ патчноутов оформлен | кто:alenagromova | {is_ok} | Ссылка на тикет|| '
            '|#'
        ).format(is_ok='!!(зел)ОК!!/!!НЕОК!!?')

    @staticmethod
    def get_firmware_refs():
        return '<TODO>'

    def st_description(self, release_num=None):
        return (
            '<{{Памятка релиз инженеру\n\n{reminder}\n}}>'
            'Ссылки на прошивки:\n{firmware_refs}\n\n'
            '{base_descr}\n\n'
            '=====CHECKS\n{checks_table}\n\n'
        ).format(
            reminder=self.get_release_reminder(),
            firmware_refs=self.get_firmware_refs(),
            base_descr=super(QuasarInfo, self).st_description(release_num),
            checks_table=self.get_checks_table()
        )

    def get_last_release(self, stage=None):
        """This is an empty iterator."""
        return
        yield


class QuasarMultiplatformInfo(QuasarInfo):
    def set_platform(self, platform):
        self.platform = platform

    def set_release_item(self, release_item):
        self.release_item = release_item

    @property
    def st_tags(self):
        if hasattr(self, 'platform'):
            issue_tags = ['quasar_{}'.format(self.platform), '{}_release'.format(self.name)]
        else:
            issue_tags = [self.name]
        return ["rm_main_ticket"] + issue_tags

    def st_description(self, release_num=None):
        if hasattr(self, 'platform'):
            # platofrm release ticket
            return QuasarInfo.st_description(self, release_num)
        else:
            # branch ticket
            return rm_comp.ReferenceComponent.st_description(self, release_num)

    def st_summary(self, release_num):
        if hasattr(self, 'platform'):
            # platofrm release ticket
            return self.notify_cfg__st__platform_summary_template.format(self.platform, release_num)
        else:
            # branch ticket
            return super(QuasarMultiplatformInfo, self).st_summary(release_num)

    def get_mail_subject(self, **kwargs):
        if hasattr(self, 'platform'):
            return '[{name}] Релиз платформы {platform} из ветки {name}/{branch_prefix}-{major_release_num}@{revision}'.format(
                platform=self.platform, **kwargs)
        else:
            return super(QuasarMultiplatformInfo, self).get_mail_subject(**kwargs)

    @property
    def notify_cfg__st__queue(self):
        if hasattr(self, 'platform'):
            return self.notify_cfg__st__platform_queue
        else:
            return self._cfg.notify_cfg.st.queue

    def get_last_deploy(self, token=None, only_level=None):
        # release_status = only_level or ctt.ReleaseStatus.STABLE
        release_status = ctt.ReleaseStatus.TESTING
        released_resources = []

        for res_info in self.releases_cfg__resources_info:
            res_type = res_info.resource_type

            if res_type == "QUASAR_DAEMONS":
                logging.info(
                    "Skip deploy for %s since daemons are always releasing together with images",
                    res_info.resource_name
                )
                continue

            eh.verify(res_type, "Resource type is {}! Check your component's resources_info!".format(res_type))
            attrs = {"released": release_status}
            attrs.update(res_info.attributes or {})
            last_released_res = sdk2.Resource.find(
                type=res_type,
                arch='linux',
                attrs=attrs
            ).order(-sdk2.Resource.id).first()

            if last_released_res is None:
                logging.info("Got no %s releases for resource %s, try to fallback", release_status, res_type)
                released_resources.append(ReleasedResource(
                    resource_name=res_info.resource_name,
                    status=release_status,
                    component=self.name,
                ))
            else:
                logging.info(
                    "Got last released %s resource for '%s' component: %s",
                    res_type, self.name, last_released_res.id
                )

                release_item = ReleasedItem(res_info.resource_name, last_released_res, res_info.build_ctx_key)
                major_num, minor_num = self.get_release_numbers(release_item)
                last_released_res = ReleasedResource(
                    id=last_released_res.id,
                    build_task_id=last_released_res.task_id,
                    major_release=major_num,
                    minor_release=minor_num,
                    timestamp=calendar.timegm(last_released_res.updated.timetuple()),
                    component=self.name,
                    status=release_status,
                    owner=last_released_res.owner,
                    resource_name=res_info.resource_name,
                )
                released_resources.append(last_released_res)

        return released_resources


class QuasarAndroidInfo(QuasarInfo):
    @staticmethod
    def get_firmware_refs():
        return (
            'USER:\n<TODO:user>\n\n'
            'ENG:\n<TODO:eng>\n\n'
            'USERDEBUG:\n<TODO:userdebug>\n\n'
        )


class QuasarReleaseDebugMixin(object):
    @staticmethod
    def get_firmware_refs():
        return (
            'RELEASE:\n<TODO:release>\n\n'
            'DEBUG:\n<TODO:debug>\n\n'
        )


class QuasarBuildTypesMixin(object):
    @staticmethod
    def get_firmware_refs():
        return (
            'USER:\n<TODO:user>\n\n'
            'ENG:\n<TODO:eng>\n\n'
            'USERDEBUG:\n<TODO:userdebug>\n\n'
        )
