# -*- coding: utf-8 -*-
from __future__ import unicode_literals


import logging
import re

import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
from sandbox.sdk2.vcs.svn import Arcadia

RTMR_SAN_TYPES = ["GCC", "ASAN", "MSAN", "TSAN", "UBSAN", "DEBUG", "VALGRIND"]


class RtmrInfo(rm_comp.BranchedFromBranch, rm_comp.mixin.Changelogged, rm_comp.mixin.Startreked):

    retrieved_source_branch_url = None

    @staticmethod
    def check_source_branch_override():
        override_path = "arcadia:/arc/trunk/arcadia/rtmapreduce/ci/rm/override_source_branch"
        override_nums = rm_svn.SvnHelper.read_file(override_path).strip().split("-")
        override_num_major = int(override_nums[0])
        override_num_minor = int(override_nums[1])
        return override_num_major, override_num_minor

    @property
    def source_branch_url(self):
        if not self.retrieved_source_branch_url:
            logging.info("Looking for kikimr branch to fork rtmr branch from..")
            kikimr_rootdir = "arcadia:/arc/branches/kikimr"

            # Get prev released rtmr branch
            prev_rtmr = self.last_branch_path
            logging.info("previously released rtmr: %s", prev_rtmr)

            # Get parent branch of the previously released rtmr
            last_rtmr_parent = rm_svn.SvnHelper.get_parent_branch(prev_rtmr)
            # Subversion prints folders with a trailing slash in "svn list", let's have it too
            if last_rtmr_parent[-1] != "/":
                last_rtmr_parent += "/"
            logging.info("parent of previous rtmr: %s", last_rtmr_parent)

            # The first actual kikimr release with RM is numbered '18-7',
            # let's target it by default
            prev_kikimr_branch_major = 19
            prev_kikimr_branch_minor = 2
            match = re.search(r"branches/kikimr/stable-(\d+)-(\d+)/", last_rtmr_parent,)
            if match:
                prev_kikimr_branch_major = int(match.group(1))
                prev_kikimr_branch_minor = int(match.group(2))
            logging.info("using %d-%d as prev kikimr branch num", prev_kikimr_branch_major, prev_kikimr_branch_minor)

            next_kikimr_num_major = prev_kikimr_branch_major
            next_kikimr_num_minor = prev_kikimr_branch_minor + 2
            logging.info("using %d-%d as next kikimr branch num", next_kikimr_num_major, next_kikimr_num_minor)

            # Check if source branch for the next release is overridden in a special place in Subversion
            override_next_num_major, override_next_num_minor = RtmrInfo.check_source_branch_override()
            logging.info("Got override branch num %d-%d", override_next_num_major, override_next_num_minor)
            if override_next_num_major > next_kikimr_num_major:
                next_kikimr_num_major = override_next_num_major
                next_kikimr_num_minor = override_next_num_minor
                logging.info("overriding next source branch num with %d-%d", override_next_num_major, override_next_num_minor)
            elif override_next_num_major == next_kikimr_num_major:
                if override_next_num_minor > next_kikimr_num_minor:
                    next_kikimr_num_minor = override_next_num_minor
                    logging.info("overriding next source branch num with %d-%d", override_next_num_major, override_next_num_minor)

            # Check the kikimr branch exists
            kikimr_branch_names = rm_svn.SvnHelper.list_files_with_pattern(
                kikimr_rootdir,
                "^stable-{}-{}/$".format(next_kikimr_num_major, next_kikimr_num_minor)
            )
            if len(kikimr_branch_names) != 1:
                logging.error("detected %d kikimr branches for num %d-%d",
                              len(kikimr_branch_names),
                              next_kikimr_num_major,
                              next_kikimr_num_minor)
                raise Exception("unable to find next source branch")

            self.retrieved_source_branch_url = kikimr_rootdir + "/" + kikimr_branch_names[0]
            logging.info("source branch url is found: %s", self.retrieved_source_branch_url)
        return self.retrieved_source_branch_url

    def get_latest_trunk_revision_before_branch(self, branch_path):
        """
        :param branch_path: SVN path for branch
        :return: trunk revision as an integer
        """
        logging.info("Starting at the branch %s", branch_path)
        cur_revision = "HEAD"

        """
        Two steps:
        - get from rtmr branch to kikimr branch
        - get from kikimr branch to the trunk
        """

        for it in range(2):
            # Go down to the branching point
            svn_log = Arcadia.log(
                branch_path,
                revision_from=cur_revision,
                revision_to="1",
                limit=10000,
                stop_on_copy=True,
            )
            branching_point = svn_log[-1]["revision"]
            logging.info("branching point at r%s", branching_point)
            # Get past the branching point
            svn_log = Arcadia.log(
                branch_path,
                revision_from=branching_point,
                revision_to="1",
                limit=2,
            )
            assert len(svn_log) == 2
            cur_revision = svn_log[-1]["revision"]
            logging.info("parent point at r%s", cur_revision)
        return int(cur_revision)

    @property
    def first_rev(self):
        if not self._first_rev:
            if self.prev_branch_num > 0:
                logging.info("Detecting the first revision for %s", self.name)
                branch_path = self.prev_branch_path
                self._first_rev = self.get_latest_trunk_revision_before_branch(branch_path) + 1
            else:
                # return something sensible in case of the first release
                self._first_rev = int(self.last_rev) - 1
            logging.info("First revision detected: %s", self._first_rev)
        return self._first_rev

    def st_description(self, release_num=None):
        rtmr = self.last_branch_path
        arcanum_rtmr = rm_const.Urls.make_arcanum_url(rtmr)
        kikimr = rm_svn.SvnHelper.get_parent_branch(rtmr)
        arcanum_kikimr = rm_const.Urls.make_arcanum_url(kikimr)
        descr = "Stable branch: {}\n".format(arcanum_rtmr)
        descr += "Parent kikimr branch: {}\n".format(arcanum_kikimr)
        descr += super(RtmrInfo, self).st_description(release_num)
        return descr
