# -*- coding: utf-8 -*-
from __future__ import unicode_literals


import logging
import calendar

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.components_info as rm_comp

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine.core import ReleasedItem
from sandbox.projects.release_machine.core import ReleasedResource
from sandbox.projects.release_machine.components.configs.smart_devices import _common as common
from sandbox.projects.release_machine.components.configs.smart_devices.smart_devices import JOB_GRAPHS


class SmartDevicesInfo(rm_comp.ReferenceComponent):
    def get_device_job_graph(self):
        job_graph, kwargs = JOB_GRAPHS[self.platform]
        return job_graph(platform=self.platform, component_name=self.name, **kwargs)

    def set_platform(self, platform):
        self.platform = platform

    @property
    def st_tags(self):
        if hasattr(self, 'platform'):
            issue_tags = ['quasar_{}'.format(self.platform), '{}_release'.format(self.name)]
        else:
            issue_tags = [self.name]
        return ["rm_main_ticket"] + issue_tags

    def st_description(self, release_num=None):
        desc = super(SmartDevicesInfo, self).st_description(release_num)
        if hasattr(self, 'platform'):
            desc = (
                '((https://wiki.yandex-team.ru/quasar/release-duty/ Памятка релиз инженеру))\n\n'
                'Ссылки на прошивки:\n{firmware_refs}\n\n'
                '{base_desc}\n\n'
            ).format(
                firmware_refs=self.get_firmware_refs(),
                base_desc=desc,
            )
        return desc

    def get_firmware_refs(self):
        device_job_graph = self.get_device_job_graph()
        todo_refs = ''
        for build_variant_name in device_job_graph.get_all_build_variant_names():
            if build_variant_name is None:
                todo_refs += '<TODO>\n\n'
            else:
                todo_refs += '{}:\n<TODO:{}>\n\n'.format(build_variant_name.upper(), build_variant_name)
        return todo_refs

    def st_summary(self, release_num):
        if hasattr(self, 'platform'):
            return self.notify_cfg__st__release_summary_template.format(self.platform, release_num)
        else:
            return super(SmartDevicesInfo, self).st_summary(release_num)

    @property
    def notify_cfg__st__queue(self):
        if hasattr(self, 'platform'):
            return self.notify_cfg__st__release_queue
        else:
            return self._cfg.notify_cfg.st.queue

    def get_mail_subject(self, **kwargs):
        if hasattr(self, 'platform'):
            return '[{name}] Релиз платформы {platform} из ветки {name}/{branch_prefix}-{major_release_num}@{revision}'.format(
                platform=self.platform, **kwargs)
        else:
            return '[{name}] Стабильная ветка {name}/{branch_prefix}-{major_release_num} отведена от trunk@{revision}'.format(**kwargs)

    def notify_cfg__mail__get_start_letter(self, st_issue_key, **kwargs):
        subject = self.get_mail_subject(name=self.name, branch_prefix=self.svn_cfg__branch_prefix, **kwargs)
        content = (
            '{subject}\n\n'
            'Процесс приемки проходит в тикете: {st_base_url}{st_key}\n\n'
        ).format(
            subject=subject,
            st_base_url=rm_const.Urls.STARTREK,
            st_key=st_issue_key,
        )
        return {
            "recipients": self.notify_cfg__mail__mailing_list,
            "subject": subject,
            "body": content,
        }

    def get_last_deploy(self, token=None, only_level=None):
        return []

    def get_last_release(self, stage=None):
        release_statuses = [
            ctt.ReleaseStatus.STABLE,
            ctt.ReleaseStatus.PRESTABLE,
            ctt.ReleaseStatus.TESTING,
        ]

        if not hasattr(self, 'platform'):
            # use only per platform changelogs
            return

        resources_info = self.get_device_job_graph().release_items
        for res_info in resources_info:
            if res_info.name != common.ReleaseItemsNames.ota(self.platform):
                logging.info(
                    "Skip release item %s since all resources are released together, use only ota image",
                    res_info.resource_name
                )
                continue

            last_released_res = None
            last_released_res_status = None
            for status in release_statuses:
                res_type = res_info.resource_type
                eh.verify(res_type, "Resource type is {}! Check your component's resources_info!".format(res_type))
                attrs = {"released": status}
                attrs.update(res_info.attributes or {})
                res = sdk2.Resource.find(
                    type=res_type,
                    arch='linux',
                    attrs=attrs
                ).order(-sdk2.Resource.id).first()
                if res is not None and (last_released_res is None or res.id > last_released_res.id):
                    last_released_res = res
                    last_released_res_status = status

            if last_released_res is None:
                logging.info("Got no releases for resource %s, try to fallback", res_info.resource_type)
                yield ReleasedResource(
                    resource_name=res_info.resource_name,
                    status=ctt.ReleaseStatus.STABLE,
                    component=self.name,
                )
            else:
                logging.info(
                    "Got last released %s resource for '%s' component: %s",
                    res_info.resource_type, self.name, last_released_res.id
                )

                release_item = ReleasedItem(res_info.resource_name, last_released_res, res_info.build_ctx_key)
                major_num, minor_num = self.get_release_numbers(release_item)
                yield ReleasedResource(
                    id=last_released_res.id,
                    build_task_id=last_released_res.task_id,
                    major_release=major_num,
                    minor_release=minor_num,
                    timestamp=calendar.timegm(last_released_res.updated.timetuple()),
                    component=self.name,
                    status=last_released_res_status,
                    owner=last_released_res.owner,
                    resource_name=res_info.resource_name,
                )
