# -*- coding: utf-8 -*-
from __future__ import unicode_literals


import json
import logging
import re
import requests

import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.changelogs as rm_ch
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
from sandbox.projects.common import decorators
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine import rm_utils


class UpperInfo(rm_comp.ReferenceComponentMetricsed):

    @decorators.retries(3)
    def st_post_release(self, task, auth_token, release_num=None):
        st_helper = rm_st.STHelper(token=auth_token)
        try:
            changelog_resource = rm_ch.get_changelog_resource(self.name, release_num)
            if changelog_resource:
                main_ticket = st_helper.find_ticket_by_release_number(release_num, self)
                comment_text = "Component {} is releasing to production. Release includes commits of this ticket.\n" \
                    "https://st.yandex-team.ru/{}\nSandbox task: {}\nRelease number: {}"
                changelog = rm_ch.get_rm_changelog(changelog_resource)
                for change in changelog.get("changes", []):
                    for ticket in change.get("startrek_tickets", []):
                        issue = st_helper.get_ticket_by_key(ticket)
                        issue.comments.create(
                            text=comment_text.format(
                                self.name, main_ticket.key, lb.task_wiki_link(task.id), release_num
                            )
                        )
        except Exception as e:
            eh.log_exception("Couldn't write in tickets from changelog", e)

    def update_config_wiki_pages(self, token):  # SEARCH-1535
        try:
            session = requests.Session()
            session.headers['Authorization'] = "OAuth {}".format(token)
            session.verify = False
            next_page_api_url = self.next_config_wiki_page(api=True)
            next_page_info = json.loads(session.get(next_page_api_url).text)
            logging.info("Next page info:\n%s", next_page_info)
            if "error" in next_page_info:
                if next_page_info["error"].get("error_code") != "NOT_FOUND":
                    logging.info(
                        "Error retrieving wiki page: %s\nError:\n%s",
                        self.next_config_wiki_page(),
                        next_page_info["error"]
                    )
                else:
                    logging.info("Create new wiki page")
                    response = session.post(next_page_api_url, data={
                        "title": "Изменения в {}-oй версии noapache".format(self.next_branch_num + 1),
                        "body": self.config_wiki_page_text,
                    })
                    logging.info("Creation finished with status code: %s", response.status_code)
                    logging.info("Update current wiki page")
                    current_page_api_url = self.current_config_wiki_page(api=True)
                    raw_text = json.loads(session.get("{}/.raw".format(current_page_api_url)).text)
                    current_body = raw_text.get("data", {}).get("body", "")
                    current_title = raw_text.get("data", {}).get("title", "")
                    if current_body and current_title:
                        response = session.post(current_page_api_url, data={
                            "title": current_title,
                            "body": (
                                "!!Ветка закрыта для feature комитов (только исправления критических ошибок), "
                                "новые пожелания на изменения в конфигурации писать (({} сюда)).!!\n\n{}"
                            ).format(self.next_config_wiki_page(), current_body),
                        })
                        logging.info("Updating finished with status code: %s", response.status_code)
                    else:
                        logging.info("Something wrong with current page:\n%s\nPage is not updated!", raw_text)
            else:
                logging.info("Page '%s' is already exists", self.next_config_wiki_page())
        except Exception as e:
            eh.log_exception("Cannot update config pages, error", e)

    def current_config_wiki_page(self, api=False):
        return "{}poiskovajaplatforma/upple/sequence/upple{}/".format(
            rm_const.Urls.WIKI_API if api else rm_const.Urls.WIKI, self.next_branch_num
        )

    def next_config_wiki_page(self, api=False):
        return "{}poiskovajaplatforma/upple/sequence/upple{}/".format(
            rm_const.Urls.WIKI_API if api else rm_const.Urls.WIKI, self.next_branch_num + 1
        )

    def get_stable_release_label(self, release_item, release_num):
        arcadia_checkout_url = rm_utils.get_input_or_ctx_field(release_item.build_task_id, release_item.build_ctx_key)
        logging.debug('Checkout arcadia url: %s=%s', release_item.build_ctx_key, arcadia_checkout_url)
        build_tag = re.search(r'.*/tags/(.*)/[^/]', arcadia_checkout_url).group(1)
        if release_item.name == "rearrange_data" and build_tag.endswith("/arcadia_tests_data"):
            build_tag = build_tag[:-19]

        return "{} ({}): {}".format(self.name, release_item.name, build_tag)

    @property
    def config_wiki_page_text(self):
        return (
            "Здесь пишутся пожелания на изменения в конфигурации (кроме опций переранжирований - см. ниже)\n"
            "Напоминаем, что коммиты, влияющие на выдачу noapache, "
            "должны тегироваться [diff:upper] в комментарии к коммиту!\n"
            "**Разное**\n"
            "**!!Конфиг!!**\n"
            "!!Изменения в конфигурацию переранжирований вносят сами разработчики исправлениями "
            "((https://a.yandex-team.ru/arc/trunk/arcadia/search/web/rearrs_upper/conf/conf.json файла)).\n"
            "С вопросами по формату обращаться к "
            "((https://wiki.yandex-team.ru/jandekspoisk/kachestvopoiska/rearrange/config/formalnoeopisanie/ "
            "документации)) и автору staff:velavokr!!"
        )

    # @property
    def st_description(self, release_num=None):
        return (
            "Ветка noapacheupper-{branch_num} отведена из trunk@{rev}\n"
            "Используются YAPPY беты приёмки.\n"
            "Беты генерируются по номерам тегов и имеют вид: "
            "noapache-<subtype>-{branch_num}-<tag_num>.hamster.yandex.ru\n"
            "Контрольная бета: upper-hamster.hamster.yandex.ru\n"
        ).format(
            branch_num=release_num or self.last_branch_num,
            rev=self.last_rev,
        ) + super(UpperInfo, self).st_description(release_num=release_num)

    def notify_cfg__mail__get_start_letter(self, st_issue_key, **kwargs):
        """
            Custom letter for developers (revised in SEARCH-2616)
        """
        subject = "[upper] Стабильная ветка noapacheupper-{} отведена из trunk@{}".format(
            self.last_branch_num, self.last_rev
        )
        content = (
            "{subj}\n\n"
            "Changelog: {changelog_link}\n\n"
            "Отслеживание процесса приёмки: {st_base_url}{st_key}\n\n"
            "Заявки по изменению в конфигурации для следующей стабильной ветки пишутся на странице\n"
            "https://wiki.yandex-team.ru/poiskovajaplatforma/upple/sequence/upple{branch_num}\n\n"
            "Опции переранжирований *ReArrangeOptions* разработчики правят сами в trunk\n"
            "в файле https://a.yandex-team.ru/arc/trunk/arcadia/search/web/rearrs_upper/conf/conf.json\n"
        ).format(
            subj=subject,
            changelog_link="".join([rm_const.Urls.WIKI, self.changelog_major_url()]),
            branch_num=self.last_branch_num + 1,
            st_base_url=rm_const.Urls.STARTREK,
            st_key=st_issue_key,
        )
        return {
            "recipients": self.notify_cfg__mail__mailing_list,
            "subject": subject,
            "body": content,
        }

    def new_release_trigger_activated(self, *args, **kwargs):
        return self.stable_release_trigger_activated(*args, **kwargs)
