# -*- coding: utf-8 -*-
from __future__ import unicode_literals


import logging
import re

import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
from sandbox.sdk2.vcs.svn import Arcadia


class UserfeatInfo(rm_comp.BranchedFromBranch, rm_comp.mixin.Changelogged, rm_comp.mixin.Startreked):
    retrieved_source_branch_url = None

    @staticmethod
    def check_source_branch_override():
        override_path = "arcadia:/arc/trunk/arcadia/quality/userdata/override_source_branch"
        override_num_string = rm_svn.SvnHelper.read_file(override_path).strip()
        if not override_num_string.isdigit():
            return 0
        return int(override_num_string)

    @property
    def source_branch_url(self):
        if not self.retrieved_source_branch_url:
            logging.info("Looking for user_sessions branch to fork userfeat branch from..")
            usersessions_rootdir = "arcadia:/arc/branches/user_sessions"

            # Get prev released userfeat branch
            prev_userfeat = self.last_branch_path
            logging.info("previously released userfeat: %s", prev_userfeat)

            # Get parent branch of the previously released userfeat
            last_userfeat_parent = rm_svn.SvnHelper.get_parent_branch(prev_userfeat)
            # Subversion prints folders with a trailing slash in "svn list", let's have it too
            if last_userfeat_parent[-1] != "/":
                last_userfeat_parent += "/"
            logging.info("parent of previous userfeat: %s", last_userfeat_parent)

            # The first actual user_sessions release with RM is numbered '8',
            # let's target it by default
            prev_usersessions_num = 7
            match = re.search(r"branches/user_sessions/stable-(\d+)/", last_userfeat_parent)
            if match:
                prev_usersessions_num = int(match.group(1))
            logging.info("using %d as prev user_sessions branch num", prev_usersessions_num)
            next_usersessions_num = prev_usersessions_num + 1

            # Check if source branch for the next release is overridden in a special place in Subversion
            override_next_num = UserfeatInfo.check_source_branch_override()
            if override_next_num > next_usersessions_num:
                logging.info("overriding next source branch num with %d", override_next_num)
                next_usersessions_num = override_next_num

            # Check the user_sessions branch exists
            usersessions_branch_names = rm_svn.SvnHelper.list_files_with_pattern(
                usersessions_rootdir,
                "^stable-{}/$".format(next_usersessions_num)
            )
            if len(usersessions_branch_names) != 1:
                logging.error("detected %d user_sessions branches for num %d",
                              len(usersessions_branch_names),
                              next_usersessions_num)
                raise Exception("unable to find next source branch")

            self.retrieved_source_branch_url = usersessions_rootdir + "/" + usersessions_branch_names[0]
            logging.info("source branch url is found: %s", self.retrieved_source_branch_url)
        return self.retrieved_source_branch_url

    def get_latest_trunk_revision_before_branch(self, branch_path):
        """
        :param branch_path: SVN path for branch
        :return: trunk revision as an integer
        """
        logging.info("Starting at the branch %s", branch_path)
        cur_revision = "HEAD"

        """
        Two steps:
        - get from userfeat branch to userdata branch
        - get from userdata branch to the trunk
        """

        for it in range(2):
            # Go down to the branching point
            svn_log = Arcadia.log(
                branch_path,
                revision_from=cur_revision,
                revision_to="1",
                limit=10000,
                stop_on_copy=True,
            )
            branching_point = svn_log[-1]["revision"]
            logging.info("branching point at r%s", branching_point)
            # Get past the branching point
            svn_log = Arcadia.log(
                branch_path,
                revision_from=branching_point,
                revision_to="1",
                limit=2,
            )
            assert len(svn_log) == 2
            cur_revision = svn_log[-1]["revision"]
            logging.info("parent point at r%s", cur_revision)
        return int(cur_revision)

    @property
    def first_rev(self):
        if not self._first_rev:
            if self.prev_branch_num > 0:
                logging.info("Detecting the first revision for %s", self.name)
                branch_path = self.prev_branch_path
                self._first_rev = self.get_latest_trunk_revision_before_branch(branch_path) + 1
            else:
                # return something sensible in case of the first release
                self._first_rev = int(self.last_rev) - 1
            logging.info("First revision detected: %s", self._first_rev)
        return self._first_rev

    def st_description(self, release_num=None):
        userfeat = self.last_branch_path
        arcanum_userfeat = rm_const.Urls.make_arcanum_url(userfeat)
        userdata = rm_svn.SvnHelper.get_parent_branch(userfeat)
        arcanum_userdata = rm_const.Urls.make_arcanum_url(userdata)
        descr = "Stable branch: {}\n".format(arcanum_userfeat)
        descr += "Parent user_sessions branch: {}\n".format(arcanum_userdata)
        descr += super(UserfeatInfo, self).st_description(release_num)
        return descr
