# -*- coding: utf-8 -*-
import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.release_machine.core import DeployedResource
from sandbox.projects.release_machine.components import components_info as rm_comp
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.yabs.release.version.server import ServerVersionHelper


logger = logging.getLogger(__name__)


class YabsServerInfo(rm_comp.ReferenceComponent):
    def get_last_deploy(self, token=None, only_level=None):
        if only_level is not None and only_level != rm_const.ReleaseStatus.stable:
            return None
        nanny_token = token or rm_sec.get_rm_token(None)
        server_version_helper = ServerVersionHelper(nanny_token=nanny_token)
        versions = server_version_helper.get_deployed_versions_info('bsfront_production', 'BS_RELEASE_TAR')
        logger.debug("Got service versions: %s", versions)
        server_version, version_info = max(versions.items(), key=lambda v: len(v[1]["nanny_services"]))
        return [
            DeployedResource(
                id=version_info["resource_id"],
                build_task_id=version_info["build_task_id"],
                timestamp=version_info["deploy_time"],
                major_release=str(server_version.major),
                minor_release=str(server_version.minor),
                component=self.name,
            )
        ]

    def new_release_requirements_satisfied(self, *args, **kwargs):
        """
        New release can be created on:
        - Mon...Fri 00:00 - 10:00 UTC + 3 (Moscow Time)
        """
        now = datetime.utcnow() + timedelta(hours=3)  # Moscow Time

        # Mon: 0, Tue: 1, Wed: 2, Thu: 3, Fri: 4, Sat: 5, Sun: 6
        # Releases from Monday till Friday
        if now.weekday() > 4:
            return False

        # Releases from 4am till 10am
        if now.hour <= 3 or now.hour >= 10:
            return False

        return True

    def get_new_release_revision(self, *args, **kwargs):
        """
        Returns revision to launch new release on. Checks last successful run for yabs_server's GET_LAST_GOOD_REVISION_MONITOR scheduler
        """
        last_good_revision_monitor_scheduler_id = 12674  # https://sandbox.yandex-team.ru/scheduler/12674/view
        last_successful_task = sdk2.Task["GET_LAST_GOOD_REVISION_MONITOR"].find(
            scheduler=last_good_revision_monitor_scheduler_id,
            limit=1,
            status=ctt.Status.SUCCESS,
        ).first()
        if not last_successful_task:
            logger.error("No successful run for scheduler #%s found", last_good_revision_monitor_scheduler_id)
            return None
        good_revision = last_successful_task.Parameters.good_revision
        if not good_revision:
            logger.error("No good revision found, check task #%s for more info", last_successful_task.id)
            return None
        return good_revision
