import logging
import typing  # noqa

import sandbox.projects.release_machine.components.job_graph.job_data as jg_data
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.core.const as rm_const


class Arrow(object):
    """ Interface for passing data from one job to another """

    def set_params_dict(self, params_dict, params, rm_config):
        raise NotImplementedError

    def _check_params_dict(self, input_key, params_dict, rm_config):
        if input_key in params_dict:
            logging.error(
                "[{comp_name}] Got input_key '{input_key}' already filled in params_dict {params_dict}".format(
                    comp_name=rm_config.name,
                    input_key=input_key,
                    params_dict=params_dict,
                )
            )


class GlobalResourceData(jg_data.JobData, Arrow):
    """ Interface for getting data from global resources """

    def __init__(self, input_key, resource_name, override=False):
        """
        :param input_key:  str, name of parameter in child sandbox task
        :param resource_name: str, name of global resource
        """
        jg_data.JobData.__init__(self, input_key, override=override)
        self.resource_name = resource_name

    def data(self, params, *args):
        return params.global_resources[self.resource_name]

    def set_params_dict(self, params_dict, params, rm_config):
        self._check_params_dict(self.input_key, params_dict, rm_config)
        params_dict[self.input_key] = self.data(params)


class ParentsData(jg_data.JobData, Arrow):
    """ Interface for transforming data from different resources """

    def __init__(self, input_key, triggers, transform, override=False):
        """
        :param input_key: str, parameter name in child sandbox task.
        :param triggers: list of objects with type Arrow.
        Trigger gets value from parent_job and puts it in child_job params.
        :param transform: function, makes one value from many parent_job parameters.
        """
        jg_data.JobData.__init__(self, input_key, override=override)
        self.triggers = jg_utils.tuplify(triggers)
        self.transform = transform

    def data(self, params, rm_config):
        params_dict = {}
        for trigger in self.triggers:
            # We add this check in case, when some of triggers in ParentsData are None's.
            # In this case we should add default params in function transform.
            if trigger:
                trigger.set_params_dict(params_dict, params, rm_config)
        return self.transform(**params_dict)

    def set_params_dict(self, params_dict, params, rm_config):
        self._check_params_dict(self.input_key, params_dict, rm_config)
        params_dict[self.input_key] = self.data(params, rm_config)


class JustPassData(jg_data.JobData, Arrow):
    """
        Puts in child job `input_key` value `value`.
        Mostly needs for escape free variables in lambda-function transform.
    """

    def __init__(self, input_key, value, override=False):
        super(JustPassData, self).__init__(input_key, override)
        self.value = value

    def data(self, *args):
        return self.value

    def set_params_dict(self, params_dict, params, rm_config):
        self._check_params_dict(self.input_key, params_dict, rm_config)
        params_dict[self.input_key] = self.data()


class ParamsData(jg_data.JobData, Arrow):
    def __init__(self, input_key, transform=lambda x, rm_config: x, override=False):
        jg_data.JobData.__init__(self, input_key, override=override)
        self.transform = transform

    def data(self, params, rm_config):
        return self.transform(params, rm_config)

    def set_params_dict(self, params_dict, params, rm_config):
        self._check_params_dict(self.input_key, params_dict, rm_config)
        params_dict[self.input_key] = self.data(params, rm_config)


class ValueAndParamsData(ParamsData, Arrow):
    def __init__(self, input_key, value, transform=lambda y, x, rm_config: x, override=False):
        super(ValueAndParamsData, self).__init__(input_key, transform=transform, override=override)
        self.transform = transform
        self.value = value

    def data(self, params, rm_config):
        return self.transform(self.value, params, rm_config)


class JobTrigger(Arrow):
    """ Class used for passing values from parent job to child job """
    def __init__(self, job_type, parent_job_data=(), job_name_parameter=""):
        """
        :param job_type: str, should be one of JobTypes
        :param parent_job_data: list, contains objects to pass values from parent to child
        :param job_name_parameter: str, additional parameter like build_item or release_to
        """
        self.job_type = job_type
        self.job_name_parameter = job_name_parameter
        self.parent_job_data = jg_utils.tuplify(parent_job_data)  # type: typing.Tuple[jg_data.JobData]

    def this_job_name(self, comp_name):
        # type: (typing.AnyStr) -> typing.AnyStr
        return rm_const.JobTypes.rm_job_name(self.job_type, comp_name, self.job_name_parameter)

    def set_params_dict(self, params_dict, params, rm_config):
        for job_data in self.parent_job_data:
            if not job_data:
                # We should get data from every element in triggers.
                # If any job_data in parent_job_data is empty we should fail.
                raise TypeError
            if job_data.input_key in params_dict:
                logging.error(
                    "[{comp_name}] Got input_key '{input_key}' already filled in params_dict {params_dict}".format(
                        comp_name=rm_config.name,
                        input_key=job_data.input_key,
                        params_dict=params_dict,
                    ))
            params_dict[job_data.input_key] = job_data.data(params, self.this_job_name(rm_config.name))


class CompareTaskDataResource(jg_data.CompareJobData, Arrow):
    """ Puts in compare child job 'input_key' resource from field 'resource_name' in current job """
    def __init__(self, input_key, resource_name, override=False, task_number=1):
        """
        :param input_key:  str, name of parameter in compare child sandbox task
        :param resource_name: str, name of parameter in current sandbox task
        """
        jg_data.CompareJobData.__init__(self, input_key, override=override, task_number=task_number)
        self.resource_name = resource_name

    def set_params_dict(self, params_dict, params, rm_config):
        self._check_params_dict(self.input_key, params_dict, rm_config)
        params_dict[self.input_key] = self.data(params, rm_config)

    def data(self, params, parent_job_name):
        return getattr(params, "task{}Resources".format(self.task_number))[self.resource_name][0]


class CompareTaskDataId(jg_data.CompareJobData, Arrow):
    """ Puts in compare child job 'input_field' current job task id """

    def set_params_dict(self, params_dict, params, rm_config):
        self._check_params_dict(self.input_key, params_dict, rm_config)
        params_dict[self.input_key] = self.data(params, rm_config)

    def data(self, params, parent_job_name):
        return getattr(params, "task{}_id".format(self.task_number))


class CompareTaskDataCtx(jg_data.CompareParentDataOutput, Arrow):
    """ Puts in compare child job in field 'input_key' value 'output_key' from current job context """

    def set_params_dict(self, params_dict, params, rm_config):
        self._check_params_dict(self.input_key, params_dict, rm_config)
        params_dict[self.input_key] = self.data(params, rm_config)

    def data(self, params, parent_job_name):
        compare_task_ctx = getattr(params, "task{}Ctx".format(self.task_number))
        if self.output_key in compare_task_ctx:
            return self.transform(compare_task_ctx[self.output_key], params)
