# Here we define base classes for transferring parameters from one job to another

import typing  # noqa

from sandbox.projects.common.constants import constants as common_const
from sandbox.projects.release_machine.components.job_graph import utils as jg_utils


class JobData(object):
    """ Interface for getting data from parents """

    def __init__(self, input_key, override=False):
        # type: (typing.AnyStr, bool) -> typing.NoReturn
        """
        :param input_key: str, name of parameter in child sandbox task
        """
        self.input_key = input_key
        self.override = override

    def data(self, *args):
        raise NotImplementedError

    def set_input(self, params, *args):
        """
        Method that fills child job parameters with parent's values
        :param params: job params
        :param args: job args
        """
        value = self.data(params, *args)
        if value is not None:
            if self.input_key in params.custom_fields:
                if self.override:
                    params.custom_fields[self.input_key] = value
            else:
                params.custom_fields[self.input_key] = value

    def as_tasklet_input(self, *args):
        # type: () -> typing.Dict[typing.AnyStr, typing.Any]
        return {}


class CompareJobData(JobData):
    """ Interface for getting data from parents for compare task"""

    def __init__(self, input_key, override=False, task_number=1):
        """
        :param input_key: str, name of parameter in child sandbox task
        :param task_number: int, should be 1 or 2, uses for getting required property from params
        """
        super(CompareJobData, self).__init__(input_key, override)
        self.task_number = task_number

    def data(self, *args):
        raise NotImplementedError


class ParentDataId(JobData):
    """ Puts in child job 'input_field' parent job task id """

    def data(self, params, parent_job_name):
        return params.parent_test_task_id[parent_job_name]

    def as_tasklet_input(self, parent_job_name):
        # type: (typing.AnyStr) -> typing.Dict[typing.AnyStr, typing.Any]
        return {
            self.input_key: "${{(tasks.{}.resources)[0].task_id}}".format(jg_utils.job_name_for_ci(parent_job_name))
        }


class CompareParentDataId(CompareJobData):
    """ Puts in compare child job 'input_field' parent job task id """

    def data(self, params, parent_job_name):
        return getattr(params, "parentTest{}TaskId".format(self.task_number))[parent_job_name]


class ParentDataResource(JobData):
    """ Puts in child job 'input_key' resource from field 'resource_name' in parent job """

    def __init__(self, input_key, resource_name, override=False):
        """
        :param input_key:  str, name of parameter in child sandbox task
        :param resource_name: str, name of parameter in parent sandbox task
        """
        super(ParentDataResource, self).__init__(input_key, override=override)
        self.resource_name = resource_name

    def data(self, params, parent_job_name):
        return params.parent_test_resources[parent_job_name][self.resource_name][0]

    def as_tasklet_input(self, parent_job_name):
        return {self.input_key: "tasks.{}.resources[?type == '{}'])[0].id".format(
            jg_utils.job_name_for_ci(parent_job_name), self.resource_name
        )}


class CompareParentDataResource(CompareJobData):
    """ Puts in compare child job 'input_key' resource from field 'resource_name' in parent job """

    def __init__(self, input_key, resource_name, override=False, task_number=1):
        """
        :param input_key:  str, name of parameter in compare child sandbox task
        :param resource_name: str, name of parameter in parent sandbox task
        """
        super(CompareParentDataResource, self).__init__(input_key, override=override, task_number=task_number)
        self.resource_name = resource_name

    def data(self, params, parent_job_name):
        return getattr(params, "parentTest{}Resources".format(self.task_number))[parent_job_name][self.resource_name][0]


class ParentDataDict(JobData):
    """ Puts in child job dict with name 'input_key' and pairs where key is 'dict_key' and value is 'resource_name' """

    def __init__(self, input_key, dict_key, resource_name, override=False):
        """
        :param input_key: str, name of dict in child sandbox task
        :param dict_key: str, name of key in child sandbox task dict
        :param resource_name: str, name of parameter in parent sandbox task
        """
        super(ParentDataDict, self).__init__(input_key, override=override)
        self.resource_name = resource_name
        self.dict_key = dict_key

    def set_input(self, params, *args):
        params.custom_fields.setdefault(self.input_key, {})
        if self.dict_key in params.custom_fields[self.input_key]:
            if self.override:
                params.custom_fields[self.input_key][self.dict_key] = self.data(params, *args)
        else:
            params.custom_fields[self.input_key][self.dict_key] = self.data(params, *args)

    def data(self, params, parent_job_name):
        try:
            return params.parent_test_resources[parent_job_name][self.resource_name][0]
        except KeyError:
            if parent_job_name not in params.parent_test_resources:
                raise KeyError('parent_test_resources: missing {}'.format(parent_job_name))
            else:
                raise KeyError('Parent task {}: missing {} as an output resource'.format(
                    parent_job_name, self.resource_name
                ))

    def jmes_parent_resource_id(self, parent_job_name):
        return "${{(tasks.{}.resources[?type == \'{}\'])[0].id}}".format(
            jg_utils.job_name_for_ci(parent_job_name), self.resource_name
        )

    def as_tasklet_input(self, parent_job_name):
        jmes_parent_resource_id = self.jmes_parent_resource_id(parent_job_name)

        return {
            self.input_key: '{{"{}": {}}}'.format(self.dict_key, jmes_parent_resource_id)
        }

    @classmethod
    def as_tasklet_input_merged(cls, parent_job_name, parent_data_dict_list):

        result_lines = []

        for item in parent_data_dict_list:

            if not isinstance(item, cls):
                raise TypeError("Unexpected type for {}.as_tasklet_input_merged: expected {}, got {}".format(
                    cls.__name__,
                    cls.__name__,
                    type(item),
                ))

            result_lines.append('"{}": {}'.format(item.dict_key, item.jmes_parent_resource_id(parent_job_name)))

        return "{{{}}}".format(", ".join(result_lines))


class TasksArchiveParentDataOutput(JobData):
    """
    Puts in child job apiargs["tasks_archive_resource"] resource id from parent job custom_fields.
    Read discussion in DEVTOOLSSUPPORT-2043 for better understanding.
    """

    def __init__(self, tasks_archive_resource):
        """
        :param tasks_archive_resource: str, name of build resource in child sandbox task
        """
        super(TasksArchiveParentDataOutput, self).__init__("tasks_archive_resource")
        self.tasks_archive_resource = tasks_archive_resource

    def set_input(self, params, *args):
        params.apiargs["tasks_archive_resource"] = self.data(params, *args)

    def data(self, params, parent_job_name):
        if self.tasks_archive_resource in params.parent_test_resources[parent_job_name]:
            return params.parent_test_resources[parent_job_name][self.tasks_archive_resource][0]


class TaskletDataOutput(JobData):
    def __init__(self, input_key, jmespath):
        super(TaskletDataOutput, self).__init__(input_key)
        self._jmespath = jmespath

    def data(self, parent_job_name):
        return "${{tasks.{}.{}}}".format(
            jg_utils.job_name_for_ci(parent_job_name), self._jmespath
        )

    def as_tasklet_input(self, parent_job_name):
        return {self.input_key: self.data(parent_job_name)}


class ParentDataOutput(JobData):
    """ Puts in child job in field 'input_key' value 'output_key' from parent job custom_fields """

    def __init__(self, input_key, output_key, transform=lambda x, params: x, override=False):
        """
        :param input_key: str, name of parameter in child sandbox task
        :param output_key: str, name of parameter in parent sandbox task
        :param transform: function, that transforms parameter from parent job
        """
        super(ParentDataOutput, self).__init__(input_key, override=override)
        self.output_key = output_key
        self.transform = transform

    def data(self, params, parent_job_name):
        if self.output_key in params.parent_test_custom_fields[parent_job_name]:
            return self.transform(params.parent_test_custom_fields[parent_job_name][self.output_key], params)

    def as_tasklet_input(self, parent_job_name):
        if self.input_key == common_const.ARCADIA_URL_KEY:
            # this parameter is default in tasklets
            return {}
        return {self.input_key: "${{tasks.{}.output_params.{}}}".format(
            jg_utils.job_name_for_ci(parent_job_name), self.output_key
        )}


class CompareParentDataOutput(CompareJobData):
    """ Puts in compare child job in field 'input_key' value 'output_key' from parent job custom_fields """

    def __init__(self, input_key, output_key, transform=lambda x, params: x, override=False, task_number=1):
        """
        :param input_key: str, name of parameter in compare child sandbox task
        :param output_key: str, name of parameter in parent sandbox task
        :param transform: function, that transforms parameter from parent job
        """
        super(CompareParentDataOutput, self).__init__(input_key, override=override, task_number=task_number)
        self.output_key = output_key
        self.transform = transform

    def data(self, params, parent_job_name):
        parent_test_custom_fields = getattr(params, "parentTest{}CustomFields".format(self.task_number))
        if self.output_key in parent_test_custom_fields[parent_job_name]:
            return self.transform(parent_test_custom_fields[parent_job_name][self.output_key], params)


class ParentDataCtx(ParentDataOutput):
    """ Puts in child job in field 'input_key' value 'output_key' from parent job context """

    def data(self, params, parent_job_name):
        if self.output_key in params.parent_test_ctx[parent_job_name]:
            return self.transform(params.parent_test_ctx[parent_job_name][self.output_key], params)


class CompareParentDataCtx(CompareParentDataOutput):
    """ Puts in compare child job in field 'input_key' value 'output_key' from parent job context """

    def data(self, params, parent_job_name):
        parent_test_ctx = getattr(params, "parentTest{}Ctx".format(self.task_number))
        if self.output_key in parent_test_ctx[parent_job_name]:
            return self.transform(parent_test_ctx[parent_job_name][self.output_key], params)
