import typing  # noqa

import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri


class BasePreset(object):
    """
    Base class for presets
    """

    def __call__(self, job_graph_instance):
        """
        Return list with generated jobs

        :param job_graph_instance: JobGraph class instance.
        Needed for arrows from release to build job
        """
        return []


class SinglePreset(BasePreset):
    """
    Base class for simple presets, i.e. SinglePackagePreset and SingleBuildPreset
    """

    def __init__(
        self,
        build_task_name="YA_PACKAGE",
        build_ctx=None,
        build_apiargs=None,
        release_task_name="RELEASE_RM_COMPONENT_2",
        deploy_system=rm_const.DeploySystem.ya_deploy.name,
        release_ctx=None,
        stages=None,
        build_frequency=jg_utils.CheckEachCommit(),
        release_job_arrows_dict=None,  # a dict of stage => job_arrows for release job - additional arrows per stage
    ):
        self._build_task_name = build_task_name
        self._build_ctx = build_ctx
        self._build_apiargs = build_apiargs
        self._build_frequency = build_frequency
        self._release_task_name = release_task_name
        self._release_ctx = self._construct_release_ctx(release_ctx, deploy_system)
        self._stages = stages
        self._release_job_arrows_dict = release_job_arrows_dict

    @staticmethod
    def _output_resources(job_graph_instance):
        result = {}
        for releasable_item in job_graph_instance.releasable_items:
            if isinstance(releasable_item, ri.DynamicReleasableItem):  # scheduling case
                continue
            if isinstance(releasable_item.data, ri.SandboxResourceData):
                result[releasable_item.data.resource_type] = releasable_item.data.ttl
        return result

    @staticmethod
    def _construct_release_ctx(release_ctx, deploy_system_from_input):
        release_ctx = release_ctx or {}
        deploy_system_from_release_ctx = release_ctx.get("deploy_system")
        ds1 = bool(deploy_system_from_input)
        ds2 = bool(deploy_system_from_release_ctx)
        assert not (ds1 and ds2), "Deploy system should be defined either as parameter or as ctx value. Got both"
        assert ds1 or ds2, "Deploy system should be defined either as parameter or as ctx value. Got nothing"
        release_ctx["deploy_system"] = deploy_system_from_input or deploy_system_from_release_ctx
        return release_ctx

    def get_component_resources_data(self, releasable_items):
        """
        Prepare list with ParentDataDict entities for JobGraphElementReleaseBranched element
        :param releasable_items: releasable_items parameter from class Releases.
        """
        if self._release_ctx["deploy_system"] == rm_const.DeploySystem.ya_tool.name:
            return jg_job_data.ParentDataId("build_task_id")

        component_resources_data = []
        for releasable_item in releasable_items:
            if isinstance(releasable_item, ri.DynamicReleasableItem):  # scheduling case
                continue
            if isinstance(releasable_item.data, ri.SandboxResourceData):
                component_resources_data.append(
                    jg_job_data.ParentDataDict(
                        input_key="component_resources",
                        dict_key=releasable_item.name,
                        resource_name=releasable_item.data.resource_type,
                    )
                )
            else:
                # TODO: Add action for DockerImageData
                continue

        return component_resources_data

    def build_job(self, job_graph_instance):
        """
        JobGraphElement for build, i.e. JobGraphElementYaMakeBuildBranched or JobGraphElementBuildPackageBranched
        """
        raise NotImplementedError

    def release_job(self, stage, job_graph_instance):
        # type: (jg_utils.StageReleaseFrequency, typing.Any) -> jg_release.JobGraphElementReleaseBase
        raise NotImplementedError

    def release_action(self, stage):
        # type: (jg_utils.StageReleaseFrequency) -> typing.Optional
        return None

    def __call__(self, job_graph_instance):
        """
        Return list with generated jobs

        :param job_graph_instance: JobGraph class instance.
        Needed for arrows from release to build job
        """
        preset_jobs = [self.build_job(job_graph_instance)]
        for stage in self._stages:
            preset_jobs.append(self.release_job(stage, job_graph_instance))
            release_action = self.release_action(stage)
            if release_action:
                preset_jobs.append(release_action)
        return preset_jobs


# Releases


class SingleReleasePreset(SinglePreset):
    def release_job(self, stage, job_graph_instance):
        return jg_release.JobGraphElementReleaseBranched(
            task_name=self._release_task_name,
            release_to=stage.release_stage,
            job_params={
                "ctx": self._release_ctx,
            },
            job_arrows=[
                jg_job_triggers.JobTriggerBuild(
                    parent_job_data=self.get_component_resources_data(
                        releasable_items=job_graph_instance.releasable_items,
                    ),
                ),
                jg_job_triggers.JobTriggerNewTag(
                    parent_job_data=[
                        jg_job_data.ParentDataOutput("major_release_num", "branch_number_for_tag"),
                        jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
                    ],
                ),
            ],
        )

    def release_action(self, stage):
        return jg_release.JobGraphElementActionReleaseBranched(
            job_params=stage.frequency_type.get_freq_params(),
            release_to=stage.release_stage,
        )


class SinglePackagePreset(SingleReleasePreset):
    """
    Contains one package job (default is YaPackage) + release jobs for different stages i.e. stable, testing.
    """

    def __init__(
        self,
        build_task_name="YA_PACKAGE",
        package_names=None,
        use_compression=True,
        build_ctx=None,
        build_apiargs=None,
        release_task_name="RELEASE_RM_COMPONENT_2",
        deploy_system=rm_const.DeploySystem.ya_deploy.name,
        release_ctx=None,
        stages=None,
        build_frequency=jg_utils.CheckEachCommit(),
    ):
        """
        :param build_task_name: task name for build job
        :param package_names: string with paths to packages joined with ';'
        :param use_compression: enable flag 'compress_package_archive' in build task
        :param build_ctx: context parameters for build job
        :param build_apiargs: apiargs parameters for build job
        :param release_task_name: task name for release job
        :param deploy_system: where should service deploy, etc. ya_deploy, nanny, qloud.
        :param release_ctx: context parameters for release jobs
        :param stages: list StageReleaseFrequency objects
        """
        super(SinglePackagePreset, self).__init__(
            build_task_name=build_task_name,
            build_ctx=build_ctx,
            build_apiargs=build_apiargs,
            release_task_name=release_task_name,
            deploy_system=deploy_system,
            release_ctx=release_ctx,
            stages=stages,
            build_frequency=build_frequency,
        )
        self._package_names = package_names
        self._use_compression = use_compression

    def build_job(self, job_graph_instance):
        job_params = dict(self._build_frequency.get_freq_params(), apiargs=self._build_apiargs)
        return jg_build.JobGraphElementBuildPackageBranched(
            resource_names=";".join([i.data.resource_type for i in job_graph_instance.releasable_items]),
            task_name=self._build_task_name,
            job_params=job_params,
            ctx=self._build_ctx,
            out=self._output_resources(job_graph_instance),
            package_names=self._package_names,
            use_compression=self._use_compression,
        )


class SingleBuildPreset(SingleReleasePreset):
    """
    Contains one build job + release jobs for different stages i.e. stable, testing.
    """

    def __init__(
        self,
        build_task_name="KOSHER_YA_MAKE",
        build_ctx=None,
        build_apiargs=None,
        release_task_name="RELEASE_RM_COMPONENT_2",
        deploy_system=rm_const.DeploySystem.ya_deploy.name,
        release_ctx=None,
        stages=None,
        build_frequency=jg_utils.CheckEachCommit(),
    ):
        """
        :param build_task_name: task name for build job
        :param build_ctx: context parameters for build job
        :param build_apiargs: apiargs parameters for build job
        :param release_task_name: task name for release job
        :param deploy_system: where should service deploy, etc. ya_deploy, nanny, qloud.
        :param release_ctx: context parameters for release jobs
        :param stages: list StageReleaseFrequency objects
        """
        super(SingleBuildPreset, self).__init__(
            build_task_name=build_task_name,
            build_ctx=build_ctx,
            build_apiargs=build_apiargs,
            release_task_name=release_task_name,
            deploy_system=deploy_system,
            release_ctx=release_ctx,
            stages=stages,
            build_frequency=build_frequency,
        )

    def build_job(self, job_graph_instance):
        job_params = dict(self._build_frequency.get_freq_params(), apiargs=self._build_apiargs)
        return jg_build.JobGraphElementYaMakeBuildBranched(
            task_name=self._build_task_name,
            job_params=job_params,
            ctx=self._build_ctx,
            out=self._output_resources(job_graph_instance),
        )


# Scheduling via Morty


class SingleSchedulePreset(SinglePreset):
    def release_job(self, stage, job_graph_instance):
        return jg_release.JobGraphElementScheduleRelease(
            release_to=stage.release_stage,
            job_params={
                "ctx": self._release_ctx,
            },
            job_arrows=jg_job_triggers.JobTriggerBuild(
                parent_job_data=self.get_component_resources_data(
                    releasable_items=job_graph_instance.releasable_items,
                ),
            ),
        )

    def release_action(self, stage):
        return jg_release.JobGraphElementActionScheduleBranched(
            release_to=stage.release_stage,
            job_params=stage.frequency_type.get_freq_params(),
        )


class SinglePackageSchedulePreset(SingleSchedulePreset):
    """
    Contains one package job (default is YaPackage) + scheduling jobs for different stages i.e. stable, testing.
    """

    def __init__(
        self,
        build_task_name="YA_PACKAGE",
        package_names=None,
        use_compression=True,
        build_ctx=None,
        build_apiargs=None,
        deploy_system=rm_const.DeploySystem.nanny_push.name,
        release_ctx=None,
        stages=(jg_utils.StageReleaseFrequency(rm_const.ReleaseStatus.stable), ),
        build_frequency=jg_utils.CheckEachCommit(),
    ):
        super(SingleSchedulePreset, self).__init__(
            build_task_name=build_task_name,
            build_ctx=build_ctx,
            build_apiargs=build_apiargs,
            deploy_system=deploy_system,
            release_ctx=release_ctx,
            stages=stages,
            build_frequency=build_frequency,
        )
        self._package_names = package_names
        self._use_compression = use_compression

    def build_job(self, job_graph_instance):
        job_params = dict(self._build_frequency.get_freq_params(), apiargs=self._build_apiargs)
        return jg_build.JobGraphElementBuildPackageBranched(
            resource_names=";".join([i.data.resource_type for i in job_graph_instance.releasable_items]),
            task_name=self._build_task_name,
            job_params=job_params,
            ctx=self._build_ctx,
            out=self._output_resources(job_graph_instance),
            package_names=self._package_names,
            use_compression=self._use_compression,
        )
