import yaml
import typing
import collections

from sandbox.projects.release_machine.components.job_graph import utils as jg_utils
from sandbox.projects.release_machine.components.job_graph import job_arrows as jg_arrows  # noqa: UnusedImport
from sandbox.projects.release_machine.components.job_graph import job_data as jg_data  # noqa: UnusedImport
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.common.string import shift_right


class JobGraphElement(object):

    JOB_NAME_KEY = "job_name"
    JOB_TYPE_KEY = "job_type"
    JOB_NAME_PARAMETER_KEY = "job_name_parameter"

    """ Class that describes on job class in job_graph """
    def __init__(self, job_params, job_arrows=()):
        # type: (dict, typing.Union[typing.Sequence[jg_arrows.Arrow], jg_arrows.Arrow]) -> typing.NoReturn
        """
        :param job_params: dict, contains all input_params, must contain key 'job_type'
        :param job_arrows: list of objects with type JobTrigger
        """
        self.job_params = job_params or {}
        self.job_arrows = jg_utils.tuplify(job_arrows)  # type: typing.Tuple[jg_arrows.Arrow]

    def construct_job_name(self, component_name):

        if self.JOB_NAME_KEY not in self.job_params.keys():

            self.job_params[self.JOB_NAME_KEY] = rm_const.JobTypes.rm_job_name(
                self.job_type,
                component_name,
                self.job_name_parameter,
            )

        return self.job_params[self.JOB_NAME_KEY]

    @property
    def job_type(self):
        return self.job_params[self.JOB_TYPE_KEY]

    @property
    def job_name_parameter(self):
        return self.job_params.get(self.JOB_NAME_PARAMETER_KEY, "")

    # params for a.yaml:

    def ci_job_input_dict(self, component_name):

        ci_job_input = {"component_name": component_name}

        job_name = self.construct_job_name(component_name)

        job_params_ctx = self.job_params.get("ctx")

        if job_params_ctx:
            ci_job_input.update(job_params_ctx)

        for arrow in self.job_arrows:

            if not isinstance(arrow, jg_arrows.JobTrigger):
                continue

            arrow_dict_data = collections.defaultdict(list)

            for job_arrow_data in arrow.parent_job_data:

                input_part = job_arrow_data.as_tasklet_input(arrow.this_job_name(component_name))

                if isinstance(job_arrow_data, jg_data.ParentDataDict):
                    arrow_dict_data[job_arrow_data.input_key].append(job_arrow_data)
                    continue

                override_keys = set(input_part.keys()) & set(ci_job_input.keys())

                assert not override_keys, (
                    "Override keys found for {}: {}\n"
                    "- Input part (job arrows): {}\n"
                    "- Current CI: {}\n".format(
                        job_name,
                        override_keys,
                        input_part,
                        ci_job_input,
                    )
                )

                ci_job_input.update(input_part)

            for dict_arrow_item_key in arrow_dict_data:

                ci_job_input[dict_arrow_item_key] = jg_data.ParentDataDict.as_tasklet_input_merged(
                    arrow.this_job_name(component_name), parent_data_dict_list=arrow_dict_data[dict_arrow_item_key],
                )

        return ci_job_input

    def ci_job_input_as_yaml(self, component_name, indent=0):
        return shift_right(yaml.dump(self.ci_job_input_dict(component_name)), indent)

    def ci_job_name(self, component_name):
        return jg_utils.job_name_for_ci(self.construct_job_name(component_name))

    def ci_requirements_dict(self):
        r = {}
        as_is = frozenset(["ram", "cores", "tmpfs"])
        sandbox_requirements = frozenset([
            "client_tags", "container_resource", "cpu_model", "dns", "host", "platform", "privileged", "priority",
        ])
        for requirement_key, requirement_val in self.job_params.get("apiargs", {}).get("requirements", {}).items():
            if requirement_key == "disk_space":
                r["disk"] = requirement_val
            elif requirement_key in sandbox_requirements:
                r.setdefault("sandbox", {})[requirement_key] = requirement_val
            elif requirement_key in as_is:
                r[requirement_key] = requirement_val
            else:
                raise KeyError("Unknown requirement key: {}".format(requirement_key))

        return r

    def ci_requirements(self, indent=0):
        r = self.ci_requirements_dict()
        return shift_right(yaml.dump({"requirements": r}), indent) if r else ""

    def ci_manual(self):
        frequency = self.job_params.get("frequency")
        if isinstance(frequency, dict):
            frequency = frequency["branch"]
        if frequency:
            if frequency[0] == jg_utils.TestFrequency.LAZY:
                return True
            elif frequency[0] == jg_utils.TestFrequency.CHECK_EACH_COMMIT:
                return False
            else:
                raise ValueError("Ci does not support this job frequency: {}".format(frequency))
        return False

    @property
    def task_ci_registry_location(self):
        return rm_const.TASK_CI_REG_LOCATIONS[self.job_params["task_name"]]


class JobGraphElementTasklet(JobGraphElement):
    def ci_job_input_dict(self, component_name, indent=0):
        config = super(JobGraphElementTasklet, self).ci_job_input_dict(component_name)
        tasklet_input = {"config": config}
        return tasklet_input
