import sandbox.projects.release_machine.components.job_graph.stages.job_graph_element as jg_element
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.core.const as rm_const


class JobGraphElementNewBranch(jg_element.JobGraphElement):
    def __init__(self, job_params=None, job_arrows=(), branching_mode=rm_const.BranchingMode.TRUNK_TO_BRANCH):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.NEW_BRANCH,
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                "task_name": "CREATE_BRANCH_OR_TAG",
                "ctx": {
                    "tag_or_branch_mode": branching_mode,
                },
            },
        )
        super(JobGraphElementNewBranch, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("revision_for_trunk", lambda params, rm_config: params.revision),
        )


class JobGraphElementPreliminaryChangelogBase(jg_element.JobGraphElement):
    """
    Base class for preliminary changelog jobs of all components.
    Preliminary means created for branch first revision without considering merges to branch
    """
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.CHANGELOG,
                "task_name": "RELEASE_MACHINE_CHANGELOG",
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                "ctx": ctx,
                "out": {"RELEASE_MACHINE_CHANGELOG": 720},
                "apiargs": {"requirements": {"disk_space": 2 * (1024 ** 3)}},  # 2 Gb
            },
        )
        super(JobGraphElementPreliminaryChangelogBase, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementPreliminaryChangelogBranched(JobGraphElementPreliminaryChangelogBase):
    """ Element for preliminary changelog job of branched components """
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        super(JobGraphElementPreliminaryChangelogBranched, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            ctx=ctx,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerNewBranch([
                jg_job_data.ParentDataOutput("major_release_num", "new_branch_number"),
                jg_job_data.ParentDataCtx("candidate_revision", "result_revision"),
                jg_job_data.ParentDataCtx("candidate_path", "result_path"),
            ]),
        )


class JobGraphElementPreliminaryChangelogTagged(JobGraphElementPreliminaryChangelogBase):
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        super(JobGraphElementPreliminaryChangelogTagged, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            ctx=ctx,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerNewTag([
                jg_job_data.ParentDataOutput("major_release_num", "new_tag_number"),
                jg_job_data.ParentDataCtx("candidate_revision", "result_revision"),
                jg_job_data.ParentDataCtx("candidate_path", "result_path"),
            ]),
        )


class JobGraphElementCloneDb(jg_element.JobGraphElement):
    def __init__(self, job_params=None, job_arrows=()):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.CLONE_DB,
                "task_name": "CLONE_TESTENV_DB",
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
            },
        )
        super(JobGraphElementCloneDb, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerNewBranch([
                jg_job_data.ParentDataCtx("branch_path", "result_path"),
                jg_job_data.ParentDataCtx("start_revision", "result_revision"),
            ]),
        )


class JobGraphElementStartrekBase(jg_element.JobGraphElement):
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        def get_additional_info(params, rm_config):
            db_name = "Intermediate db name: {}".format(params.db.db_name or "") if params.db.db_name else ""
            return params.custom_fields.get("additional_info", "") + db_name

        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.STARTREK,
                "task_name": "CREATE_STARTREK_TICKET",
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                "ctx": ctx,
                "apiargs": {"requirements": {"disk_space": 2 * (1024 ** 3)}},  # in Bytes
            },
        )
        super(JobGraphElementStartrekBase, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("additional_info", get_additional_info),
        )


class JobGraphElementStartrek(JobGraphElementStartrekBase):
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        super(JobGraphElementStartrek, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            ctx=ctx,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerChangelog([
                jg_job_data.ParentDataResource("changelog", "RELEASE_MACHINE_CHANGELOG"),
            ]),
        )


class JobGraphElementWikiBase(jg_element.JobGraphElement):
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.WIKI,
                "task_name": "CREATE_WIKI_PAGE",
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                "ctx": ctx,
                "apiargs": {"requirements": {"disk_space": 2 * (1024 ** 3)}},  # in Bytes
            },
        )
        super(JobGraphElementWikiBase, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementWikiNoStartrek(JobGraphElementWikiBase):
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        super(JobGraphElementWikiNoStartrek, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            ctx=ctx,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerChangelog([
                jg_job_data.ParentDataResource("changelog", "RELEASE_MACHINE_CHANGELOG"),
            ]),
        )


class JobGraphElementWiki(JobGraphElementWikiNoStartrek):
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        super(JobGraphElementWiki, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            ctx=ctx,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerStartrek(),
        )


class JobGraphElementActionPreReleaseBase(jg_element.JobGraphElement):
    def __init__(
        self, job_params=None, job_arrows=(), filter_targets=None, next_revision_custom_schedule=None, use_arc=False
    ):
        frequency = (jg_utils.TestFrequency.LAZY, None)
        if next_revision_custom_schedule:
            frequency = (jg_utils.TestFrequency.DEFINED_BY_CODE, None)

        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.ACTION_PRE_RELEASE,
                "task_name": "CHECK_TASKS_CORRECTNESS",
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
                "frequency": frequency,
                "filter_targets": filter_targets,
                "next_revision_custom_schedule": next_revision_custom_schedule,
                "observed_paths": ["arcadia"] if use_arc else [],
            },
        )
        super(JobGraphElementActionPreReleaseBase, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("tasks_to_check", jg_utils.get_tasks_to_check),
            jg_job_triggers.JobTriggerCloneDb(),
        )


class JobGraphElementActionPreReleaseWiki(JobGraphElementActionPreReleaseBase):
    def __init__(
        self, job_params=None, job_arrows=(), filter_targets=None, next_revision_custom_schedule=None, use_arc=False,
    ):
        super(JobGraphElementActionPreReleaseWiki, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            filter_targets=filter_targets,
            next_revision_custom_schedule=next_revision_custom_schedule,
            use_arc=use_arc,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerWiki(),
        )


class JobGraphElementActionPreReleaseStartrek(JobGraphElementActionPreReleaseBase):
    def __init__(
        self, job_params=None, job_arrows=(), filter_targets=None, next_revision_custom_schedule=None, use_arc=False,
    ):
        super(JobGraphElementActionPreReleaseStartrek, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            filter_targets=filter_targets,
            next_revision_custom_schedule=next_revision_custom_schedule,
            use_arc=use_arc,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerStartrek(),
        )


class JobGraphElementActionPreReleaseStartrekWiki(JobGraphElementActionPreReleaseBase):
    def __init__(
        self, job_params=None, job_arrows=(), filter_targets=None, next_revision_custom_schedule=None, use_arc=False
    ):
        super(JobGraphElementActionPreReleaseStartrekWiki, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            filter_targets=filter_targets,
            next_revision_custom_schedule=next_revision_custom_schedule,
            use_arc=use_arc,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerWiki(),
        )
