import sandbox.projects.release_machine.components.job_graph.stages.job_graph_element as jg_element
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.core.const as rm_const


class JobGraphElementNewTag(jg_element.JobGraphElement):
    def __init__(
        self,
        job_params=None,
        job_arrows=(),
        branching_mode=rm_const.BranchingMode.BRANCH_TO_TAG,
        wait_for_release_ticket=False,
    ):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.NEW_TAG,
                "task_name": "CREATE_BRANCH_OR_TAG",
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
                "ctx": {
                    "tag_or_branch_mode": branching_mode,
                    "wait_for_release_ticket": wait_for_release_ticket,
                },
            },
        )
        super(JobGraphElementNewTag, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("revision_for_trunk", lambda params, rm_config: params.revision),
            jg_arrows.ParamsData("branch_number_for_tag", jg_utils.get_major_release_number),
        )


class JobGraphElementNewTagTagged(JobGraphElementNewTag):
    def __init__(self, job_params=None, job_arrows=(), branching_mode=rm_const.BranchingMode.TRUNK_TO_TAG):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
            },
        )
        super(JobGraphElementNewTagTagged, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
            branching_mode=branching_mode,
        )


class JobGraphElementChangelogFinal(jg_element.JobGraphElement):
    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.CHANGELOG_FINAL,
                "task_name": "RELEASE_MACHINE_CHANGELOG",
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                "ctx": ctx,
                "out": {"RELEASE_MACHINE_CHANGELOG": 720},
            },
        )
        super(JobGraphElementChangelogFinal, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("major_release_num", jg_utils.get_major_release_number),
            jg_job_triggers.JobTriggerNewTag([
                jg_job_data.ParentDataOutput("candidate_revision", "result_revision"),
                jg_job_data.ParentDataOutput("candidate_path", "result_path"),
                jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
            ]),
        )


class JobGraphElementWikiFinal(jg_element.JobGraphElement):
    def __init__(self, job_params=None, job_arrows=()):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.WIKI_FINAL,
                "task_name": "CREATE_WIKI_PAGE",
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
                "apiargs": {"requirements": {"disk_space": 2 * (1024 ** 3)}},  # in Bytes
            },
        )
        super(JobGraphElementWikiFinal, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerChangelogFinal([
                jg_job_data.ParentDataResource("changelog", "RELEASE_MACHINE_CHANGELOG"),
            ]),
        )


class JobGraphElementScheduleRelease(jg_element.JobGraphElement):
    def __init__(self, release_to, job_params=None, job_arrows=()):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.SCHEDULE_RELEASE,
                "task_name": "SCHEDULE_RELEASE",
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
                "ctx": {"where_to_release": release_to},
            },
        )
        if not job_params or job_params.get("ctx", {}).get("schedule_mode") != "just_schedule":
            merged_job_params = jg_utils.merge_job_params(
                merged_job_params,
                {"out": {"SCHEDULED_RM_RELEASE_DATA": 360}}
            )
        super(JobGraphElementScheduleRelease, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerNewTag([
                jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
            ]),
        )


class JobGraphElementActionReleaseBase(jg_element.JobGraphElement):
    def __init__(self, job_name_parameter, job_params=None, job_arrows=()):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.ACTION_RELEASE,
                "job_name_parameter": job_name_parameter,
                "task_name": "CHECK_TASKS_CORRECTNESS",
            },
        )
        super(JobGraphElementActionReleaseBase, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("tasks_to_check", jg_utils.get_tasks_to_check),
        )


class JobGraphElementActionScheduleBranched(JobGraphElementActionReleaseBase):

    JOB_NAME_PARAMETER_SUFFIX = "MORTY"

    def __init__(self, release_to, job_params=None, job_arrows=(), release_item=""):

        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
            },
        )

        super(JobGraphElementActionScheduleBranched, self).__init__(
            job_name_parameter=self.get_job_name_parameter(release_item, release_to),
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )

        if release_to == rm_const.ReleaseStatus.stable:
            self.job_arrows += (
                jg_job_triggers.JobTriggerScheduleRelease(),
                jg_job_triggers.JobTriggerChangelogFinal(),
            )

    def get_job_name_parameter(self, release_item, release_to):
        job_name_parameter_prefix = "{}__{}".format(release_item, release_to) if release_item else release_to
        return "{}__{}".format(job_name_parameter_prefix, self.JOB_NAME_PARAMETER_SUFFIX)


class JobGraphElementActionSingleRelease(JobGraphElementActionReleaseBase):
    def __init__(self, job_name_parameter, job_params=None, job_arrows=()):
        super(JobGraphElementActionSingleRelease, self).__init__(
            job_name_parameter=job_name_parameter,
            job_params=job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerRelease(job_name_parameter=job_name_parameter),
        )


class JobGraphElementActionReleaseBaseBranched(JobGraphElementActionSingleRelease):
    def __init__(self, release_to, job_name_parameter, job_params=None, job_arrows=()):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
            },
        )
        super(JobGraphElementActionReleaseBaseBranched, self).__init__(
            job_name_parameter=job_name_parameter,
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        if release_to == rm_const.ReleaseStatus.stable:
            self.job_arrows += (jg_job_triggers.JobTriggerChangelogFinal(),)


class JobGraphElementActionReleaseBranched(JobGraphElementActionReleaseBaseBranched):
    def __init__(self, release_to, job_params=None, job_arrows=(), release_item=""):
        job_name_parameter = "{}__{}".format(release_item, release_to) if release_item else release_to
        super(JobGraphElementActionReleaseBranched, self).__init__(
            release_to=release_to,
            job_name_parameter=job_name_parameter,
            job_params=job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementActionPushReleaseBranched(JobGraphElementActionReleaseBaseBranched):
    def __init__(self, release_to, job_params=None, job_arrows=(), release_item=""):
        job_name_parameter = "{}__{}".format(release_item, release_to) if release_item else release_to
        job_name_parameter = "{}__{}".format(job_name_parameter, "PUSH")
        super(JobGraphElementActionPushReleaseBranched, self).__init__(
            release_to=release_to,
            job_name_parameter=job_name_parameter,
            job_params=job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementActionReleaseBaseTagged(JobGraphElementActionSingleRelease):
    def __init__(self, release_to, job_name_parameter, job_params=None, job_arrows=()):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
            },
        )
        super(JobGraphElementActionReleaseBaseTagged, self).__init__(
            job_name_parameter=job_name_parameter,
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        if release_to == rm_const.ReleaseStatus.stable:
            self.job_arrows += (jg_job_triggers.JobTriggerChangelog(),)


class JobGraphElementActionReleaseTagged(JobGraphElementActionReleaseBaseTagged):
    def __init__(self, release_to, job_params=None, job_arrows=(), release_item=""):
        job_name_parameter = "{}__{}".format(release_item, release_to) if release_item else release_to
        super(JobGraphElementActionReleaseTagged, self).__init__(
            release_to=release_to,
            job_name_parameter=job_name_parameter,
            job_params=job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementActionPushReleaseTagged(JobGraphElementActionReleaseBaseTagged):
    def __init__(self, release_to, job_params=None, job_arrows=(), release_item=""):
        job_name_parameter = "{}__{}".format(release_item, release_to) if release_item else release_to
        job_name_parameter = "{}__{}".format(job_name_parameter, "PUSH")
        super(JobGraphElementActionPushReleaseTagged, self).__init__(
            release_to=release_to,
            job_name_parameter=job_name_parameter,
            job_params=job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementReleaseBase(jg_element.JobGraphElement):
    def __init__(
        self,
        release_to,
        job_params=None,
        job_arrows=(),
        release_item="",
        wait_for_deploy=False,
        task_name="RELEASE_RM_COMPONENT",
    ):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.RELEASE,
                "job_name_parameter": "{}__{}".format(release_item, release_to) if release_item else release_to,
                "task_name": task_name,
                "ctx": {
                    "where_to_release": release_to,
                    "wait_for_deploy": wait_for_deploy,
                    "release_item_name": release_item,
                },
            },
        )
        super(JobGraphElementReleaseBase, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementReleaseBranched(JobGraphElementReleaseBase):
    def __init__(
        self,
        release_to,
        job_params=None,
        job_arrows=(),
        release_item="",
        wait_for_deploy=False,
        task_name="RELEASE_RM_COMPONENT",
    ):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
            },
        )
        super(JobGraphElementReleaseBranched, self).__init__(
            release_to=release_to,
            job_params=merged_job_params,
            job_arrows=job_arrows,
            release_item=release_item,
            wait_for_deploy=wait_for_deploy,
            task_name=task_name,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("release_number", jg_utils.get_major_release_number),
        )


class JobGraphElementReleaseCI(jg_element.JobGraphElement):
    """For tasklets"""
    def __init__(
        self,
        release_to,
        job_params=None,
        job_arrows=(),
        release_item="",
        wait_for_deploy=False,
        task_name="RELEASE_RM_COMPONENT_2",
    ):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.RELEASE,
                "job_name_parameter": "{}__{}".format(release_item, release_to) if release_item else release_to,
                "task_name": task_name,
                "ctx": {
                    "where_to_release": release_to,
                    "wait_for_deploy": wait_for_deploy,
                    "major_release_num": "${context.version_info.major}",
                    "minor_release_num": "${not_null(context.version_info.minor, `0`)}",
                },
            },
        )
        super(JobGraphElementReleaseCI, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementReleaseTagged(JobGraphElementReleaseBase):
    def __init__(
        self,
        release_to,
        job_params=None,
        job_arrows=(),
        release_item="",
        wait_for_deploy=False,
        task_name="RELEASE_RM_COMPONENT",
    ):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "should_add_to_db": jg_utils.should_add_to_db_trunk,
            },
        )
        super(JobGraphElementReleaseTagged, self).__init__(
            release_to=release_to,
            job_params=merged_job_params,
            job_arrows=job_arrows,
            release_item=release_item,
            wait_for_deploy=wait_for_deploy,
            task_name=task_name,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerNewTag(
                parent_job_data=[
                    jg_job_data.ParentDataOutput(
                        input_key="release_number",
                        output_key="new_tag_number",
                    )
                ]
            ),
        )


class JobGraphElementReleasePushBranched(JobGraphElementReleaseBranched):
    def __init__(
        self,
        release_to,
        job_params=None,
        job_arrows=(),
        release_item="",
        wait_for_deploy=False,
        task_name="RELEASE_RM_COMPONENT",
    ):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_name_parameter": "{}__{}".format(
                    "{}__{}".format(release_item, release_to) if release_item else release_to,
                    "PUSH",
                ),
                "ctx": {
                    "urgent_release": True,
                }
            },
        )
        super(JobGraphElementReleasePushBranched, self).__init__(
            release_to=release_to,
            job_params=merged_job_params,
            job_arrows=job_arrows,
            release_item=release_item,
            wait_for_deploy=wait_for_deploy,
            task_name=task_name,
        )


class JobGraphElementLogMerge(jg_element.JobGraphElement):
    def __init__(self, job_params=None, job_arrows=()):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.LOG_MERGE,
                "task_name": "LOG_MERGES_IN_STARTREK",
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
            },
        )
        super(JobGraphElementLogMerge, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("release_number", jg_utils.get_major_release_number),
            jg_arrows.ParamsData("merge_revision", lambda params, rm_config: params.revision),
        )


class JobGraphElementStartrekOkApprovementBase(jg_element.JobGraphElement):
    """
    https://wiki.yandex-team.ru/releasemachine/blokirovka-relizov/#blokirovkadook-ovotzadannyxsotrudnikov
    """

    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "job_type": rm_const.JobTypes.STARTREK_OK_APPROVEMENT,
                "task_name": "CREATE_STARTREK_OK_APPROVEMENT",
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                "ctx": ctx,
            },
        )
        super(JobGraphElementStartrekOkApprovementBase, self).__init__(
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )


class JobGraphElementStartrekOkApprovement(JobGraphElementStartrekOkApprovementBase):
    """
    https://wiki.yandex-team.ru/releasemachine/blokirovka-relizov/#blokirovkadook-ovotzadannyxsotrudnikov
    """

    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        super(JobGraphElementStartrekOkApprovement, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            ctx=ctx,
        )
        self.job_arrows += (
            jg_arrows.ParamsData("major_release_num", jg_utils.get_major_release_number),
            jg_job_triggers.JobTriggerNewTag([
                jg_job_data.ParentDataOutput("minor_release_num", "new_tag_number"),
            ]),
        )


class JobGraphElementStartrekOkApprovementTagged(JobGraphElementStartrekOkApprovementBase):
    """
    https://wiki.yandex-team.ru/releasemachine/blokirovka-relizov/#blokirovkadook-ovotzadannyxsotrudnikov
    """

    def __init__(self, job_params=None, job_arrows=(), ctx=None):
        super(JobGraphElementStartrekOkApprovementTagged, self).__init__(
            job_params=job_params,
            job_arrows=job_arrows,
            ctx=ctx,
        )
        self.job_arrows += (
            jg_job_triggers.JobTriggerNewTag([
                jg_job_data.ParentDataOutput("major_release_num", "new_tag_number"),
            ]),
        )
