import re
import enum


class Enum(enum.Enum):
    """
    Don't use IntEnum as it does not allow to customize JSON encoding.
    """
    __test__ = False  # prevent nose from treating enums as tests

    def __lt__(self, other):
        return self.value < other.value

    def __json__(self):
        """Used in smartgrid.get_json."""
        return self.value


class TestFrequency(Enum):
    CHECK_EACH_COMMIT = 1
    RUN_IF_DELAY_N_MINUTES = 2
    LAZY = 3
    EVERY_N_COMMIT = 4
    RUN_N_MINUTES_AFTER_LAST_RUN = 5
    DEFINED_BY_CODE = 6


class BaseFrequency(object):
    def get_freq_params(self):
        raise


class CheckEachCommit(BaseFrequency):
    def get_freq_params(self):
        return {
            "frequency": (TestFrequency.CHECK_EACH_COMMIT, None),
        }


class RunIfDelayNMinutes(BaseFrequency):
    def __init__(self, minutes_count):
        self._minutes_count = minutes_count

    def get_freq_params(self):
        return {
            "frequency": (TestFrequency.RUN_IF_DELAY_N_MINUTES, self._minutes_count),
        }


class Lazy(BaseFrequency):
    def get_freq_params(self):
        return {
            "frequency": (TestFrequency.LAZY, None),
        }


class EveryNCommit(BaseFrequency):
    def __init__(self, commits_count):
        self._commits_count = commits_count

    def get_freq_params(self):
        return {
            "frequency": (TestFrequency.EVERY_N_COMMIT, self._commits_count),
        }


CIAuto = CheckEachCommit
CIManual = Lazy


class RunNMinutesAfterLastRun(BaseFrequency):
    def __init__(self, minutes_count):
        self._minutes_count = minutes_count

    def get_freq_params(self):
        return {
            "frequency": (TestFrequency.RUN_N_MINUTES_AFTER_LAST_RUN, self._minutes_count),
        }


class CustomFrequency(BaseFrequency):
    def __init__(self, custom_schedule):
        self._custom_schedule = custom_schedule

    def get_freq_params(self):
        return {
            "frequency": (TestFrequency.DEFINED_BY_CODE, None),
            "next_revision_custom_schedule": self._custom_schedule,
        }


class StageReleaseFrequency(object):
    """
    Needed for preset object filling.
    """

    def __init__(self, release_stage, frequency_type=None):
        """
        :param release_stage: release stage, i.e. stable, testing.
        :param frequency_type: one of BaseFrequency descendants or None.
        """
        self.release_stage = release_stage
        self.frequency_type = frequency_type or Lazy()


class CustomSchedule(object):
    # Copy of CustomSchedule class from
    # https://a.yandex-team.ru/arc/trunk/arcadia/testenv/core/engine/test.py?rev=5668663#L491
    def __init__(
            self,
            time_interval_list=None,
            weekday_list=None,
            seconds_from_last_run_commit=None,
            seconds_from_last_run_start_time=None,
            ignored_jobs=None,
            last_run_finished=None,
            seconds_from_last_run_commit_if_run_not_finished=None,
            once_a_day=None,
    ):
        self.time_interval_list = time_interval_list
        self.weekday_list = weekday_list
        self.seconds_from_last_run_commit = seconds_from_last_run_commit
        self.seconds_from_last_run_start_time = seconds_from_last_run_start_time
        self.ignored_jobs = ignored_jobs
        self.last_run_finished = last_run_finished
        self.seconds_from_last_run_commit_if_run_not_finished = seconds_from_last_run_commit_if_run_not_finished
        self.once_a_day = once_a_day
    # time_interval_list: pairs of time intervals (in hours from range(24))
    # weekday_list: weekdays (Monday is 0, ..., Sunday is 6)
    # seconds_from_last_run_commit: time between runs (from last commit)
    # seconds_from_last_run_start_time: time between runs (from last run)
    # ignored_jobs: optional parameter of function params.get_last_run_start_time(ignored_jobs)
    # last_run_finished: run if last run finished
    # seconds_from_last_run_commit_if_run_not_finished: time between runs (from last commit) if last run not finished
    # once_a_day: launch no more than once a day (according to calendar)
    time_interval_list = None
    weekday_list = None
    seconds_from_last_run_commit = None
    seconds_from_last_run_start_time = None
    ignored_jobs = None
    last_run_finished = None
    seconds_from_last_run_commit_if_run_not_finished = None
    once_a_day = None


class SimpleFilterRegex(object):
    def __init__(self, path_regex=None, commit_message_regex=None):
        self.path_regex = re.compile(path_regex) if path_regex else None
        self.commit_message_regex = re.compile(commit_message_regex) if commit_message_regex else None


class Tag(enum.IntEnum):
    NO_PRECOMMIT_CHECKS = 1
    LIGHT_PRECOMMIT_CHECKS = 2
    FULL_PRECOMMIT_CHECKS = 3

    @staticmethod
    def from_str(type):
        return getattr(Tag, type)


class TestType(object):
    CHECK_TEST = 1
    DIFF_TEST = 2
    META_TEST = 3
    DELETED_TEST = 4

    @staticmethod
    def from_str(type):
        return getattr(TestType, type)

    @classmethod
    def to_str(cls, value):
        for name in dir(cls):
            if not name.startswith("_") and getattr(cls, name) == value:
                return name


class TaskNotifications:
    class Transport:
        EMAIL = "email"
        JABBER = "jabber"


class TaskStatus(str, Enum):
    """
    New-style statuses
    """
    DRAFT = "DRAFT"
    ENQUEUING = "ENQUEUING"
    ENQUEUED = "ENQUEUED"
    ASSIGNED = "ASSIGNED"
    PREPARING = "PREPARING"
    EXECUTING = "EXECUTING"
    TEMPORARY = "TEMPORARY"
    FINISHING = "FINISHING"
    WAIT_RES = "WAIT_RES"
    WAIT_TASK = "WAIT_TASK"
    WAIT_TIME = "WAIT_TIME"
    SUCCESS = "SUCCESS"
    RELEASING = "RELEASING"
    RELEASED = "RELEASED"
    FAILURE = "FAILURE"
    DELETED = "DELETED"
    EXCEPTION = "EXCEPTION"
    NO_RES = "NO_RES"
    TIMEOUT = "TIMEOUT"
    STOPPING = "STOPPING"
    STOPPED = "STOPPED"
    NOT_RELEASED = "NOT_RELEASED"

    def __str__(self):
        return self.value


class ChartParams(object):
    # Copypasted from https://a.yandex-team.ru/arc/trunk/arcadia/testenv/core/engine/test.py?rev=5668663#L528
    # "line" or "area"
    ChartType = "line"
    # None or "normal"
    AreaStacking = None
    # "3h" or "2w"
    DefaultZoom = "2w"
    # Don't show point comments in flags if False
    UsePointComments = True

    def ChartDataCacheToDictConverter(self, params):
        # TODO: deprecated
        """
            params.data_cache
            params.row
        """
        return None

    # load some columns from test_results table and pass them to ChartDataCacheToDictConverter
    TestResultColumns = []

    class Line:
        def __init__(self, data_key, type=None, color=None, z_index=None, name=None, y_value_type=None):
            self.Name = name
            # string or tuple (string, string) for arearange
            self.DataKey = data_key
            self.Type = type
            self.Color = color
            self.ZIndex = z_index
            # None or "seconds"
            self.YValueType = y_value_type
    # e.g.:
    # ("rps1", "rps2")
    # (Line("rps1"), Line("rps2", color="red"))
    ChartLines = None

    class YAxis:
        def __init__(self, name, data_keys=None, value_type=None, height=None):
            self.Name = name
            self.DataKeys = data_keys
            # None or "seconds"
            self.ValueType = value_type
            self.Height = height

    # e.g.:
    # single axis for all lines:
    #     "rps"
    # grouping lines:
    #     (YAxis("rps", ["rps1", "rps2"]), YAxis("memory", ("m",)))
    ChartYAxes = None

    KpiLineValues = None

    # in pixels
    Height = None


def union_dicts(a, b):
    if a:
        a.update(b)
    else:
        a = b
    return a


def construct_resource_description(params, rm_config):
    return "r{}: {}".format(
        params.revision,
        params.commit_message.encode("utf-8") if params.commit_message else "<testenv manual runs lost this value>",
    )


def arcadia_svn_url_with_revision(x, rm_config):
    return "{svn_ssh_url}/arcadia@{revision}".format(
        svn_ssh_url=x.svn_ssh_url,
        revision=x.revision,
    )


def args_to_list(**args):
    return args.values()


def join_args_with_sep(sep=",", *args):
    return sep.join(map(str, list(args)))


def get_major_release_number(params, rm_config):
    return rm_config.svn_cfg.get_major_release_num(params.svn_ssh_url)


def tuplify(obj):
    """
    Constructs tuple from object.
    :param obj: list, tuple or single element
    :return: tuple
    """
    if isinstance(obj, (list, tuple)):
        return tuple(obj)
    return (obj,)


def should_add_to_db_branch(rm_config, dbName):
    if rm_config.is_branched:
        return rm_config.testenv_cfg.branch_db_regex.match(dbName) is not None
    return False


def should_add_to_db_trunk(rm_config, dbName):
    return rm_config.testenv_cfg.trunk_db == dbName


def get_tasks_to_check(params, rm_config):
    return ",".join([str(t_id) for t_id in params.parent_test_task_id.values()])


def merge_job_params(user_job_params, default_job_params):
    job_params = {}
    job_params.update(user_job_params or {})
    default_job_params = default_job_params or {}
    for key in default_job_params:
        if key not in job_params:
            job_params[key] = default_job_params[key]
        else:
            if isinstance(default_job_params[key], dict):
                job_params[key] = merge_job_params(job_params[key], default_job_params[key])
    return job_params


def job_name_for_ci(te_job_name):
    return te_job_name.lstrip("_").lower()
