from sandbox.projects.common.constants import ARCADIA_URL_KEY
from sandbox.projects.release_machine.core import const as rm_const


class Result(object):
    _ok = None

    def __init__(self, value=None, context=None):
        self._value = value
        self._context = context or {}

    @property
    def ok(self):
        if self._ok is None:
            raise NotImplementedError
        return self._ok

    @property
    def result(self):
        return self._value

    def get_context_field(self, key):
        return self._context.get(key)

    def set_context_field(self, key, value):
        self._context[key] = value

    def __eq__(self, other):
        return self.ok == other.ok and self.result == other.result

    def __str__(self):
        return "{}({})".format(self.__class__.__name__, self._value)

    __repr__ = __str__


class Ok(Result):
    _ok = True


class Error(Result):
    _ok = False


class ReleasedItem(object):
    """ Item for releases using release-machine """

    def __init__(
        self, name, resource,
        build_ctx_key=None,
        display_name=None,
        local_path=None,
        storage=None
    ):
        """
            :param name: Item name
            :param resource: Instance of SandboxResource or sdk2.Resource
            :param build_ctx_key: Key of getting
        """
        self.name = name
        self.display_name = display_name or name
        self.resource = resource
        self.build_task_id = resource.task_id
        self.build_ctx_key = build_ctx_key or ARCADIA_URL_KEY
        self.local_path = local_path
        self.storage = storage

    def to_json(self):
        return self.__dict__

    def __str__(self):
        return self.name

    __repr__ = __str__


class ReleasedResource(object):
    """
        Release = press release button (prepare to deploy)

        :param id: Resource id
        :param build_task_id: Id task, which built this resource
        :param timestamp:
        :param major_release: Branch number, which resource was released from
        :param minor_release: None or tag number
        :param component: component name
        :param status: one item from ReleaseStatus enum
        :param owner: release owner
        :param info: field for additional info (now used only for nanny tickets)
    """
    FORMATTER = "{resource_name}[{status}], component={component} major={major_release}, minor={minor_release}, id={id}"

    def __init__(
        self,
        id=None,
        build_task_id=None,
        timestamp=None,
        major_release=None,
        minor_release=None,
        component=None,
        status=rm_const.ReleaseStatus.stable,
        owner=None,
        info=None,
        resource_name="",
    ):
        self.id = id
        self.build_task_id = build_task_id
        self.timestamp = timestamp
        self.major_release = major_release
        self.minor_release = minor_release
        self.component = component
        self.status = status
        self.owner = owner
        self.info = info
        self.resource_name = resource_name

    def to_json(self):
        return self.__dict__

    def __repr__(self):
        return "{} {}".format(self.__class__.__name__, self.FORMATTER.format(**self.to_json()))

    def __lt__(self, other):
        if self.major_release and other.major_release:
            if self.major_release < other.major_release:
                return True
            elif self.major_release > other.major_release:
                return False
        if self.minor_release and other.minor_release:
            return self.minor_release < other.minor_release
        return False

    @classmethod
    def format_from_dict(cls, resource_as_dict):
        return cls.FORMATTER.format(**resource_as_dict)


class DeployedResource(ReleasedResource):
    """ Deploy = actually switch production to new version """

    @staticmethod
    def from_released(released_resource):
        return DeployedResource(**released_resource.to_json())


class ReleaseDiff(object):
    POS = ["same", "new", "old"]

    class Type(object):
        major = "major"
        minor = "minor"

    class Position(object):
        old = -1
        same = 0
        new = 1

    def __init__(self, position, release_type=Type.major, staging=rm_const.ReleaseStatus.stable):
        self.release_type = release_type
        self.position = position
        self.staging = staging

    def __eq__(self, other):
        return (
            self.release_type == other.release_type and
            self.position == other.position and
            self.staging == other.staging
        )

    def __repr__(self):
        return "{}[{}, {}, {}]".format(
            self.__class__.__name__,
            self.POS[self.position],
            self.release_type,
            self.staging,
        )
