"""
    Here are constants which are related to release-machine.
    Please, do NOT put here either release logic or specific dependencies!
    This file will be imported almost everywhere in release machine.
    Maintainer: ilyaturuntaev@, glebov-da@
"""
import enum
import re
import string
from collections import namedtuple

AB_ADMINKA_TOKEN_NAME = "ab_formulas_release_token"
ARC_ACCESS_TOKEN_NAME = "arc_access_key"
COMMON_TOKEN_NAME = "common_release_token"
TELEGRAM_TOKEN_NAME = "rm_telegram_token"
RM_TELEGRAM_TOKEN_VAULT_UUID = "sec-01dbz09vh1bbzxqwee1rakc30d"
RM_TELEGRAM_TOKEN_VAULT_VERSION_UUID = "ver-01dbz6f6j28dgacfvw7my8krmg"
Q_MESSENGER_TOKEN_NAME = "q_messenger_token"
YAPPY_TOKEN_NAME = "yappy_auth_token"
YQL_TOKEN_NAME = "yql_oauth_token"
MORTY_TOKEN_NAME = "morty_oauth_token"

COMMON_TOKEN_OWNER = "SEARCH-RELEASERS"
ROBOT_RELEASER_USER_NAME = "robot-srch-releaser"
RM_ABC_GROUP = "96651"
RM_ABC_SERVICE_NAME = "releasemachine"
COMPONENT_CTX_KEY = "component_name"
ACCEPTANCE_FAIL_KEY = "acceptance_fail"
REVISION_CTX_KEY = 'revision'
RELEASE_NUMBER = "release_number"
TEST_RESULTS_CONTEXT_FIELD = "rm_test_results"
RELEASE_MACHINE_ERROR_REPORT = "release_machine_error_report"

ABC_SCOPE_VIRTUAL = "virtual"

REVIEW_RE = re.compile(r"review:\s*([0-9]{5,})", flags=re.IGNORECASE)
REVIEW_REPLACE_RE = re.compile(r"(review)\s*:\s*([0-9]+)", flags=re.IGNORECASE)
REVIEW_URL_RE = re.compile(r"https?://a.yandex-team.ru/review/(\d{5,})")

MERGE_RE = re.compile(r"merge\s*from\s*trunk\s*:\s*([r0-9, ]+)", flags=re.IGNORECASE)
INCLUDE_REV_RE = re.compile(r"([ >]+) INCLUDE: r(\d{6,11})\D")

DIFF_RESOLVER_MASK = "[diff-resolver:"
DIFF_RESOLVER_RE = re.compile(r"\[ *diff-resolver *: *([a-z\-0-9]+) *\]")
DEVEXP_RE = re.compile(
    r"({}.*{})".format(re.escape("<!-- DEVEXP BEGIN -->"), re.escape("<!-- DEVEXP END -->")),
    flags=re.DOTALL
)
CHANGELOG_MARKER = "<# <!-- CHANGELOG --> #>"

# "TE:<db_name>:<job_name>:<job_id>" or "TE:<db_name>:<job_name>"
GSID_CONTEXT_KEY = "__GSID"
CI_CONTEXT_KEY = "__CI_CONTEXT"
GSID_TE_RE = re.compile(r"TE:(?P<db_name>[a-z0-9\-_]+):(?P<job_name>[A-Za-z_0-9]+):?(?P<job_id>\d*)")
GSID_SVN_RE = re.compile(r"SVN:(?P<svn_revision>\d+)")  # SVN:<svn_revision>
GSID_SVNDIFF_RE = re.compile(r"SVNDIFF:(?P<rev1>\d+):(?P<rev2>\d+)")  # SVN:<rev1>:<rev2>
SYNCED_ARC_COMMIT = re.compile(r"Sync[\s]*arc[\s]*commit[\s]*:[\s]*([a-z\d]+(,[\s]*[a-z\d]+)*)")
SYNCED_ARC_MANUAL_COMMIT_MSG = re.compile(r"Sync[\s]*manual[\s]*commit[\s]*in[\s]*arc[\s]*branch")
APPHOST_VERTICAL_NAME_VALID_SYMBOLS = frozenset(string.ascii_lowercase + string.digits)
COMPONENT_NAME_VALID_SYMBOLS = frozenset(string.ascii_lowercase + string.digits + "_")
ARC_HASH_LENGTH = 40
STARTREK_RELEASE_NUMBER_TAG_TEMPLATE = "release_number_{}"
EVENT_HASH_SEPARATOR = "::"

MAIN_LOCATIONS = ["sas", "man", "vla"]
AUX_LOCATIONS = ["myt", "iva"]
ALL_LOCATIONS = MAIN_LOCATIONS + AUX_LOCATIONS


class CommitMode(object):
    RELEASE_MACHINE_MODE = "release_machine"
    CUSTOM_MODE = "custom"


class RollbackMode(CommitMode):
    TRUNK_MODE = "trunk"
    TRUNK_AND_MERGE_MODE = "trunk_and_merge"


class ActionType(object):
    MERGE = "merge"
    ROLLBACK = "rollback"
    MANUAL_COMMIT = "manual_commit"


class BranchCommit(object):
    """ Contains information about arc commit and synced svn commit (if any). """

    __slots__ = "action_type", "current_hash", "current_rev", "arc_hashes", "svn_revs"

    def __init__(self, action_type, current_hash=None, current_rev=None, arc_hashes=None, svn_revs=None):
        """
        Initialization.

        :param action_type: one of ActionType values
        :param current_hash: arc hash of this commit
        :param current_rev: svn revision of this commit
        :param arc_hashes: arc hashes, which are merged or rollbacked in this commit, correspond to svn_revs
        :param svn_revs: svn revisions, which are merged or rollbacked in this commit, correspond to arc_hashes
        """
        self.action_type = action_type
        self.current_hash = current_hash
        self.current_rev = current_rev
        self.arc_hashes = arc_hashes
        self.svn_revs = svn_revs

    def __repr__(self):
        return (
            "<action_type: {action_type_val}, current_hash: {current_hash_val}, current_rev: {current_rev_val}, "
            "arc_hashes: {arc_hashes_val}, svn_revs: {svn_revs_val}>".format(
                action_type_val=self.action_type,
                current_hash_val=self.current_hash,
                current_rev_val=self.current_rev,
                arc_hashes_val=self.arc_hashes,
                svn_revs_val=self.svn_revs,
            )
        )


class RepoActionResult(object):
    def __init__(self, status):
        self.status = status


class CommitResult(RepoActionResult):
    def __init__(self, status, revision=None):
        super(CommitResult, self).__init__(status)
        self.revision = revision

    def __repr__(self):
        return "<CommitResult {status} revision={revision}>".format(
            status=self.status,
            revision=self.revision,
        )

    def __str__(self):
        return repr(self)


class ReviewResult(RepoActionResult):
    def __init__(self, status, review_id=None):
        super(ReviewResult, self).__init__(status)
        self.review_id = review_id


class ActionItem(object):
    def __init__(self, mode, c_info=None, merge_types=None):
        self.mode = mode
        self.c_info = c_info
        self.merge_types = merge_types

    @property
    def name(self):
        """
        Action item name.

        :return: c_info.name if c_info is specified, mode otherwise.
        """
        if self.c_info:
            return self.c_info.name
        else:
            return self.mode


class PipelineStage(object):
    release = "release"
    deploy = "deploy"


class ReleaseCycleType(object):
    BRANCH = 1
    TAG = 2
    TRUNK = 3
    CI = 4

    @staticmethod
    def to_str(number):
        """ Usage: ReleaseCycleType.to_str(ReleaseCycleType.BRANCH) -> 'branched' """
        return ["branched", "tagged", "trunk", "ci"][number - 1]


class ReleaseStatus(object):
    stable = "stable"
    testing = "testing"
    prestable = "prestable"
    unstable = "unstable"


class CiTestResult(enum.Enum):
    failed = "failed"
    success = "success"
    running = "in_progress"


class CommitStatus(enum.Enum):
    failed = "Failed"
    changeset_is_empty = "Empty"
    success = "Success"


class DeploySystem(enum.Enum):
    others = 0
    nanny = 1
    qloud = 2
    ya_deploy = 3
    sandbox = 4
    samogon = 5
    z2 = 6
    ab = 7
    ya_tool = 8
    nanny_push = 9
    kosher_sandbox_release = 10
    ci_tasklet_registry = 11


class PermissionType(object):
    ALLOWED = 0
    BANNED = 1
    CHANGED = 2


class RMNames(object):
    DEFAULT = "unknown_component"
    DWH = "dwh"
    BASE = "base"
    BIATHLON = "biathlon"
    FEEDBACK = "feedback_platform"
    FLOYD = "floyd"
    FINDURL = "findurl"
    VIDEO_BASE = "video_base"
    VIDEO_INDEX = "video_index"
    VIDEO_REPORT = "video_report"
    FRONTEND_VIDEO_HOSTING = "frontend_video_hosting"
    MIDDLE = "middle"
    EXP_FORMULAS_BASE = "exp_formulas_base"
    EXP_FORMULAS_MMETA = "exp_formulas_mmeta"
    UPPER = "upper"
    RD = "rearrange_dynamic"
    RELEASE_MACHINE_TEST = "release_machine_test"
    RELEASE_MACHINE_TEST_TAGGED = "release_machine_test_tagged"
    BALANCER = "balancer"
    BEGEMOT = "begemot"
    BEGEMOT_SPELLCHECKER = "begemot_spellchecker"
    BEGEMOT_SHARDS = "begemot_shards"
    BULBASAUR = "bulbasaur"
    LYNCHER = "lyncher"
    SUPERMODERATION = "supermoderation"
    BRANDS = "brands"
    CUPID = "cupid"
    IMAGES_BASE = "images_base"
    IMAGES_BASE_DEV = "images_base_dev"
    IMAGES_ROBOT = "images_robot"
    IMAGES_SEMIDUPS = "images_semidups"
    ANSWERS_BASE = "answers_base"
    ADDRS_BASE = "addrs_base"
    ADDRS_META = "addrs_meta"
    SRC_SETUP = "src_setup"
    UGC_SERVER = "ugc_server"
    UGC_DB_BACKEND = "ugc_db_backend"
    UGC_FEEDER = "ugc_feeder"
    JUPITER = "jupiter"
    JUPITER_VIEWERS = "jupiter_viewers"
    GEMINI = "gemini"
    REPORT_WEB = "report"
    APACHE = "apache"
    KWYT = "kwyt"
    RTHUB = "rthub"
    SAVED_COPY = "savedcopy"
    RTCC = "rtcc"
    ENTITYSEARCH_BACKEND = "entitysearch"
    TUNNELLER = "tunneller"
    PRS_OPS = "prs_ops"
    SCRAPER_OVER_YT = "scraper_over_yt"
    APP_HOST = "app_host"
    HTTP_ADAPTER = "http_adapter"
    USERFEAT = "userfeat"
    USERFEAT_VIEWERS = "userfeat_viewers"
    WEB_GRAPH_MAPPING = "web_graph_mapping"
    IMAGES_GRAPH_MAPPING = "images_graph_mapping"
    VIDEO_GRAPH_MAPPING = "video_graph_mapping"
    TURBO_PAGES = "turbo_pages"
    TURBO_MERGER = "turbo_merger"
    TURBO_INIT = "turbo_init"
    TURBO_DDM = "turbo_ddm"
    ADFOX = "adfox"
    ADFOX_MATCHID = "adfox_matchid"
    ADFOX_SOLO = "adfox_solo"
    CLICKDAEMON = "clickdaemon"
    URL_MONITORING = "url_monitoring"
    BALANCER_CONFIG = "balancer_config"
    SERVICE_CONTROLLER = "service_controller"
    POD_AGENT = "pod_agent"
    REQUEST_SAMPLER = "request_sampler"
    RPS_LIMITER = "rpslimiter"
    GRPC_CLIENT = "grpc_client"
    MUSIC = "music"
    ITDITP_MIDDLE = "itditp_middle"
    USER_SESSIONS = "user_sessions"
    BASS = "bass"
    YABS_BSTR = "yabs_bstr"
    YABS_SERVER = "yabs_server"
    YABS_SERVER_VALIDATOR = "yabs_server_validator"
    YABS_SERVER_L7BALANCER = "yabs_server_l7balancer"
    ABT = "abt"
    EXPADM = "expadm"
    EXPDAEMON = "expdaemon"
    CORES = "cores"
    WOLAND = "woland"
    TUTOR = "tutor"
    TUTOR_ADMIN = "tutor_admin"
    RTMR = "rtmr"
    MAPS_CORE_TEAPOT = "maps_core_teapot"
    UNIPROXY_EXPERIMENTS = "uniproxy_experiments"
    WEATHER = "weather"
    FLAGS_PROVIDER = "flags_provider"
    RESOURCE_CACHE_CONTROLLER = "resource_cache_controller"
    DEPLOY_MONITORING_CONTROLLER = "deploy_monitoring_controller"
    HORIZONTAL_POD_AUTOSCALER_CONTROLLER = "horizontal_pod_autoscaler_controller"
    BOLTALKA = "boltalka"
    COLLECTIONS_BACKEND = "collections_backend"
    YP_DNS = "yp_dns"
    YP_SERVICE_DISCOVERY = "yp_service_discovery"
    ASR_SERVER = "asr_server"
    VINS = "vins"
    TRAVEL_HOTELS_ADMINISTRATOR = "travel_hotels_administrator"
    COLLECTIONS_FEED_READER = "collections_feed_reader"
    YP_IDM_ROLE_PROVIDER = "yp_idm_role_provider"
    PASSPORT_API = "passport_api"
    QUASAR_LINUX = "quasar_linux"
    QUASAR_YANDEXSTATION_2 = "quasar_yandexstation_2"
    SMART_DEVICES = "smart_devices"
    YP_INET_MNGR = "yp_inet_mngr"
    SOCIAL_API = "social_api"
    SOCIAL_BROKER = "social_broker"
    SOCIAL_CONFIGS = "social_configs"
    SOCIAL_METRICS = "social_metrics"
    SOCIAL_NGINX = "social_nginx"
    SOCIAL_PROXY2 = "social_proxy2"
    SOCIAL_SECRETS_YAV = "social_secrets_yav"
    SOCIAL_UTILS_CELERYBEAT = "social_utils_celerybeat"
    SOCIAL_UTILS_DUMPERS = "social_utils_dumpers"
    SOCIAL_UTILS = "social_utils"
    STAGE_CONTROLLER = "stage_controller"
    ARC_CLIENT = "arc_client"
    YP_CAUTH_EXPORT = "yp_cauth_exort"
    STATINFRA = "statinfra"
    FANOUT = "fanout"
    METASEARCH = "metasearch"
    LB_SUBSCRIBER = "lb_subscriber"
    CS_SERVER = "cs_server"
    DATA_COLLECTOR = "data_collector"
    GEO_BOT = "geobot"
    MEDIA_CRM = "media_crm"
    YP_EXPORT = "yp_export"
    ALISA_SKILLS_REC = "alisa_skills_rec"
    AGENCY_REWARDS = "agency_rewards"
    BALANCE = "balance"
    VIDEO_HOSTING_GRAPHS = "video_hosting_graphs"
    TYCOON_API = "tycoon_api"
    DJ_UNITY = "dj_unity"
    DJ_RTHUB = "dj_rthub"
    BALANCE_DCSAAP = "balance_dcsaap"
    YDO_SEARCHPROXY = "ydo_searchproxy"
    DIALOGOVO = "dialogovo"
    PASKILLS_BILLING = "paskills_billing"
    DRIVE = "drive"
    AUTH_CONTROLLER = "auth_controller"
    DEVOPS = "devops"
    FURY_ADFOX = "fury_adfox"
    FURY_CAPTCHA = "fury_captcha"
    FURY_MARKET = "fury_market"
    FURY_RTXARON = "fury_rtxaron"
    FURY_SUPERMOD = "fury_supermod"
    FURY_MEGAERA = "fury_megaera"
    FURY_RTPREBILLING = "fury_rtprebilling"
    XURMA_ADMET = "xurma_admet"
    XURMA_ADMOD = "xurma_admod"
    XURMA_AFRAUDITA = "xurma_afraudita"
    XURMA_BS = "xurma_bs"
    XURMA_BTR = "xurma_btr"
    XURMA_DSP = "xurma_dsp"
    XURMA_EDADEAL = "xurma_edadeal"
    XURMA_ONLINE = "xurma_online"
    XURMA_PREBILLING = "xurma_prebilling"
    XURMA_TAXI = "xurma_taxi"
    XURMA_ZEN = "xurma_zen"
    MEDIA_STORIES = "media_stories"
    YANDEXMICRO_FACTORY = "yandexmicro_factory"
    YANDEXMIDI_FACTORY = "yandexmidi_factory"
    YANDEXMINI_2_FACTORY = "yandexmini_2_factory"
    YANDEXMODULE_2_FACTORY = "yandexmodule_2_factory"
    YANDEXPLUTO_FACTORY = "pluto_factory"

    @classmethod
    def props(cls):
        return sorted([v for k, v in cls.__dict__.iteritems() if k[:1] != '_' and v != cls.DEFAULT and type(v) == str])


class Urls(object):
    A_URL = "https://a.yandex-team.ru/"
    NANNY_API = "https://nanny.yandex-team.ru/v2/"
    NANNY_BASE_URL = "https://nanny.yandex-team.ru/"
    RM_HOST = "rm.z.yandex-team.ru"
    RM_TESTING_HOST = "rm-testing.z.yandex-team.ru"
    RM_URL = "https://{}/".format(RM_HOST)
    RM_TESTING_URL = "https://{}/".format(RM_TESTING_HOST)
    STAFF_URL = "https://staff.yandex-team.ru"
    STAFF_API = "https://staff-api.yandex-team.ru/v3/"
    SVN_ARCADIA_URL = "arcadia.yandex.ru/arc/"
    WIKI = "https://wiki.yandex-team.ru/"
    WIKI_API = "https://wiki-api.yandex-team.ru/_api/frontend/"
    SANDBOX_URL = "http://sandbox.yandex-team.ru/"
    STARTREK = "https://st.yandex-team.ru/"
    STARTREK_API = "https://st-api.yandex-team.ru/v2/"
    TESTPALM_URL = "https://testpalm.yandex-team.ru/"
    YAPPY = "https://yappy.z.yandex-team.ru/"
    METRICS = "https://metrics.yandex-team.ru/"
    TESTENV = "https://testenv.yandex-team.ru/"
    SOLOMON_URL = "http://solomon.yandex.net"
    SOLOMON_PUSH_API_URL = "{}/api/v2/push".format(SOLOMON_URL)
    RELEASE_MACHINE_INVITE_LINK = "https://t.me/joinchat/AAAAAEN49wUGP2uImQ6Dfw"
    UPS_INVITE_LINK = "https://t.me/joinchat/CaUODlfDasYVnJ5W7Q00_A"
    CALENDAR_API_URL = "https://api.calendar.yandex-team.ru/intapi"
    INFRA_API = "https://infra-api.yandex-team.ru/v1"
    AB_EXPERIMENTS = "https://ab.yandex-team.ru"

    @classmethod
    def te_db_screen(cls, db_name, screen_name="manage"):
        return "{}?screen={}&database={}".format(cls.TESTENV, screen_name, db_name)

    @classmethod
    def make_arcanum_url(cls, arcadia_url):
        arcadia_prefix = "arcadia:/"
        if arcadia_url.startswith(arcadia_prefix):
            return arcadia_url.replace(arcadia_prefix, cls.A_URL, 1)
        return arcadia_url


# Sandbox tasks RestartPolicy actions
class RestartPolicyActions(object):
    # Restart task
    RESTART = 'restart'

    # Recreate task and tasks for all parent tests
    RECREATE = 'recreate'

    # Do nothing
    NOOP = 'noop'


beta_creation_restart_policy = (  # SEARCH-2456
    # Restart attempt #1
    {
        'sleep_before_action': 20.0,  # min
        'default_action': RestartPolicyActions.RESTART,
        'override_default_action': {
            'NO_RES': RestartPolicyActions.RECREATE,
            'TIMEOUT': RestartPolicyActions.NOOP,
        },
    },
    # Restart attempt #2
    {
        'sleep_before_action': 40.0,  # min
        'default_action': RestartPolicyActions.RECREATE,
        'override_default_action': {
            'NO_RES': RestartPolicyActions.RECREATE,
            'TIMEOUT': RestartPolicyActions.NOOP,
        },
    },
)


class Workflow(object):
    OPEN = 'open'
    OPENED = 'opened'
    AUTOTESTING = 'autoTesting'
    AUTOTESTS_OK = 'autoTestsOK'
    FIX_PROBLEMS = 'fixProblems'
    ACCEPTING = 'accepting'
    QUALITY_OK = 'qualityOK'
    DEPLOYING = 'deploying'
    PRODUCTION = 'production'
    CLOSE = 'close'
    CLOSED = 'closed'
    REOPEN = 'reopen'
    BETA_TEST = {
        OPEN: AUTOTESTING,
        AUTOTESTING: AUTOTESTS_OK,
        AUTOTESTS_OK: ACCEPTING,
        FIX_PROBLEMS: ACCEPTING,
        ACCEPTING: QUALITY_OK,
        QUALITY_OK: DEPLOYING,
        DEPLOYING: PRODUCTION,
        PRODUCTION: CLOSE,
        CLOSED: REOPEN,
    }


class TicketGroups(object):
    E2ETest = "====E2E test"
    MergedRevisions = "====Merged revisions"
    MetricsTest = "====Metrics test"
    AquaTest = "====Aqua tests"
    BetaCreation = "====Beta creation"
    XMLTest = "====Xml test"
    PerfTest = "====Performance test"
    BuildTest = "====Build Test"
    TicketTableHeaders = {
        MergedRevisions: ["Tag", "Revisions merged", "Branch revision", "Revision info", "Tickets", "Action"],
    }


class SearchSubtypes(object):
    WEB = "web"
    IMAGES = "images"
    VIDEO = "video"
    GEO = "geo"
    QUICK = "quick"
    NEWS = "news"
    MUSIC = "music"
    ALL = (WEB, IMAGES, VIDEO, GEO, QUICK, NEWS, MUSIC)
    DEFAULT = WEB


class StartrekTicketFilterTypes(object):
    include = "Include"
    exclude = "Exclude"


class TicketFilter(namedtuple("TicketFilter", ["queues", "tickets", "filter_type"])):
    def __new__(cls, queues, tickets=(), filter_type=StartrekTicketFilterTypes.exclude):
        return super(TicketFilter, cls).__new__(cls, queues, tickets, filter_type)


class ChangelogTypes(object):
    MAJOR = "major"
    MINOR = "minor"
    all = [MAJOR, MINOR]


class BranchingMode(object):
    BRANCH_TO_TAG = "branch->tag"
    TRUNK_TO_BRANCH = "trunk->branch"
    TRUNK_TO_TAG = "trunk->tag"  # create tag from trunk, without branching
    TRUNK = "trunk"  # create resource with path to trunk (no arcadia modification)
    BRANCH_TO_BRANCH = "branch->branch"

    all = [
        BRANCH_TO_TAG,
        TRUNK_TO_BRANCH,
        TRUNK_TO_TAG,
        TRUNK,
        BRANCH_TO_BRANCH,
    ]


class TestFrequency(object):
    """
        Copypasted from test_environment/engine/data/metadata.py
    """
    CHECK_EACH_REVISION = 1
    EVERY_N_MINUTES = 2
    LAZY = 3
    EVERY_N_COMMIT = 4
    RUN_N_MINUTES_AFTER_LAST_RUN = 5
    DEFINED_BY_CODE = 6


class TestFrequencies(object):
    LAZY_TEST = {"frequency": TestFrequency.LAZY}
    EACH_REV_TEST = {"frequency": TestFrequency.CHECK_EACH_REVISION}
    ONCE_A_DAY_TEST = {"frequency": TestFrequency.EVERY_N_MINUTES, "frequency_param": 60 * 24}
    EVERY_TWO_DAYS_TEST = {"frequency": TestFrequency.EVERY_N_MINUTES, "frequency_param": 60 * 24 * 2}
    FORTY_TWO_TEST = {"frequency": TestFrequency.EVERY_N_MINUTES, "frequency_param": 42}
    DEFINED_BY_CODE_TEST = {"frequency": TestFrequency.DEFINED_BY_CODE}


class RMStatus(object):
    on_success = "on_success"
    on_break = "on_break"
    on_failure = "on_failure"
    custom = "custom"  # Should not be in ALL
    ALL = [on_success, on_break, on_failure]


class TaskType(object):
    def __init__(self, name, tasks):
        self.name = name
        self.tasks = tasks


class TaskTypes(object):
    ALL_RM_TASKS = TaskType(
        "ALL_RM_TASKS",
        [
            "LOG_MERGES_IN_STARTREK",
            "RELEASE_MONITOR_CRAWL_TICKETS",
        ]
    )
    RELEASE_MONITOR_CRAWL_RELEASES = TaskType(
        "RELEASE_MONITOR_CRAWL_RELEASES",
        [
            "RELEASE_MONITOR_CRAWL_RELEASES",
        ]
    )
    BUILD = TaskType(
        "BUILD",
        [
            "BUILD_BASESEARCH_AND_RTY",
            "BUILD_BASESEARCH",
            "BUILD_BASESEARCH_2",  # ??
            "BUILD_MIDDLESEARCH",
            "BUILD_NOAPACHE_UPPER",
            "BUILD_REARRANGE_DYNAMIC",
            "BUILD_WIZARD_GZT_OR_REMORPH",
            "BUILD_WEB_DYNAMIC_MODELS",
            "BUILD_WEB_DYNAMIC_MODELS_FLAG",
            "BUILD_APP_HOST_SRC_SETUP",
            "BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE",
            "BUILD_PRS_OPS",
            "BUILD_BALANCER_CONFIGS_L7",
            "BUILD_UPPER_CONFIG",
        ]
    )
    BALANCER_CONFIG = TaskType(
        "BALANCER_CONFIG",
        [
            "COMPARE_BALANCER_CONFIGS",
        ]
    )
    PRS_OPS = TaskType(
        "PRS_OPS",
        [
            "COMPARE_YT_TABLES_PRS_OPS",
            "GET_SAMPLE_FOR_PRS_OPS",
            "PRS_OPS_CLEAN_TABLES",
            "PRS_OPS_FACTOR_DIFF",
            "RUN_PRS_OPS",
            "RUN_PRS_OPS_CASTING",
            "RUN_PRS_OPS_CHECK_FILES",
            "RUN_PRS_OPS_NIRVANA",
        ]
    )
    ADDR_BASE_SPEC = TaskType(
        "ADDR_BASE_SPEC",
        [
            "BUILD_SEARCH",
            "GET_ADDRS_REQUESTS",
            "GENERATE_PLAN_FROM_QUERIES",
            "CHECK_GEOBASESEARCH_MMS_COMPABILITY",
            "ADDRS_DEPLOY",
            "RELEASE_ADDRS_SHARDMAP",
            "ADDRS_MAPKIT_ACCEPTANCE_SDK_2",
            "ADDRS_BASE_PERFORMANCE_PARALLEL_ACCEPTANCE",
            "BUILD_GEOSEARCH",
            "COMPARE_GEOMETASEARCH_BINARIES",
        ]
    )
    INIT_SETUP = TaskType(
        "INIT_SETUP",
        [
            "CREATE_BRANCH_OR_TAG",
            "CREATE_CHANGELOG",
            "RELEASE_MACHINE_CHANGELOG",
            "CREATE_STARTREK_TICKET",
            "CREATE_WIKI_PAGE",
            "CLONE_TESTENV_DB",
        ]
    )
    CREATE_STARTREK_TICKET = TaskType(
        "CREATE_STARTREK_TICKET",
        [
            "CREATE_STARTREK_TICKET",
        ]
    )
    CREATE_BRANCH_OR_TAG = TaskType(
        "CREATE_BRANCH_OR_TAG",
        [
            "CREATE_BRANCH_OR_TAG",
        ]
    )
    CLONE_TESTENV_DB = TaskType(
        "CLONE_TESTENV_DB",
        [
            "CLONE_TESTENV_DB",
        ]
    )
    RELEASE_MACHINE_CHANGELOG = TaskType(
        "RELEASE_MACHINE_CHANGELOG",
        [
            "RELEASE_MACHINE_CHANGELOG",
        ]
    )
    CREATE_WIKI_PAGE = TaskType(
        "CREATE_WIKI_PAGE",
        [
            "CREATE_WIKI_PAGE",
        ]
    )
    RELEASE_RM_COMPONENT = TaskType(
        "RELEASE_RM_COMPONENT",
        [
            "RELEASE_RM_COMPONENT",
            "RELEASE_RM_COMPONENT_2",
            "RELEASE_RM_COMPONENT_AB",
            "RELEASE_SEARCH_COMPONENT_Z_2",
        ]
    )
    LAUNCH_METRICS = TaskType("LAUNCH_METRICS", ["LAUNCH_METRICS"])
    PRIEMKA = TaskType(
        "PRIEMKA",
        [
            "CHECK_NOAPACHEUPPER_PRIEMKA_STATE",
            "PRIEMKA_BASESEARCH_BINARY",
            "PRIEMKA_MIDDLESEARCH_BINARY",
        ]
    )
    MERGE_TO_STABLE = TaskType("MERGE_TO_STABLE", ["MERGE_TO_STABLE"])
    ROLLBACK_COMMIT = TaskType("ROLLBACK_COMMIT", ["ROLLBACK_COMMIT"])
    ACCEPTANCE_NOAPACHE = TaskType("ACCEPTANCE_TEST_NOAPACHEUPPER", [
        "ACCEPTANCE_TEST_NOAPACHEUPPER",
        "CHECK_UPPER_EXP",
        "TEST_ALL_TDI_AND_REARRANGE",
    ])
    GENERATE_BETA = TaskType("GENERATE_BETA", ["GENERATE_YAPPY_BETA"])
    DUNNO = TaskType(
        "DUNNO",
        [
            "BISECT_NOAPACHEUPPER_DIFF_REASON",
            "BISECT_WIZARD_DIFF_REASON",
            "CHECK_CACHEHIT_FOR_BETA",
            "GET_MIDDLESEARCH_RESOURCES",
            "REARRANGE_ACCEPTANCE_2",  # ??
            "RESOURCE_LOADER_FROM_NANNY",
            "TEST_FIND_URL_BUCKET",
        ]
    )
    KPI = TaskType("KPI", ["RELEASE_MONITOR_KPI_ALERT"])
    RM_NOT_NOTIFIED = TaskType(
        "RM_NOT_NOTIFIED",
        [
            "CHECK_TASKS_CORRECTNESS",
            "WAIT_TRUNK_DB",
        ]
    )
    USER_SESSIONS = TaskType(
        "USER_SESSIONS",
        [
            "RELEASE_USER_SESSIONS_SCRIPTS",
            "RELEASE_USER_SESSIONS_BINARIES",
            "GET_DIFF_RESOURCE_FOR_US",
            "GET_LAST_GOOD_RELEASE_INFO",
            "RELEASE_USER_SESSIONS_BIN_ON_TIME",
            "SEND_MESSAGE_TO_ST_FROM_RM",
            "PRIEMKA_CREATE_SESSIONS",
            "PREPARE_USRMRESOURCE",
            "DEPLOY_RULE_MONITORING",
        ]
    )
    VIDEO_FORMULAS = TaskType(
        'VIDEO_FORMULAS',
        [
            'VIDEO_BUILD_AND_TEST_MIDDLE_DYNAMIC_MODELS',
        ]
    )
    XML_SEARCH = TaskType('XML_SEARCH', ['TEST_XML_SEARCH_2', ])
    DEVOPS = TaskType(
        'DEVOPS',
        [
            'UPDATE_MARTY_PANELS',
            'DEV_OPS_ALERT',
        ]
    )
    OTHER = TaskType("OTHER", [""])
    ALL_TYPES = [
        ALL_RM_TASKS,
        RELEASE_MONITOR_CRAWL_RELEASES,
        BUILD,
        RELEASE_MACHINE_CHANGELOG,
        CLONE_TESTENV_DB,
        CREATE_WIKI_PAGE,
        CREATE_BRANCH_OR_TAG,
        PRS_OPS,
        DUNNO,
        KPI,
        RM_NOT_NOTIFIED,
        RELEASE_RM_COMPONENT,
        LAUNCH_METRICS,
        MERGE_TO_STABLE,
        ROLLBACK_COMMIT,
        ADDR_BASE_SPEC,
        ACCEPTANCE_NOAPACHE,
        GENERATE_BETA,
        BALANCER_CONFIG,
        USER_SESSIONS,
        VIDEO_FORMULAS,
        XML_SEARCH,
        DEVOPS,
    ]


class JobTypes(object):
    # pre-release
    NEW_BRANCH = "NEW_BRANCH"
    CLONE_DB = "CLONE_DB"
    CHANGELOG_MAJOR = "CHANGELOG_MAJOR"
    STARTREK = "STARTREK"
    LINK_STARTREK_TICKETS = "LINK_ST_TICKETS"
    WIKI = "WIKI"
    ACTION_PRE_RELEASE = "ACTION_PRE_RELEASE"
    # test
    TEST = "TEST"
    RUN_ACCEPTANCE = "RUN_ACCEPTANCE"
    ACTION_RUN_ACCEPTANCE = "ACTION_RUN_ACCEPTANCE"
    ACTION_RUN_ACCEPTANCE_BRANCHED = "ACTION_RUN_ACCEPTANCE_BRANCHED"
    # release
    NEW_TAG = "NEW_TAG"
    BUILD = "BUILD"
    BUILD_TRUNK = "BUILD_TRUNK"
    CLEANUP = "CLEANUP"
    DEPLOY = "DEPLOY"
    CHANGELOG_FINAL = "CHANGELOG_FINAL"
    CHANGELOG_MINOR = "CHANGELOG_MINOR"
    CHANGELOG = "CHANGELOG"
    CHANGELOG_FORMAT = "CHANGELOG_FORMAT"
    CHANGELOG_POST_ST = "CHANGELOG_POST_STARTREK"
    WIKI_FINAL = "WIKI_FINAL"
    RELEASE = "RELEASE"
    ACTION_RELEASE = "ACTION_RELEASE"
    # other
    LOG_MERGE = "LOG_MERGE"
    RM_CRAWLER = "RM_CRAWLER"
    YAPPY_BETA = "GENERATE_BETA"
    TEST_E2E = "TEST_E2E"
    TEST_XML_SEARCH = "TEST_XML_SEARCH"
    TEST_ALL_RA2 = "TEST_ALL_RA2"
    LAUNCH_METRICS = "LAUNCH_METRICS"
    WAIT_TRUNK = "WAIT_TRUNK"
    META_TEST = "META_TEST"
    SCHEDULE_RELEASE = "SCHEDULE_RELEASE"
    RUN = "RUN"
    ICEFLAME = "ICEFLAME"
    STARTREK_OK_APPROVEMENT = "STARTREK_OK_APPROVEMENT"

    @staticmethod
    def rm_job_name(job_type, component_name="", parametrized=""):
        name = "_{}".format(job_type)
        if component_name:
            name = "{}__{}".format(name, component_name.upper())
        if parametrized:
            name = "{}__{}".format(name, parametrized.upper())
        return name


class PeopleGroups(object):
    RM_MAINTAINERS = "rm_maintainers_group"
    CHAT = "chat"
    RESP = "responsible_for_release"
    NOTIFY_TESTER = "glebov-da"
    ALL = [
        RM_MAINTAINERS,
        CHAT,
        RESP,
        NOTIFY_TESTER,
    ]


# DEPRECATED
# Consider using our event-based notifications https://wiki.yandex-team.ru/releasemachine/notifications/
rm_config = {
    PeopleGroups.RM_MAINTAINERS:
        {
            RMStatus.on_success: [],
            RMStatus.on_break: [
                TaskTypes.ALL_RM_TASKS.name,
                TaskTypes.RELEASE_MONITOR_CRAWL_RELEASES.name,
                TaskTypes.KPI.name,
                TaskTypes.RELEASE_RM_COMPONENT.name,
                TaskTypes.LAUNCH_METRICS.name,
                TaskTypes.CREATE_STARTREK_TICKET.name,
                TaskTypes.CREATE_WIKI_PAGE.name,
                TaskTypes.CREATE_BRANCH_OR_TAG.name,
                TaskTypes.BUILD.name,
                TaskTypes.MERGE_TO_STABLE.name,
            ],
            RMStatus.on_failure: [
                TaskTypes.RELEASE_RM_COMPONENT.name,
                TaskTypes.ALL_RM_TASKS.name,
                # spam disabled, see better solution in RMDEV-2601
                # TaskTypes.RELEASE_MONITOR_CRAWL_RELEASES.name,

            ],
            RMStatus.custom: []
        },
    PeopleGroups.CHAT:
        {
            RMStatus.on_success: {
            },
            RMStatus.on_break: {
                RMNames.USER_SESSIONS: [
                    TaskTypes.USER_SESSIONS.name,
                ],
                RMNames.BALANCER_CONFIG: [
                    TaskTypes.BALANCER_CONFIG.name,
                ],
                RMNames.PRS_OPS: [
                    TaskTypes.PRS_OPS.name,
                ],
                RMNames.ADDRS_BASE: [
                    TaskTypes.ADDR_BASE_SPEC.name,
                ],
                RMNames.ADDRS_META: [
                    TaskTypes.ADDR_BASE_SPEC.name,
                ],
                RMNames.UPPER: [
                    TaskTypes.ACCEPTANCE_NOAPACHE.name,
                ],
            },
            # Remember, launch_metrics for priemka is too noisy with on_failure and pretty useless
            RMStatus.on_failure: {
                RMNames.USER_SESSIONS: [
                    TaskTypes.USER_SESSIONS.name,
                ],
                RMNames.BALANCER_CONFIG: [
                    TaskTypes.BALANCER_CONFIG.name,
                ],
                RMNames.PRS_OPS: [
                    TaskTypes.PRS_OPS.name,
                ],
                RMNames.UPPER: [
                    TaskTypes.ACCEPTANCE_NOAPACHE.name,
                ],
                RMNames.ADDRS_BASE: [
                    TaskTypes.ADDR_BASE_SPEC.name,
                ],
                RMNames.ADDRS_META: [
                    TaskTypes.ADDR_BASE_SPEC.name,
                ],
                RMNames.ENTITYSEARCH_BACKEND: [
                    TaskTypes.ADDR_BASE_SPEC.name,
                ],
                RMNames.DEVOPS: [
                    TaskTypes.DEVOPS.name,
                ],
            },
            RMStatus.custom: {
                RMNames.BALANCER_CONFIG: [
                    TaskTypes.CREATE_STARTREK_TICKET.name,
                ],
                RMNames.BALANCER: [
                    TaskTypes.CREATE_STARTREK_TICKET.name,
                ],
                RMNames.BEGEMOT: [
                    TaskTypes.CREATE_STARTREK_TICKET.name,
                ],
            }
        },
    PeopleGroups.RESP:
        {
            RMStatus.on_success: [],
            RMStatus.on_break: [],
            RMStatus.on_failure: [TaskTypes.RELEASE_RM_COMPONENT.name],
            RMStatus.custom: [],
        },

    PeopleGroups.NOTIFY_TESTER:
        {
            RMStatus.on_success: [],
            RMStatus.on_break: True,
            RMStatus.on_failure: [],
            RMStatus.custom: [],
        }
}


class NotifyPreferences:
    def __init__(self, telegram=None, email=None, q_messenger=None):
        self.telegram = telegram
        self.email = email
        self.q_messenger = q_messenger


# DEPRECATED
# No more users or groups should be added here
# Consider using https://wiki.yandex-team.ru/releasemachine/notifications
RM_USERS = {
    "dkvasov": NotifyPreferences(telegram=132385326),
    "mvel": NotifyPreferences(telegram=161811982),
    "glebov-da": NotifyPreferences(
        telegram=53663143,
        q_messenger="d4a9ff74-b695-4b46-9d62-b3df9075d741_fab58c4a-4b7f-4c74-9ca5-44fffe2c9f78",
    ),
    "gaiduk": NotifyPreferences(telegram=127861159),
    "avitella": NotifyPreferences(telegram=118038400),
    "elshiko": NotifyPreferences(telegram=235176614),
    "bochkarev": NotifyPreferences(telegram=13031052),
    "primorial": NotifyPreferences(telegram=86922937),
    "dima-zakharov": NotifyPreferences(telegram=3241818),
    "rm_emergency": NotifyPreferences(telegram=-1001234787736),
    "rm_maintainers": NotifyPreferences(
        telegram=-1001141031809,
        q_messenger="0/0/ebb66469-c89a-4d66-ba77-d20dfdf15d13",
    ),
    "middle_comp": NotifyPreferences(telegram=-1001131859157),
    "upper_comp": NotifyPreferences(telegram=-1001129063068),
    "base_comp": NotifyPreferences(telegram=-1001116979923),
    "lebedev-aa": NotifyPreferences(telegram=114661447),
    "ilyaturuntaev": NotifyPreferences(telegram=-371494699),
    "iot_dev": NotifyPreferences(telegram=-1001341281788),
    "begemot_comp": NotifyPreferences(telegram=-1001132040627),
    "formula_comp": NotifyPreferences(telegram=-1001116982900),
    "jupiter_comp": NotifyPreferences(telegram=-1001118363001),
    "src_setup_comp": NotifyPreferences(telegram=-1001310632991),
    "entitysearch_comp": NotifyPreferences(telegram=-1001176428519),  # -310380407),
    "geobasesearch_comp": NotifyPreferences(telegram=-1001155709291),
    "rthub_comp": NotifyPreferences(telegram=-1001225418957),
    "apphost-releases": NotifyPreferences(telegram=-1001166749721),
    "frontend-vh": NotifyPreferences(telegram=-235160714),
    "clickdaemon_comp": NotifyPreferences(telegram=-315743765),
    "url_monitoring_comp": NotifyPreferences(telegram=-1001287953445),
    "bigb_eagle_comp": NotifyPreferences(telegram=-1001131014907),
    "prs_ops_comp": NotifyPreferences(telegram=-1001190307909),
    "balancer_config_comp": NotifyPreferences(telegram=-1001365799575),
    "Music Backend": NotifyPreferences(telegram=-1001376721929),
    "feedback_comp": NotifyPreferences(telegram=-1001349100624),
    "medchats_comp": NotifyPreferences(telegram=-1001286703235),
    "rtcc_comp": NotifyPreferences(telegram=-1001438611113),
    "bass_comp": NotifyPreferences(telegram=-1001161003159),
    "user_sessions_comp": NotifyPreferences(telegram=-1001174341921),
    "userfeat_comp": NotifyPreferences(telegram=-1001311109698),
    "yabs-server_comp": NotifyPreferences(telegram=-1001118046399),
    "release_engine_comp": NotifyPreferences(telegram=-1001221488483),
    "pod_agent_comp": NotifyPreferences(telegram=-1001333823477),
    "yp_master_comp": NotifyPreferences(telegram=-379488660),
    "rtmr_comp": NotifyPreferences(telegram=-1001469078932),
    "turbo_devops": NotifyPreferences(telegram=-1001372551255),
    "turbo_rthub_devops": NotifyPreferences(telegram=-1001343451157),
    "maps_infra": NotifyPreferences(telegram=-1001173072578),
    "gebetix": NotifyPreferences(telegram=20660035),
    "weather_releases": NotifyPreferences(telegram=-1001410272376),
    "report": NotifyPreferences(telegram=-1001175598947),
    "cores_comp": NotifyPreferences(telegram=-1001317914515),
    "cores_dev": NotifyPreferences(telegram=-1001232046772),
    "cores_emergency": NotifyPreferences(telegram=-1001167653139),
    "collections_backend": NotifyPreferences(telegram=-1001102398937),
    "collections_backend_releases": NotifyPreferences(telegram=-1001196255796),
    "images_robot": NotifyPreferences(telegram=-1001238270400),
    "travel_releases": NotifyPreferences(telegram=-375038891),
    "images-semidups": NotifyPreferences(telegram=-319322635),
    "afisha-releases": NotifyPreferences(telegram=-334154458),
    "media-crm-releases": NotifyPreferences(telegram=-348772710),
    "adfox-releases": NotifyPreferences(telegram=-1001445787211),
    "release_machine_test": NotifyPreferences(
        telegram=-286496231,
        q_messenger="0/0/ebb66469-c89a-4d66-ba77-d20dfdf15d13",
    ),
    "middle_priemka": NotifyPreferences(telegram=-1001570275709),
    "video_hosting_graphs": NotifyPreferences(telegram=-356178430),
    "balance": NotifyPreferences(telegram=-250579387),
    "xml_search_monitor": NotifyPreferences(
        q_messenger="0/0/e7b496d9-030b-43c0-a741-5ea134d95c65",
    ),
    "dialogovo-releases": NotifyPreferences(telegram=-1001302450824),

    "maps-geo_monitorings": NotifyPreferences(telegram=-1001358152683),
    "maps-infra": NotifyPreferences(telegram=-1001173072578),
    "maps-dataprod": NotifyPreferences(telegram=-1001224313210),
    "maps-masstransit": NotifyPreferences(telegram=-1001346417526),
    "maps-navi_mapkit": NotifyPreferences(telegram=-1001373183455),
    "maps-navi_tech": NotifyPreferences(telegram=-1001465293474),
    "maps-carparks": NotifyPreferences(telegram=-1001217647715),
    "maps-geodata_analysis": NotifyPreferences(telegram=-1001497305530),
    "maps-mapkit": NotifyPreferences(telegram=-1001216113844),
    "maps-navi_platform": NotifyPreferences(telegram=-1001406497154),
    "maps-renderer": NotifyPreferences(telegram=-1001234586175),
    "maps-front": NotifyPreferences(telegram=-311235357),
    "maps-geoquality": NotifyPreferences(telegram=-1001486729987),
    "maps-auto": NotifyPreferences(telegram=-1001202093708),
    "maps-infra_spam": NotifyPreferences(telegram=-1001453466473),
    "tts_release": NotifyPreferences(telegram=-353257098),
    "cvduty": NotifyPreferences(telegram=-1001209107106),
}


class SortOrder:
    ASCENDING = 'ASCENDING'
    DESCENDING = 'DESCENDING'


class CIActionKind:
    ACTION = "action"
    RELEASE = "release"


class CIJMESPathCommon:
    MAJOR_RELEASE_NUM = "${context.version_info.major}"
    MINOR_RELEASE_NUM = "${not_null(context.version_info.minor, `0`)}"
    ARCADIA_URL = "arcadia-arc:/#${context.target_revision.hash}"
    ARCADIA_URL_SVN = "arcadia:/arc/trunk/arcadia@${context.target_revision.number}"
    TARGET_COMMIT_HASH = "${context.target_revision.hash}"


TASK_CI_REG_LOCATIONS = {
    "KOSHER_YA_MAKE": "common/nanny/kosher_ya_make",
    "KOSHER_YA_PACKAGE": "common/arcadia/kosher_ya_package",
    "YA_MAKE_2": "common/arcadia/ya_make",
    "YA_PACKAGE": "common/arcadia/ya_package",
    "YA_PACKAGE_2": "common/arcadia/ya_package_2",
    "RELEASE_RM_COMPONENT_2": "common/releases/release_rm_component_2",
    "RELEASE_RM_COMPONENT": "common/releases/release_rm_component",
    "SCHEDULE_RELEASE": "projects/release_machine/schedule_release",
    "BUILD_ARCADIA_PROJECT_FOR_ALL": "common/arcadia/build_arcadia_project_for_all",
    "BUILD_FINDURL": "projects/findurl/build_findurl",
    "BUILD_IRT_BOT": "projects/irt/build_irt_bot",
    "BUILD_CORES": "projects/cores/build_cores",
    "BUILD_KWYT_PACKAGE": "projects/kwyt/build_kwyt_package",
    "BUILD_VCS_INDEXER": "projects/vcs_indexer/build_vcs_indexer",
    "BUILD_SEARCH": "projects/search/build_search/build_search",
    "LAUNCH_METRICS": "projects/release_machine/launch_metrics",
    "BUILD_RECOMMENDER_MANAGER": "projects/recommender_manager/build_recommender_manager",
    "RUN_ICE_FLAME": "projects/iceflame/run_ice_flame",
    "GENERATE_YAPPY_BETA": "projects/release_machine/generate_yappy_beta",
    "PREPARE_CI_ENVIRONMENT": "projects/release_machine/prepare_ci_environment",
    "IRT_BACKUP_YT_TABLES": "projects/irt/backup_yt_tables",
}

KNOWN_RMTE_COMPONENTS = frozenset({
    # BRANCHED
    "base",
    "middle",
    "upper",
    "begemot",
    "addrs_base",
    "src_setup",
    "addrs_meta",
    "jupiter",
    "ugc_server",
    "ugc_db_backend",
    "images_base",
    "entitysearch",
    "rthub",
    "video_base",
    "app_host",
    "turbo_pages",
    "gemini",
    "expadm",
    "tutor",
    "abt",
    "answers_base",
    "yabs_server",
    "biathlon",
    "turbo_merger",
    "web_graph_mapping",
    "video_graph_mapping",
    "turbo_init",
    "turbo_ddm",
    "images_graph_mapping",
    "frontend_video_hosting",
    "balancer_config",
    "clickdaemon",
    "service_controller",
    "balancer",
    "report",
    "release_machine",
    "user_sessions",
    "bass",
    "weather",
    "flags_provider",
    "ugc_feeder",
    "release_machine_test",
    "rpslimiter",
    "scraper_over_yt",
    "resource_cache_controller",
    "boltalka",
    "collections_backend",
    "yp_dns",
    "expdaemon",
    "apache",
    "yp_service_discovery",
    "bigb_balancer",
    "bigb_buzzard",
    "bigb_eagle",
    "bigb_resharder",
    "vins",
    "yp_idm_role_provider",
    "geosuggest",
    "horizon",
    "yp_master",
    "yp_inet_mngr",
    "rasp_pathfinder_proxy",
    "afisha_tickets_backend",
    "arc_client",
    "yp_cauth_export",
    "rasp_export",
    "rasp_blablacar",
    "fanout",
    "rtc_mediator",
    "media_crm",
    "yp_export",
    "alisa_skills_rec",
    "uniproxy",
    "rasp_morda_backend",
    "data_collector",
    "geobot",
    "lb_subscriber",
    "cs_server",
    "deploy_monitoring_controller",
    "bulbasaur",
    "rasp_api_public",
    "graphs_shared",
    "video_hosting_graphs",
    "balance",
    "setrace",
    "rtc_sipgw",
    "rasp_train_api",
    "rasp_suburban_widget",
    "zephyr",
    "rasp_suburban_wizard_api",
    "rasp_train_wizard_api",
    "rasp_wizard_proxy_api",
    "rasp_morda",
    "begemot_megamind",
    "rasp_http_proxy_cache",
    "rasp_info_center",
    "cofe",
    "avia_wizard",
    "dj_unity",
    "avia_shared_flights_api",
    "rasp_touch",
    "rasp_suggests",
    "rasp_suburban_selling",
    "autosupbs",
    "afisha_librarian",
    "hollywood",
    "rasp_pathfinder_maps",
    "yp_heavy_scheduler",
    "avia_shared_flights_importer",
    "rasp_suburban_tasks",
    "avia_shared_flights_diff_builder",
    "weather_graphs",
    "ydo_searchproxy",
    "avia_shared_flights_status_importer",
    "avia_shared_flights_admin",
    "zen_graphs",
    "common_graphs",
    "quasar_backend",
    "mailcorp_graphs",
    "mail_graphs",
    "drive",
    "voice_graphs",
    "tutor_admin",
    "alice_graphs",
    "rasp_crosslink",
    "afisha_ticketsystem_backend",
    "altay",
    "avia_subscriptions",
    "workplace",
    "pcode_graphs",
    "buses_backend_api",
    "horizontal_pod_autoscaler_controller",
    "auth_controller",
    "music_graphs",
    "video_hosting_src_setup",
    "kinopoisk_ugc_filmrate_api",
    "avia_flight_status_receiver",
    "news_graphs",
    "afisha_draqla",
    "xurma_zen",
    "train_offer_storage",
    "kinopoisk_kp1_api",
    "hollywood_general_conversation",
    "travel_hotels_administrator",
    "rasp_train_bandit_api",
    "begemot_request_init",
    "kinopoisk_person_api",
    "cbirdaemon",
    "xurma_dsp",
    "xurma_market",
    "xurma_taxi",
    "xurma_admod",
    "xurma_edadeal",
    "xurma_btr",
    "xurma_bs",
    "xurma_admet",
    "xurma_afraudita",
    "kinopoisk_kp_graphql_api",
    "xurma_prebilling",
    "xurma_online",
    "videobass",
    "bmgen",
    "registry",
    "avia_partners_proxy",
    "avia_travelers",
    "avia_api_gateway",
    "health_graphs",
    "avia_backend",
    "avia_folk_guide_contest",
    "logistic_dispatcher",
    "apphost_cachedaemon",
    "avia_flight_extras",
    "caesar_resharder",
    "caesar_worker",
    "avia_flight_status_fetcher",
    "avia_revise",
    "kinopoisk_quiz_and_questionnaire",
    "kinopoisk_film_rating_api",
    "avia_price_index",
    "avia_statistics",
    "fury_captcha",
    "antirobot",
    "avia_suggests",
    "avia_feature_flag_api",
    "avia_aeroflot_queue_proxy",
    "avia_yeah",
    "kinopoisk_film_rating_job",
    "fury_rtxaron",
    "kinopoisk_zen_feed_api",
    "geo_graphs",
    "avia_api",
    "ugc_graphs",
    "plus_landing_backend",
    "plus_landing_gift",
    "antiadbtest_graphs",
    "ticket_daemon_api",
    "videorecom_fps",
    "dj_rthub",
    "ticket_daemon",
    "yp_eviction_alerter",
    "sprav_feedback",
    "uac_backend",
    "stat_admin",
    "example_graphs",
    "buses_backend_testcontext",
    "warden",
    "admin",
    "tickenator",
    "fei_graphs",
    "files",
    "caesar_graphs",
    "jobs_graphs",
    "travel_content_admin",
    "travel_content_backend",
    "faas_graphs",
    "ticket_daemon_processing",
    "video_hosting_playlist_service",
    "kinopoisk_graphs",
    "plus_hq_backend",
    "newtest_graphs",
    "lpc_graphs",
    "bigrt_queue_daemon",
    "fury_supermod",
    "blrt",
    "market_graphs",
    "afisha_loyalty_backend",
    "newtestt_graphs",
    "testing_vertical_graphs",
    "newspartner_graphs",
    "mediabilling_static_resolver",
    "plus_city_backend",
    "hiworld_graphs",
    "plus_mc_backend",
    "plus_backyard_backend",
    "embedding_storage",
    "lego_graphs",
    "keyinv_service",
    "remote_storage",
    "bnpl",
    "goods_graphs",
    "plus_cobra",
    "newsrobot_graphs",
    "rasp_bus_wizard_api",
    "myvert_graphs",
    "myvertt_graphs",
    "begemot_advq",
    "video_middle",
    "rrload_graphs",
    "fury_adfox",
    "booktaas_graphs",
    "forms_graphs",
    "garage_graphs",
    "int_l1",
    "ydo_graphs",
    "schoolbook_graphs",
    "smartcamerafront_graphs",
    "laas_graphs",
    "stateful_graphs",
    "crm_graphs",
    "pod_agent",
    "rtmr",
    "savedcopy",
    "adfox",
    "prs_ops",
    "yabs_server_validator",
    "userfeat",
    "url_monitoring",
    "music",
    "yabs_server_l7balancer",
    "images_semidups",
    "itditp_middle",
    "lyncher",
    "supermoderation",
    "video_index",
    "feedback_platform",
    "brands",
    "yabs_bstr",
    "images_robot",
    "passport_api",
    "ontodb",
    "asr_server",
    "floyd",
    "stage_controller",
    "rasp_tasks",
    "tts_server",
    "sawmill",
    "education_checking_system",
    "metasearch",
    "yappy",
    "ydo_request_extractor",
    "tycoon_api",
    "bigb_vulture",
    "mirrors",
    "education_schoolbook_analytics_workflows",
    "balance_dcsaap",
    "rasp_suggests_tasks",
    "dwh",
    "dialogovo",
    "botplatform",
    "tycoon_scheduler",
    "social_api",
    "social_utils_dumpers",
    "social_nginx",
    "social_metrics",
    "social_broker",
    "social_proxy2",
    "social_utils_celerybeat",
    "social_configs",
    "social_utils",
    "social_secrets_yav",
    "district_graphs",
    "education_schoolbook_infrastructure_workflows",
    "education_schoolbook_product_workflows",
    "education_pdf_gen",
    "education_infrastructure_ci_enigma",
    "mediabilling",
    "cupid",
    "education_schoolbook_reactor_automation",
    "education_ext_lms_connector",
    "quasar_yandexstation_2",
    "paskills_billing",
    "toloka_dj",
    "passport_dbscripts",
    "afisha_domiki_backend",
    "vkus_graphs",
    "dropstat_api",
    "dropstat",
    "voicetech_bio_server",
    "media_stories",
    "education_sections",
    "artmon_metric_proxy",
    "direct_monitoring",
    "rm_test_prepare",
    "smart_devices",
    "transcoder",
    "passport_uaas_proxy",
    "irt_lint",
    "test_graphs",
    "smarttv_report",
    "rasp_admin",
    "smarttv_droideka",
    "smarttv_updater",
    "education_crowd_test_yd_proxy",
    "images_report",
    "kd_multik_web",
    "bigb_raven",
    "yaphone_localization_admin",
    "bigb_oracle",
    "plus_landing_oneyear",
    "videorecom_base",
    "yaphone_advisor",
    "yaphone_jafar",
    "yaphone_dialer",
    "smarttv_alice_proxy",
    "irt_multik_web",
    "adv_machine_robot",
    "my_alice_backend",
    "bigb_jaylog",
    "avia_price_prediction",
    "plus_static_resolver",
    "fury_rtbs",
    "fury_rtprebilling",
    "avia_personalization",
    "notificator",
    "dyntable_http_proxy",
    "yandexmidi_releases",
    "yandexmidi_factory",
    "yandexmini_2_factory",
    "avia_partners_proxy_lite",
    "avia_search_results_queue_producer",
    "bs_collector",
    "travel_notifier",
    "yandexmodule_2_factory",
    "rt_user_sessions",
    "agencycabinet_graphs",
    "inverted_index_storage",
    "yandexmicro_factory",
    "multik_cpp_graph",
    "avia_travelers_deploy",
    "avia_backend_deploy",
    "avia_wizard_deploy",
    "avia_price_index_deploy",
    "avia_shared_flights_api_deploy",
    "avia_shared_flights_admin_deploy",
    "avia_shared_flights_diff_builder_deploy",
    "avia_shared_flights_status_importer_deploy",
    "travel_trips_api",
    "travel_trips_collector",
    "travel_trips_processor",
    "avia_suggests_deploy",
    "check_graphs",
    "sport_graphs",
    "ticket_daemon_deploy",
    "rm_test_for_sd",
    "ticket_daemon_api_deploy",
    "fury_market",
    "fury_megaera",
    "avia_fare_families",
    "gvdtest_graphs",
    "report_graphs",
    "marketreport_graphs",
    "avia_aeroflot_queue_proxy_deploy",
    "travel_notifier_processor",
    "begemot_graphs",
    "travel_notifier_collector",
    "pluto_factory",
    "rasp_morda_content",  # https://st.yandex-team.ru/RMDEV-2823#6218f442bbdbbf43585c68c8
    "junk_optozorax_blrt",  # junk, for testing "blrt" component
    "sportnewsrobot_graphs",
    "alicerpc_graphs",
    #
    # TAGGED
    "rearrange_dynamic",
    "exp_formulas_mmeta",
    "exp_formulas_base",
    "release_machine_test_tagged",
    "stoker",
    "zcli",
    "horadric",
    "ya_stroka",
    "logbroker",
    "cbirdaemon_data",
    "apphost",
    "morty",
    "autoru360_poi",
    "bigb_ab",
    "bigb_bsyeti_experiments",
    "bigb_hitlog_analyser",
    "bigb_request",
    "bigb_yt_profiles_dumper",
    "bigb_bb_cluster_analyzer",
    "caesar_lookup_profile",
    "caesar_profile_size_analyzer",
    "caesar_yt_sync",
    "exp_stats",
    "rtcc",
    "tunneller",
    "prod_formulas_flags_test",
    "ab_experiments",
    "yabs_linear_models",
    "ab_flags",
    "ab_flags_testids",
    "bigb_buzzard_cluster_analyzer",
    "bigrt_cli",
    "lookup_profile",
    "profile_size_analyzer",
    "yt_sync",
    #
    # TRUNK
    "userfeat_viewers",
    "woland",
    "jupiter_viewers",
    "yf",
    "smart_devices_firmware",
})

COMPONENT_AYAML_AUTOGEN_BLACKLIST = [
    "agency_rewards",
    "avia_ad_feed",  # old job graph
    "bell_server",
    "berthub",
    "bigb_eagle_ci",
    "chz_recommender",
    "chz_recommender_shard",
    "codesearch",  # old job graph
    "cores",  # old job graph
    "cv_universal_py_daemon",
    "ems",
    "findurl_worker",
    "grut_bundle",
    "grut_object_api",
    "grut_orm",
    "iceflame",  # old job graph
    "irt_bot",
    "k50_shopping",  # old job graph
    "kwyt",
    "memento",  # old job graph
    "request_sampler",
    "saas_ssm",
    "smart_devices_ci",
    "travel_app_backend",
    "xml_counter",
    "yabs_hit_models",
    "zephyr_cli",
]
