from __future__ import unicode_literals

import enum
import re
import os
# from typing import (
# AnyStr,
# Callable,
# List,
# Iterable,
# Optional,
# Union,
# Tuple,
# )
from types import GeneratorType

from sandbox.projects.common.constants import constants as sb_constants
from sandbox.projects.release_machine.core import const as rm_const


class DiskType(enum.Enum):
    HDD = 1
    SSD = 2


class AbstractReleasableItem(object):
    __slots__ = ("name", "data", "deploy_infos")

    def iter_deploy_infos(self, stage=None):
        if self.deploy_infos:
            for i in self.deploy_infos:
                if stage and i.stage != stage:
                    continue
                yield i

    def __str__(self):
        return "{}<{}> {}".format(self.__class__.__name__, self.data.__class__.__name__, self.name)

    __repr__ = __str__


class ReleasableItem(AbstractReleasableItem):
    """
    Simple releasable item.
    """

    def __init__(
        self,
        name,  # type: AnyStr
        data,  # type: Union[SandboxResourceData, DockerImageData]
        deploy_infos=None,  # type: Optional[List[DeployInfo]]
        build_data=None,  # type: Optional[BuildData]
    ):
        self.name = name
        self.data = data
        # Theoretically, there could be a list of elements with different deploy systems
        self.deploy_infos = deploy_infos
        self.build_data = build_data


class DynamicReleasableItem(AbstractReleasableItem):
    """
    Dynamically constructed releasable item.

    Produces list of simple releasable items by resolving itself in runtime.
    """

    def __init__(
        self,
        name,  # type: AnyStr
        data,  # type: SandboxResourceData
        deploy_infos=None,  # type: List[DeployInfo]
    ):
        self.name = name
        self.data = data
        self.deploy_infos = deploy_infos  # deploy infos with empty services and filled stages


class SandboxResourceData(object):
    def __init__(
        self,
        resource_type,
        build_ctx_key=sb_constants.ARCADIA_URL_KEY,
        attributes=None,  # type: Optional[dict]
        dst_path=None,  # type: AnyStr
        disk_type=DiskType.HDD,  # type: DiskType
        ttl=12,  # type: int
        docker_image_data_re=None
    ):
        self.resource_type = resource_type
        self.build_ctx_key = build_ctx_key
        # Could be like {"resource_name": "my_lovely_resource"}.
        # It is useful for resources with same resource_type
        self.attributes = attributes
        self.dst_path = dst_path  # Path of this resource inside deployed instance
        self.disc_type = disk_type
        self.ttl = ttl
        self.docker_image_data_re = docker_image_data_re

    def __str__(self):
        return "SandboxResourceData<{}>".format(self.resource_type)

    __repr__ = __str__


class DockerImageData(object):
    def __init__(
        self,
        build_ctx_key=sb_constants.ARCADIA_URL_KEY,
    ):
        self.build_ctx_key = build_ctx_key


class DeployService(object):
    """Service representation for a deploy system"""
    __slots__ = ("_name", "_tags")

    def __init__(self, name, tags=()):
        self._name = name
        self._tags = tuple(tags)

    @property
    def name(self):
        return self._name

    @property
    def tags(self):
        return self._tags

    def __eq__(self, other):
        return self._name == other.name and self._tags == other.tags

    def __hash__(self):
        return hash("{}, {}".format(self._name, self._tags))

    def __str__(self):
        return "DeployService<'{}', tags={}>".format(self._name, self._tags)

    __repr__ = __str__


class DeployInfo(object):
    """Base class for all deploy systems"""
    __slots__ = ("_services", "_stage", "_chooser")

    def __init__(self, services=None, stage="stable", chooser=min):
        # type: (Optional[Union[Iterable[DeployService], DeployService]], AnyStr, Callable) -> None
        if services is None:
            self._services = ()
        elif isinstance(services, (list, tuple, set, frozenset, GeneratorType)):
            self._services = tuple(services)
        else:
            self._services = (services,)
        self._stage = stage
        self._chooser = chooser

    @property
    def services(self):
        # type: () -> Tuple[DeployService]
        return self._services

    def add_services(self, services):
        self._services += tuple(services)

    @property
    def stage(self):
        # type: () -> AnyStr
        return self._stage

    @property
    def chooser(self):
        # type: () -> Callable
        return self._chooser

    @property
    def deploy_system(self):
        raise NotImplementedError

    def __str__(self):
        return "{}<{}>".format(self.__class__.__name__, self._stage)

    __repr__ = __str__


class YaToolDeployInfo(DeployInfo):
    deploy_system = rm_const.DeploySystem.ya_tool


class SandboxInfo(DeployInfo):
    deploy_system = rm_const.DeploySystem.sandbox


class SandboxKosherReleaseInfo(DeployInfo):
    deploy_system = rm_const.DeploySystem.kosher_sandbox_release


class YaDeployInfo(DeployInfo):
    deploy_system = rm_const.DeploySystem.ya_deploy


class QloudDeployInfo(DeployInfo):
    __slots__ = ("_version_getter", )
    deploy_system = rm_const.DeploySystem.qloud

    def __init__(
        self,
        services=None,
        stage="stable",
        version_getter=re.compile(r'registry\.yandex\.net/(?P<proj>.*)/?(?P<app>.*):(?P<major>\d+)\.(?P<minor>\d+)')
    ):
        super(QloudDeployInfo, self).__init__(services, stage)
        self._version_getter = version_getter

    @property
    def version_getter(self):
        return self._version_getter


class SamogonDeployInfo(DeployInfo):
    deploy_system = rm_const.DeploySystem.samogon


class NannyDeployInfo(DeployInfo):
    __slots__ = ("_dashboards", )
    deploy_system = rm_const.DeploySystem.nanny

    def __init__(self, services=None, stage="stable", dashboards=None):
        super(NannyDeployInfo, self).__init__(services, stage)
        self._dashboards = dashboards or []

    @property
    def dashboards(self):
        return self._dashboards


class NannyPushDeployInfo(NannyDeployInfo):
    deploy_system = rm_const.DeploySystem.nanny_push


class CiTaskletRegistryDeployInfo(DeployInfo):
    deploy_system = rm_const.DeploySystem.ci_tasklet_registry

    TASKLET_REGISTRY_PATH_PREFIX = "ci/registry"

    def __init__(self, registry_paths, stage="stable"):
        # type: (List[str], str) -> None

        self._registry_paths = [self.format_path(path) for path in registry_paths]

        super(CiTaskletRegistryDeployInfo, self).__init__(
            services=[DeployService(path) for path in self._registry_paths],
            stage=stage,
        )

    @property
    def registry_paths(self):  # type: () -> List[str]
        return self._registry_paths

    @classmethod
    def format_path(cls, path):  # type: (str) -> str

        if not path.endswith(".yaml"):
            path = "{}.yaml".format(path)

        if not path.startswith(cls.TASKLET_REGISTRY_PATH_PREFIX):
            path = os.path.join(cls.TASKLET_REGISTRY_PATH_PREFIX, path)

        return path


def single_nanny_service(service, stage="stable"):
    """
    Use in `releasable_items`, i.e.
    @property
    def releasable_items(self):
        return [
            ri.ReleasableItem(
                name="my_resource",
                data=ri.SandboxResourceData("MY_RESOURCE"),
                deploy_infos=[single_nanny_service("production_app_host_vla_common"))],
            ),
        ]
    """
    return NannyDeployInfo(DeployService(service), stage=stage)


def single_nanny_service_push(service, stage="stable"):
    """
    Use in `releasable_items`, i.e.
    @property
    def releasable_items(self):
        return [
            ri.ReleasableItem(
                name="my_resource",
                data=ri.SandboxResourceData("MY_RESOURCE"),
                deploy_infos=[single_nanny_service_push("production_app_host_vla_common"))],
            ),
        ]
    """
    return NannyPushDeployInfo(DeployService(service), stage=stage)


class BuildData(object):

    def __init__(self, target, artifact=None, is_source=False):
        self._target = target
        self._artifact = artifact or target
        self._is_source = is_source

    @property
    def target(self):
        return self._target

    @property
    def artifact(self):
        return self._artifact

    @property
    def is_source(self):
        return self._is_source
