# -*- coding: utf-8 -*-

import logging

from sandbox.projects.release_machine.core import const as rm_const

try:
    from release_machine.public import events as events_public
    from release_machine.common_proto import events_pb2
except ImportError:
    logging.error("Cannot import RM event modules. Are you running a task in a non-binary mode?")


class EventType(object):
    AcceptanceTest = "AcceptanceTest"
    MergeCommit = "MergeCommit"
    RollbackCommit = "RollbackCommit"
    BuildTest = "BuildTest"
    UpdateState = "UpdateState"
    TicketHistory = "TicketHistory"
    GenericTest = "GenericTest"
    ReleaseCreated = "ReleaseCreated"
    ReleaseDeployed = "ReleaseDeployed"
    NewBranch = "NewBranch"
    NewTag = "NewTag"
    TestenvDbClone = "TestenvDbClone"
    ChangelogCreated = "ChangelogCreated"
    ComponentUpdate = "ComponentUpdate"
    ReleaseButton = "ReleaseButton"
    PreReleaseButton = "PreReleaseButton"
    NewWikiPage = "NewWikiPage"
    CustomMessage = "CustomMessage"
    GenericDiff = "GenericDiff"
    NewBetaGeneration = "NewBetaGeneration"
    NewCore = "NewCore"
    ReleaseFailed = "ReleaseFailed"
    ComponentVersionsChange = "ComponentVersionsChange"
    KpiAlert = "KpiAlert"
    ReleaseStarted = "ReleaseStarted"
    MergeStarted = "MergeStarted"
    RollbackStarted = "RollbackStarted"
    NewBetaGenerationStarted = "NewBetaGenerationStarted"
    NewBranchCreationStarted = "NewBranchCreationStarted"
    NewTagCreationStarted = "NewTagCreationStarted"


def get_task_event_general_data(task_obj, hash_items):
    """
    Get general_data for the event spawned by task `ztask_obj`

    :param task_obj: task instance
    :type task_obj: sandbox.sdk2.Task
    :param hash_items: an iterable of items used to build event hash
    :type hash_items: iterable
    :return: events_pb2.EventGeneralData
    """
    return events_pb2.EventGeneralData(
        hash=events_public.get_event_hash(*hash_items),
        component_name=task_obj.Parameters.component_name,
        referrer=u"sandbox_task:{}".format(task_obj.id),
    )


def get_ci_context_task_context_value(task_obj):
    """
    NOTE: sdk2 tasks only!

    Given a task object retrieve the value of its __CI_CONTEXT context field and return either this value
    or None if it does not exist

    :param task_obj: an instance of sdk2.Task
    :return: str or None
    """
    return getattr(task_obj.Context, rm_const.CI_CONTEXT_KEY) or None


def get_task_event_ci_job_context(task_obj):
    """
    NOTE: sdk2 tasks only!

    :param task_obj: an instance of sdk2.Task
    :return: a ci.tasklet.proto.service.TaskletContext message
    """
    ci_context = get_ci_context_task_context_value(task_obj)
    return events_public.get_sandbox_task_ci_job_context(ci_context)


def get_task_event_task_data(task_obj, event_time_utc_iso, status=None):
    """
    Get task_data for the event spawned by task `task_obj`

    :param task_obj: task instance
    :type task_obj: sandbox.sdk2.Task
    :param event_time_utc_iso: event time ISO formatted in UTC timezone
    :type event_time_utc_iso: str
    :param status: sandbox task status
    :return: events_pb2.EventSandboxTaskData
    """
    return events_pb2.EventSandboxTaskData(
        task_id=task_obj.id,
        status=status or task_obj.status,
        created_at=task_obj.created.isoformat(),
        updated_at=event_time_utc_iso,
    )
