from __future__ import unicode_literals


import argparse
import json
import logging
import requests
import os


class ABApi(object):
    TEST_URL = "http://ab.test.yandex-team.ru/"
    PROD_URL = "http://ab.yandex-team.ru/"

    @property
    def api_url(self):
        return (self.TEST_URL if self.__test else self.PROD_URL) + "api/"

    def __init__(self, token=None, timeout=30, ssl_verify=False, test=True):
        self.logger = logging.getLogger(__name__)
        self.__test = test
        self.__token = token
        self.timeout = timeout
        self.headers = self._set_headers(token)
        self.ssl_verify = ssl_verify

    @staticmethod
    def _set_headers(token):
        headers = {'Content-type': 'application/json'}
        if token:
            headers['Authorization'] = 'OAuth ' + token
        return headers

    def _check_response(self, response, request_type):
        """
            Log response url. Fail on bad status codes.
            :return: response in json
        """
        self.logger.debug("Checking response of url: %s", response.url)
        self.logger.debug("Response headers: %s", response.headers)
        self.logger.debug('Return code: %s. Content:\n%s', response.status_code, response.content)
        if response.status_code in {requests.codes.ok, requests.codes.created}:
            return response.json()
        else:
            self.logger.error('Return code %s: %s', response.status_code, response.content)
            response.raise_for_status()

    def _do_get(self, url):
        r = requests.get(
            self.api_url + url,
            timeout=self.timeout,
            headers=self.headers,
            verify=self.ssl_verify
        )
        return self._check_response(r, "GET")

    def _do_post(self, url, data=None):
        r = requests.post(
            self.api_url + url,
            timeout=self.timeout,
            data=data,
            headers=self.headers,
            verify=self.ssl_verify,
        )
        return self._check_response(r, "POST")

    def create_test_id(self, data):
        return self._do_post("testid", json.dumps(data))

    def deploy_test_id(self, data):
        return self._do_post("deploying/item", json.dumps(data))

    def get_experiment_info(self, experiment_id):
        url = os.path.join("task", experiment_id)
        return self._do_get(url)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Console client for AB")
    subparsers = parser.add_subparsers(help='Commands')

    # get-event
    create_testid_parser = subparsers.add_parser('create-testid', help='Create test id')
    create_testid_parser.add_argument("-d", "--data", help="Data for test id")
