import json
import logging
import requests

import sandbox.projects.release_machine.core.const as rm_const


class ArcanumApi(object):
    BASE_URL = rm_const.Urls.A_URL
    API_URL = BASE_URL + "api/"

    def __init__(self, token=None, timeout=30, ssl_verify=False):
        self.logger = logging.getLogger(__name__)
        self.__token = token
        self.timeout = timeout
        self.headers = self._set_headers(token)
        self.ssl_verify = ssl_verify

    @staticmethod
    def _set_headers(token):
        headers = {
            "Content-Type": "application/json",
            "User-Agent": "Release machine"
        }
        if token:
            headers["Authorization"] = "OAuth " + token
        return headers

    def _check_response(self, response, request_type):
        """
            Log response url. Fail on bad status codes.
            :return: response in json
        """
        self.logger.debug("Checking response of url: %s", response.url)
        self.logger.debug("Response headers: %s", response.headers)
        self.logger.debug("Return code: %s. Content:\n%s", response.status_code, response.content)
        if response.status_code in {requests.codes.ok, requests.codes.created}:
            return response.json()
        else:
            self.logger.error("{} request fails with status code {}: {}".format(
                request_type, response.status_code, response.content
            ))
            return None

    def _do_get(self, url, params=None):
        r = requests.get(
            self.API_URL + url,
            timeout=self.timeout,
            headers=self.headers,
            verify=self.ssl_verify,
            params=params,
        )
        return self._check_response(r, "GET")

    def _do_post(self, url, data=None, files=None):
        r = requests.post(
            self.API_URL + url,
            timeout=self.timeout,
            data=json.dumps(data) if data else None,
            headers=self.headers,
            verify=self.ssl_verify,
            files=files,
        )
        return self._check_response(r, "POST")

    def _do_put(self, url, data=None, files=None):
        r = requests.put(
            self.API_URL + url,
            timeout=self.timeout,
            data=json.dumps(data) if data else None,
            headers=self.headers,
            verify=self.ssl_verify,
            files=files,
        )
        return self._check_response(r, "PUT")

    def get_pr(self, pr_id):
        return self._do_get("v1/pullrequest/{}".format(pr_id))

    def get_review_request(self, rr_id):
        return self._do_get("review/review-request/{}".format(rr_id))

    def get_review_tests_statuses(self, review_id):
        return self._do_get("review/review-request/{}/commit-button".format(review_id))

    def get_st_tickets_for_commits(self, commit):
        r = self._do_get("v1/merge-commits/{}/review-requests?fields=issues".format(commit))
        if "data" in r:
            return r["data"][0]["issues"]
        else:
            self.logger.error("Failed to get st tickets for commit %s: %s", commit, r.get("errors"))
        return []

    def get_dashboard(self, params):
        return self._do_get("review/dashboard/", params)

    def comment_review(self, rr_id, comment):
        self._do_post(
            "v1/review-requests/{}/comments".format(rr_id),
            data={"content": comment}
        )

    def update_review_checks(self, rr_id, content):
        """
        Update review checks values.

        :param rr_id: review id
        :param content: dict with keys `system` (`ci`, `arcanum`), `type` and `required` (boolean).
        """
        self._do_post(
            "v1/review-requests/{}/checks".format(rr_id),
            data=content
        )

    def enable_commit_button(self, rr_id):
        """
        Disable review checks, commit review.

        :param rr_id: review id
        """
        self.headers.pop("Content-Type")
        self._do_put(
            "review/review-request/{}/commit-button".format(rr_id),
            files={
                "autoCommit": ("", "true"),
            },
        )
        self.headers["Content-Type"] = "application/json"

    def create_pr(self, data=None):
        return self._do_post("v1/pullrequest", data)

    def close_review(self, rr_id):
        self._do_put(
            "v1/review-requests/{}/state".format(rr_id),
            data={"state": "closed"}
        )
