from __future__ import unicode_literals

import six

from sandbox.projects.common import link_builder as lb
from sandbox.projects.release_machine import changelogs as ch
from sandbox.projects.release_machine import table_formatter
from sandbox.projects.common.testenv_client import TEClient


class ChangeLogData(table_formatter.TableData):
    def __init__(self, c_info, header, changelog):
        self._c_info = c_info
        rows = [
            tuple(self._format_inner_cell(c_info, i, row[i]) for i in header) for row in changelog
        ]
        super(ChangeLogData, self).__init__(tuple(i.upper() for i in header), rows)

    @staticmethod
    def _format_inner_cell(c_info, cell_type, cell_content):
        raise NotImplementedError


class ChangeLogDataWiki(ChangeLogData):
    @staticmethod
    def _format_inner_cell(c_info, cell_type, cell_content):
        if cell_type == ch.ChangeLogEntry.Attrs.REVISION:
            return lb.revision_link(cell_content, link_type=lb.LinkType.wiki)
        elif cell_type == ch.ChangeLogEntry.Attrs.REVIEW_IDS:
            return [lb.review_link(i, link_type=lb.LinkType.wiki) for i in cell_content]
        elif cell_type == ch.ChangeLogEntry.Attrs.COMMIT_AUTHOR:
            return "staff:{}".format(cell_content)
        elif cell_type == ch.ChangeLogEntry.Attrs.SUMMARY:
            return "\n".join("%%{}%%".format(i) if i else i for i in cell_content.split("\n"))
        elif cell_type == ch.ChangeLogEntry.Attrs.STARTREK_TICKETS:
            return "\n".join("%%{}%%".format(i) for i in cell_content if i)
        elif cell_type == ch.ChangeLogEntry.Attrs.TE_PROBLEMS:
            result = []
            unresolved_mark = []
            for comment, problem_info in cell_content.items():
                links = []
                for p in problem_info:
                    link = TEClient.te_diff_screen(c_info.testenv_cfg__trunk_db, p["te_diff_id"])
                    if not p["resolved"]:
                        link = "??{}??".format(link)
                        if not unresolved_mark:
                            unresolved_mark = ["!!There are unresolved problems!!"]
                    links.append(link)
                result.append("{}: {}".format(comment, ", ".join(links)))
            return unresolved_mark + result
        return cell_content


class ChangeLogDataHtml(ChangeLogData):
    @staticmethod
    def _format_inner_cell(c_info, cell_type, cell_content):
        if cell_type == ch.ChangeLogEntry.Attrs.REVISION:
            return lb.revision_link(cell_content)
        elif cell_type == ch.ChangeLogEntry.Attrs.REVIEW_IDS:
            return [lb.review_link(i, link_type=lb.LinkType.href) for i in cell_content]
        elif cell_type == ch.ChangeLogEntry.Attrs.COMMIT_AUTHOR:
            return lb.staff_link(cell_content)
        elif cell_type == ch.ChangeLogEntry.Attrs.TE_PROBLEMS:
            result = []
            for comment, problem_info in cell_content.items():
                links = []
                for p in problem_info:
                    link = TEClient.te_diff_screen(c_info.testenv_cfg__trunk_db, p["te_diff_id"], wiki=False)
                    link = lb.HREF_TO_ITEM.format(link=link, name=p["te_diff_id"])
                    links.append(link)
                result.append("{}: {}".format(comment, ", ".join(links)))
            return result
        return cell_content


class ChangeLogWiki(table_formatter.TableFormatterWiki):
    def _format_cell(self, cell_content):
        if isinstance(cell_content, (list, tuple)):
            cell_content = "\n".join(map(six.text_type, cell_content))
        elif isinstance(cell_content, dict):
            cell_content = "\n".join(
                "**{}**: {}".format(k, v) for k, v in cell_content.items()
            )
        return super(ChangeLogWiki, self)._format_cell(cell_content)


class ChangeLogHtml(table_formatter.TableFormatterHtml):
    def _format_cell(self, cell_content):
        if isinstance(cell_content, (list, tuple)):
            cell_content = "<br/>".join(map(str, cell_content))
        elif isinstance(cell_content, dict):
            cell_content = "<br/>".join(
                "<strong>{}</strong>: {}".format(k, v) for k, v in cell_content.items()
            )
        return super(ChangeLogHtml, self)._format_cell(cell_content)
