# -*- coding: utf-8 -*-
from __future__ import unicode_literals


import logging
import os
import re
import time
import textwrap

from sandbox.common import errors
from sandbox.projects.common import decorators
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine import notify_helper
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers import arcanum_helper
from sandbox.projects.release_machine.helpers import merge_helper
from sandbox.projects.release_machine.helpers import svn_helper
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.sdk2 import paths
from sandbox.sdk2.vcs import svn

#  Different directories we checkout with different depth. We need this because checkout every directory in path is
#  longer than checkout some directory with --set-depth=infinity. These depths include the directories themselves.
checkout_depth = {
    'data': 4,
    'arcadia': 3,
    'arcadia_tests_data': 3,
}


tests_names = {
    "needCiBuild": "buildStatus",
    "needCiTests": "testsStatus",
    "needCiLargeTests": "largeTestsStatus",
    "needTeJobs": "teJobsStatus",
}


def _unescape_descr(descr):
    escapes = {
        '&gt;': '>',
        '&lt;': '<',
        '&quot;': "'",
    }
    pattern = "(" + "|".join(escapes.keys()) + ")"
    return re.sub(pattern, lambda matchobj: escapes.get(matchobj.group(0)), descr)


def _prepare_descr(descr):
    descr = string.all_to_str(descr)
    # SEARCH-1934
    descr = re.sub(r'<a href="([^" ]+)">(.+?)</a>', ' \\1 (\\2) ', descr)
    # SEARCH-2200
    descr = re.sub(rm_const.REVIEW_REPLACE_RE, 'REVIEW ID: \\2', descr)
    # RMDEV-458
    descr = _unescape_descr(descr)
    return descr


def check_or_complete_custom_path(custom_path):
    """
        Checks svn path correctness. Path could be "arcadia:arc/branches/middle/,,,",
        "branches/middle/..." or just "middle/..." or "arcadia:arc/trunk/data/..." or "trunk/data/..." or "data/...".
        :param custom_path: Path to branch
        :return: full path to branch in arcadia
    """
    custom_path = custom_path.lstrip('/')
    if svn.Arcadia.check(custom_path):
        return custom_path
    if svn.Arcadia.check('arcadia:/arc/branches/' + custom_path):
        return 'arcadia:/arc/branches/' + custom_path
    if svn.Arcadia.check('arcadia:/arc/' + custom_path):
        return 'arcadia:/arc/' + custom_path
    if svn.Arcadia.check('arcadia:/arc/trunk/' + custom_path):
        return 'arcadia:/arc/trunk/' + custom_path
    eh.check_failed("You've entered wrong path!")


def format_diff_resolver_msg(commit_info):
    if not commit_info.diff_resolvers:
        return ''
    return ' {}{}]'.format(rm_const.DIFF_RESOLVER_MASK, commit_info.diff_resolvers[0])


def _find_depth_to_checkout(common_path):
    """
    Count depth for partial_checkout. Depth is counted from one of checkout_depth dirs.
    For example, for "/arcadia/web" depth = 2.
    :param common_path: Max common path for all paths for our revisions,
    starts from "arcadia", "data" or "arcadia_test_data".
    :return: -1 for wrong situation, 0 if we don't need partial_checkout, depth > 0 else.
    """
    if len(common_path) == 0:
        # We have paths in different root dirs, for example, arcadia and arcadia_test_data
        # 5 because for 'data' dir we have checkout depth 4, so here we need to checkout 'data' => add 1 to depth
        return 5
    elif len(common_path) >= checkout_depth[common_path[0]]:
        return 0
    else:
        return checkout_depth[common_path[0]]


def _update_dirs_from_path(merge_path, path, common_path, curr_dir_rev):
    """
    Updates all dirs in path with set_depth = immediates and whole path with set_depth = infinity.

    :param merge_path: CommitPath entity
    :param path: path to checkout
    :param common_path: Max common path for all paths for our revisions
    :param curr_dir_rev: Revision to update
    :return:
    """
    dirs = path.split('/')
    if len(common_path) > 0:
        dirs = dirs[dirs.index(common_path[-1]) + 1:]
    else:
        root_dir_indices = [dirs.index(root_dir) for root_dir in checkout_depth.keys() if root_dir in dirs]
        if not root_dir_indices:
            eh.check_failed("Can't get root dir for path %s", path)
        dirs = dirs[min(root_dir_indices):]
    logging.debug("Len dirs %s, dirs %s", len(dirs), dirs)
    if len(common_path) > 1:
        dirs_in_path = os.listdir(merge_path.local)
        if len(dirs_in_path) > 0:
            logging.debug("We have path %s already updated\nDirs in path: %s", merge_path.local, dirs_in_path)
            return
        svn.Arcadia.update(merge_path.local, revision=curr_dir_rev, set_depth='infinity')
    else:
        # I.e. common_path = `arcadia`, dirs = `ya.make`. In this case we have already checkouted this file.
        # Or there are paths `arcadia_tests_data/blender/misspell/...` and `arcadia/sandbox/projects/ ...`.
        # We should update only `arcadia_tests_data/blender/` and `arcadia/sandbox/` dirs with depth infinity
        depth_to_update = 2 - len(common_path)
        if len(dirs) == depth_to_update:
            logging.debug("Maybe we have just one file, should checkout with depths=immediates")
            entry_kind = svn.Arcadia.info(
                os.path.join(merge_path.path_from(), *(common_path + dirs)) + "@{}".format(curr_dir_rev)
            )["entry_kind"]
            if entry_kind == 'file':
                logging.debug("This file has been already checked out %s", dirs)
                return

        # RMDEV-3205
        if dirs[:depth_to_update] == ["junk"]:
            logging.info("Forcing independent junk items update")

            svn.Arcadia.update(
                os.path.join(merge_path.local, *dirs[:depth_to_update]),
                revision=curr_dir_rev,
                set_depth='immediates',
            )

            depth_to_update += 1

        path_to_update = os.path.join(merge_path.local, *dirs[:depth_to_update])
        dirs_in_path = os.listdir(path_to_update)

        logging.info("Got path to update: %s", path_to_update)

        if len(dirs_in_path) > 0:
            logging.debug("We have path %s already updated\nDirs in path: %s", path_to_update, dirs_in_path)
            return

        svn.Arcadia.update(path_to_update, revision=curr_dir_rev, set_depth='infinity')


def _update_dirs_from_rev(merge_path, rev_for_paths, common_path, curr_dir_rev):
    """
    Gets all paths from revision and updates them to revision.
    :param merge_path: CommitPath entity
    :param rev_for_paths: Current revision to get paths from
    :param common_path: Max common path for all paths for our revisions
    :param curr_dir_rev: Revision to update
    :return:
    """
    t = svn.Arcadia.log(merge_path.path_from(), rev_for_paths)
    logging.debug("Add all paths logs for revision %s: %s", rev_for_paths, t)
    for _, path in t[0]['paths']:
        _update_dirs_from_path(
            merge_path=merge_path,
            path=path,
            common_path=common_path,
            curr_dir_rev=curr_dir_rev,
        )


def _try_partial_checkout(merge_path, revs, common_path, checkout_rev=None):
    try:
        svn.Arcadia.checkout(
            os.path.join(merge_path.path_to(), *common_path),
            merge_path.local,
            revision=checkout_rev,
            depth='immediates',
        )
        curr_dir_rev = svn.Arcadia.info(merge_path.local)['entry_revision']
        logging.debug("rev: %s", curr_dir_rev)
        svn.Arcadia.update(merge_path.local, depth='immediates', revision=curr_dir_rev)
        if len(common_path) == 0:
            logging.debug((
                "We have several paths from `arcadia`, `data` or `arcadia_tests_data`, "
                "checkout them with depth immediates"
            ))
            for root_dir in checkout_depth:
                path_to_update = os.path.join(merge_path.local, root_dir)
                svn.Arcadia.update(path_to_update, revision=curr_dir_rev, set_depth='immediates')
        for rev_for_paths in revs:
            _update_dirs_from_rev(merge_path, rev_for_paths, common_path, curr_dir_rev)

        # partial checkout is OK
        return True

    except Exception as e:
        eh.log_exception("Partial checkout failed. Will try regular one.", e)
        paths.remove_path(merge_path.local)

    # partial checkout failed for some reason
    return False


def get_arc_ssh_key(task):
    token_users = [rm_const.COMMON_TOKEN_OWNER, task.author, task.owner]
    ssh_key = None
    for user in token_users:
        try:
            ssh_key = svn_helper.get_ssh_key(task, key_owner=user, key_name=rm_const.ARC_ACCESS_TOKEN_NAME)
        except (errors.TaskFailure, errors.VaultNotAllowed, errors.VaultNotFound):
            task.set_info("Can't get {} for user '{}'".format(rm_const.ARC_ACCESS_TOKEN_NAME, user))
        else:
            break
    eh.ensure(ssh_key is not None, "arc_access_key must exist in SB vault for user or task owner.")
    return ssh_key


def create_commit_message(task, merge_path, revs, diff_resolver_msg, descr, commit_user, need_check):
    te_marker = ""
    # here hasattr is the only safe way to get parameter, that could not be in task
    if getattr(task.Parameters, "set_te_marker", False):
        component = (
            getattr(task.Parameters, "component_name_in_trunk", None) or
            getattr(task.Parameters, "component_name", None) or
            getattr(task.Parameters, "component_name_to_merge", None)
        )
        if component:
            te_marker = "[fix:{}:{}]".format(component, min(revs))
    if isinstance(revs[0], int):
        revs_str = ", ".join(["r{}".format(r) for r in revs])
    else:
        revs_str = ", ".join(revs)
    whole_commit_message = (
        "{revs_action} {revs}\n"
        # begin of marker line. Please do not put square brackets as they are not removed by hooks.
        # after removal, this commit line will contain some whitespace after, it is OK
        "{diff_resolver_msg} "
        "{mark_as_beta} "
        "{skip_review} "
        "{skip_check} "
        # end of marker line
        "Sandbox task: {task_link}\n"
        "Task author: {actor}@\n"
        "{custom_commit_user}"
        "Description: {descr}\n"
        "{te_marker}".format(
            revs_action=merge_helper.Action.COMMIT_HEAD[merge_path.action],
            revs=revs_str,
            diff_resolver_msg=diff_resolver_msg,
            mark_as_beta=" [rm:beta]" if getattr(task.Parameters, "mark_as_beta", False) else "",
            skip_review="SKIP_REVIEW SKIP_CHECK" if not need_check else " ",
            skip_check="__BYPASS_CHECKS__" if not need_check else " ",
            task_link=lb.task_link(task.id, plain=True),
            actor=string.all_to_str(task.author),
            custom_commit_user="Commit was made by robot-srch-releaser, but committer is {user}\n".format(
                user=commit_user
            ) if commit_user else "",
            descr=descr,
            te_marker=te_marker,
        )
    )
    logging.info("Formatted commit message:\n%s", whole_commit_message)
    return whole_commit_message


def commit_changes(task, merge_path, ssh_key, whole_commit_message):  # commit_user  Leave it here
    with ssh_key:
        # Don't use it, because robot-srch-releaser is not allowed to commit as user
        # revprops = ['arcanum:username={}'.format(commit_user)]
        revprops = None

        commit_result = svn.Arcadia.commit(
            merge_path.local,
            whole_commit_message,
            user=rm_const.ROBOT_RELEASER_USER_NAME,
            with_revprop=revprops,
        )
        committed_revision = None
        committed_revision_match = re.findall(r"Committed revision ([0-9]+)", commit_result)
        if committed_revision_match:
            committed_revision = committed_revision_match[0]
        task.set_info("Committed revision {}".format(committed_revision))
    return committed_revision


def find_root_dir(common_path):
    """
    Search root dir to checkout among the "arcadia", "data" and "arcadia_test_data".
    :param common_path: list of dirs for max common path for out revisions.
    :return: index of root dir in common_path list if exists. Else -1.
    """
    main_dir_index = -1
    for i in range(len(common_path)):
        if common_path[i] in checkout_depth:
            main_dir_index = i
            break
    return main_dir_index


def checkout_and_merge(
    task,
    merge_path,
    common_path,
    revs,
    checkout_rev,
    ignore_ancestry,
):
    """
    Checkout all needed paths for revs to merge_path.local and merge revs diff in this path.
    :param task: Task which performs action
    :param merge_path: MergePath entity
    :param common_path: Common_path for all revs
    :param revs: List of revisions to merge
    :param checkout_rev: Revision to checkout (if None than it will be the last rev)
    :param ignore_ancestry: Flag for force merge
    :return: None
    """
    change_revs = merge_path.format_revs(revs)
    checked_out = False
    root_dir_index = find_root_dir(common_path)
    if root_dir_index == -1:
        common_path = []
    else:
        common_path = common_path[root_dir_index:]
    logging.debug("Now common path is %s", common_path)
    if len(common_path) <= 1:
        # common_path is ['arcadia'] or ['data'], too long to download. Trying partial checkout
        checked_out = _try_partial_checkout(merge_path, revs, common_path, checkout_rev=checkout_rev)

    if not checked_out:
        notify_helper.telegram(
            task,
            "Partial checkout failed in task {}".format(lb.task_link(task.id, plain=True)),
            chat_ids=["rm_maintainers"],
        )
        svn.Arcadia.checkout(os.path.join(merge_path.path_to(), *common_path), merge_path.local, revision=checkout_rev)

    try:
        svn.Arcadia.merge(
            os.path.join(merge_path.path_from(), *common_path),
            merge_path.local,
            change=change_revs,
            ignore_ancestry=ignore_ancestry
        )
    except svn.SvnTemporaryError as exc:
        # See SEARCH-4714
        task.set_info("Got temporary error from SVN: {}, restart task".format(exc))
        raise
    except Exception as exc:
        eh.log_exception("Got exception while merging to path {path}".format(path=merge_path.local), exc)

    # SEARCH-2526
    status_str = svn.Arcadia.status(merge_path.local).strip()
    commit_status = svn_helper.process_svn_status_output(
        task,
        status_str,
        path=os.path.join(merge_path.path_to(), *common_path),
    )
    return rm_const.RepoActionResult(commit_status)


def commit(
    task,
    merge_path,
    revs,
    do_commit,
    diff_resolver_msg,
    descr,
    commit_user,
):
    """
    Check that merge status is not empty and commit merge_path.local if necessary
    :param task: Task which performs action
    :param merge_path: MergePath entity
    :param revs: List of revisions to merge
    :param do_commit: Bool, True if need commit else False
    :param diff_resolver_msg:  set, diff_resolvers from all revs
    :param descr: string, parameter from task
    :param commit_user: User name, to be commit author
    :return: CommitResult object
    """

    ssh_key = get_arc_ssh_key(task)
    whole_commit_message = create_commit_message(
        task=task,
        merge_path=merge_path,
        revs=revs,
        diff_resolver_msg=diff_resolver_msg,
        descr=descr,
        commit_user=commit_user,
        need_check=False,
    )

    committed_revision = None
    if do_commit:
        committed_revision = commit_changes(
            task,
            merge_path,
            ssh_key,
            whole_commit_message,
            # commit_user,
        )
        task.set_info("Succeeded {} for path {} of {} to {}".format(
            merge_path.action, merge_path.local, revs, merge_path.short
        ))
    else:
        task.set_info("Dry run mode, do nothing...")
    return rm_const.CommitResult(rm_const.CommitStatus.success, committed_revision)


def create_review(
    task,
    merge_path,
    common_path,
    revs,
    checkout_rev,
    ignore_ancestry,
    commit_info,
    descr,
    commit_user,
):
    diff_resolver_msg = format_diff_resolver_msg(commit_info)
    try:
        merge_result = checkout_and_merge(
            task=task,
            merge_path=merge_path,
            common_path=common_path,
            revs=revs,
            checkout_rev=checkout_rev,
            ignore_ancestry=ignore_ancestry,
        )
        if merge_result.status != rm_const.CommitStatus.success:
            logging.info("Got status `%s` after checkout_and_merge, not creating review", merge_result.status)
            return rm_const.ReviewResult(merge_result.status, None)
    except svn.SvnTemporaryError as exc:
        # See SEARCH-4714
        task.set_info("Got temporary error from SVN: {}, restart task".format(exc))
        raise

    except Exception as exc:
        message = "There was an error during checkout and merge changes for path {} of {}: {}".format(
            merge_path.local, revs, exc
        )
        if isinstance(exc, errors.VaultNotFound):
            message += (
                ". Probably you need to configure an access for the group you are using "
                "as task owner ('{}'). See "
                "https://wiki.yandex-team.ru/ReleaseMachine/create-new-release-machine/merges "
                "for details or report to SEARCH-5679 (with reopen) if it does not help. "
            ).format(task.owner)

        task.set_info(message)
        eh.log_exception(message, exc, task=task)
        return rm_const.ReviewResult(rm_const.CommitStatus.failed, None)

    ssh_key = get_arc_ssh_key(task)
    whole_commit_message = create_commit_message(
        task=task,
        merge_path=merge_path,
        revs=revs,
        diff_resolver_msg=diff_resolver_msg,
        descr=descr,
        commit_user=commit_user,
        need_check=True,
    )

    # Create review and return its id
    review_id = commit_and_create_review(
        merge_path,
        ssh_key,
        whole_commit_message,
        commit_user,
    )
    paths.remove_path(merge_path.local)
    if not review_id:
        return rm_const.ReviewResult(rm_const.CommitStatus.failed, None)
    else:
        return rm_const.ReviewResult(rm_const.CommitStatus.success, review_id)


def check_review_tests_statuses(review_info):
    for ci_test_name, test_status in tests_names.items():
        if (review_info["conditions"][ci_test_name] and
                review_info["ciCheck"][test_status] == rm_const.CiTestResult.failed.value):
            logging.error("Got test %s enable but it is broken", ci_test_name)
            return rm_const.CiTestResult.failed

    for ci_test_name, test_status in tests_names.items():
        if review_info["conditions"][ci_test_name]:
            if review_info["ciCheck"][test_status] == rm_const.CiTestResult.success.value:
                logging.info("Got test %s result success", ci_test_name)
            elif review_info["ciCheck"][test_status] == rm_const.CiTestResult.running.value:
                logging.info("Got test %s running", ci_test_name)
                return rm_const.CiTestResult.running
    logging.info("All required tests are in success status")
    return rm_const.CiTestResult.success


def wait_review_for_merge(review_id, arcanum_api):
    if not review_id:
        logging.info("Got empty review_id")
        return rm_const.CommitResult(rm_const.CommitStatus.changeset_is_empty, None)

    for i in range(10):
        logging.debug("Waiting for 30 seconds")
        time.sleep(30)
        review_status = arcanum_api.get_review_request(review_id)["status"]
        if review_status not in {"pending_review", "submitted"}:
            logging.error("Got strange review status %s", review_status)
            return rm_const.CommitResult(rm_const.CommitStatus.failed, None)
        if review_status == "submitted":
            logging.info("Review was merged")
            committed_revision = arcanum_api.get_review_request(review_id)["commits"][0]["committedAtRevision"]
            return rm_const.CommitResult(rm_const.CommitStatus.success, committed_revision)
    logging.error("Waiting too long for review merge, failing")
    return rm_const.CommitResult(rm_const.CommitStatus.failed, None)


@decorators.retries(max_tries=2, delay=5, default_instead_of_raise=True)
def commit_and_create_review(merge_path, ssh_key, whole_commit_message, commit_user):
    with ssh_key:
        revprops = ["arcanum:json=yes", "arcanum:check=yes", "arcanum:review-skip=yes"]
        created_review = None
        try:
            commit_result = svn.Arcadia.commit(
                merge_path.local,
                whole_commit_message,
                user=rm_const.ROBOT_RELEASER_USER_NAME,
                with_revprop=revprops,
            )
            logging.debug("Got commit_result while creating review: %s", commit_result)
        except svn.SvnError as exc:
            created_review = None
            logging.exception("Got exception, while committing %s", exc)
            if exc.error_code == "E165001":
                created_review = re.findall(
                    r"Check status can be monitored using this special review request: ([0-9]+)",
                    str(exc),
                )
            elif exc.error_code in ["E170013", "E210002"]:
                logging.error("There are network problems, retry")
                raise exc

        if created_review:
            created_review = created_review[0]
            logging.info("Created review %s", created_review)
            return created_review
        logging.error("Can't create review, failing")
        return None


def post_commit_actions(task, merge_path, commit_info, commit_failed, revs, committed_revision, c_info=None):
    try:
        notice = merge_helper.Action.NOTIFY[merge_path.action](merge_path, revs, committed_revision is None)

        no_issue_url = 'n/a'

        if c_info is None:
            te_trunk_db = None
            issue_url = no_issue_url
        else:
            te_trunk_db = c_info.testenv_cfg__trunk_db
            st_auth_token = rm_sec.get_rm_token(task)
            st_helper = STHelper(st_auth_token)

            issue_ticket = st_helper.find_ticket_by_release_number(merge_path.branch_num, c_info, fail=False)
            issue_url = lb.st_link(issue_ticket.key) if issue_ticket else no_issue_url

        notify_helper.email2(
            task,
            recipients=commit_info.diff_resolvers + commit_info.committers + [task.commit_user],
            subject=notice.short(commit_failed),
            body=textwrap.dedent(
                """
                Dear {diff_resolvers}!

                {mail_action_body}
                Committed revision: {committed_revision}
                TestEnv: {te_trunk_db_link}
                Ticket(s): {issue_urls}
                """
            ).format(
                diff_resolvers=', '.join(commit_info.diff_resolvers),
                mail_action_body=notice.long(commit_failed),
                committed_revision=committed_revision if committed_revision else "n/a",
                te_trunk_db_link=rm_const.Urls.te_db_screen(te_trunk_db) if te_trunk_db else "n/a",
                issue_urls=issue_url,
            ),
        )
    except Exception as exc:
        eh.log_exception("Got exception while performing post commit actions", exc)


def perform_action(
    task,
    common_path,
    revs,
    descr,
    commit_info,
    merge_path,
    do_commit=True,
    ignore_ancestry=False,
    checkout_rev=None,
    c_info=None,
    commit_user="",
):
    """
    New version of perform. Actually do commit/merge
    :param task: Task, which called perform
    :param common_path: Common path for revisions
    :param revs: List of revisions to merge
    :param descr: Description for commit_message
    :param commit_info: Instance of svn_helper.CommitInfo
    :param merge_path: MergePath entity
    :param do_commit: True if we want to commit, False else.
    :param ignore_ancestry: Flag for force merge
    :param checkout_rev: Revision to checkout (if None than it will be the last rev)
    :param c_info: Component info class instance
    :param commit_user: User name, to be commit author
    :return: Commit status(enum) and committed revision on success, None otherwise (merge failed or empty changeset)
    """
    descr = _prepare_descr(descr)
    diff_resolver_msg = format_diff_resolver_msg(commit_info)

    commit_failed = False
    # will be set if merged changeset is trivial
    committed_revision = None

    try:
        merge_result = checkout_and_merge(
            task=task,
            merge_path=merge_path,
            common_path=common_path,
            revs=revs,
            checkout_rev=checkout_rev,
            ignore_ancestry=ignore_ancestry,
        )
        if merge_result.status == rm_const.CommitStatus.success:
            result = commit(
                task=task,
                merge_path=merge_path,
                revs=revs,
                do_commit=do_commit,
                diff_resolver_msg=diff_resolver_msg,
                descr=descr,
                commit_user=commit_user,
            )
            committed_revision = result.revision
        else:
            # commit only if some changes detected
            logging.info("Got status `%s` after checkout_and_merge, not committing", merge_result.status)
            if merge_result.status == rm_const.CommitStatus.failed:
                commit_failed = True
            result = rm_const.CommitResult(merge_result.status, None)
    except svn.SvnTemporaryError as exc:
        # See SEARCH-4714
        task.set_info("Got temporary error from SVN: {}, restart task".format(exc))
        raise

    except Exception as exc:
        commit_failed = True
        message = "There was an error during {} for path {} of {}: {}".format(
            merge_path.action, merge_path.local, revs, exc
        )
        if isinstance(exc, errors.VaultNotFound):
            message += (
                ". Probably you need to configure an access for the group you are using "
                "as task owner ('{}'). See "
                "https://wiki.yandex-team.ru/ReleaseMachine/create-new-release-machine/merges "
                "for details or report to SEARCH-5679 (with reopen) if it does not help. "
            ).format(task.owner)

        task.set_info(message)
        eh.log_exception(message, exc, task=task)

    if do_commit:
        post_commit_actions(
            task,
            merge_path,
            commit_info,
            commit_failed,
            revs,
            committed_revision,
            c_info,
        )
    # free space for next branch checkout (does nothing if path does not exist)
    paths.remove_path(merge_path.local)

    if commit_failed:
        return rm_const.CommitResult(rm_const.CommitStatus.failed, None)
    # If nothing was committed, than committed_revision=None
    return result


@decorators.retries(max_tries=2, delay=10, default_instead_of_raise=True)
def post_to_arcanum(task, review_ids, commit_result):
    if not review_ids:
        return
    logging.info("Posting to arcanum")
    arcanum_api = arcanum_helper.ArcanumApi(token=rm_sec.get_rm_token(task, rm_const.COMMON_TOKEN_NAME))
    task_link = lb.task_link(task.id, plain=True)
    for commit_status, commit_paths in commit_result.all().iteritems():
        if not commit_paths:
            continue  # do not write anything if no branches
        for review_id in review_ids:
            try:
                paths_info = ", ".join(p.short for p in commit_paths)
                if len(commit_paths) > 1:
                    paths_info = 'branches ' + paths_info
                comment = (
                    "{actions} {status} for {paths_info}\n"
                    "Action was done via {task_link} (see task logs for details)"
                ).format(
                    actions=list(set(p.action.capitalize() for p in commit_paths)),
                    status=commit_status,
                    paths_info=paths_info,
                    task_link=task_link,
                )
                arcanum_api.comment_review(review_id, comment=comment)
                task.set_info("Post notify in review {}".format(lb.review_link(review_id)), do_escape=False)
            except Exception as e:
                eh.log_exception("Unable to post commit results to arcanum", e)
                raise  # raise for fallback to RBApi
