import logging
import typing
import requests.exceptions

from sandbox import sdk2
from sandbox.common.rest import Client
from sandbox.projects.common import decorators
from sandbox.projects.release_machine import core
from sandbox.projects.release_machine.core import releasable_items as ri  # noqa: UnusedImport

LOGGER = logging.getLogger(__name__)


@decorators.decorate_all_public_methods(decorators.log_start_and_finish(LOGGER))
class BasicReleaser(object):
    """Base class for releases."""

    @decorators.memoized_property
    def _sb_rest_client(self):
        return Client()

    def before_release(self):
        """
        Do something before release.

        Here could be:
          - setting rm_event_data
          - notifications about release start
          - some other preparation work
        :return: None
        """
        pass

    def do_release(self):
        """
        Put artifacts to release queue.

        :rtype: List[ReleaseResult]
        """
        raise NotImplementedError

    def after_release(self, release_results):
        """
        Do something after release.

        Here could be:
          - updating rm_event_data
          - notifications about release results
        :return: None
        """
        pass

    def cleanup_after_all(self):
        """
        Any deploy-system-specific cleaning up logic.

        :return: None
        """
        pass

    def get_data_for_deploy_results(self):
        """
        Get data to use for waiting deploy results.
        This data should contain all information needed to get current deployment state along with state itself.

        Data will be used later in `get_deploy_result`.
        :return: List[DeployData.to_dict()]
        """
        return []

    def get_deploy_result(self, data):
        # type: (dict) -> typing.Optional[core.Result]
        """
        Get list of deploy results.

        :param data: dict which can be transformed into subclass of DeployData
        :return: Optional[Result]
        """
        return

    @decorators.retries(max_tries=5, delay=2, exceptions=(requests.exceptions.HTTPError,))
    def update_deploy_results(self, deploy_results):
        # type: (typing.List[dict]) -> typing.List[core.Result]
        """
        Update deploy_results, provided by task (or tasklet) in `wait for deploy` part.

        :param deploy_results: List of dicts, provided by `get_deploy_result` and `get_data_for_deploy_results` funcs.
        :return: List of new results only. If nothing changes, empty list will be returned.
        """
        new_deploy_results = []
        for deploy_result in deploy_results:
            updated_deploy_data = self.get_deploy_result(deploy_result)
            logging.debug("Updated deploy data: %s", updated_deploy_data)
            if updated_deploy_data:
                deploy_result.update(updated_deploy_data.result)
                logging.info("Updated deploy result: %s", deploy_result)
                new_deploy_results.append(updated_deploy_data)
        return new_deploy_results

    def after_deploy(self, deploy_results):
        # type: (typing.List[dict]) -> typing.Optional[AfterDeployResults]
        """
        Do something after deploy.

        Here could be:
          - updating rm_event_data
          - notifications about deploy results
        :return: Optional[AfterDeployResults]
        """
        return


class SimpleReleaserMixin(object):
    """Mixin for releases, not related to Release Machine."""

    def __init__(self, input_parameters, token):
        super(SimpleReleaserMixin, self).__init__()
        self._input = input_parameters
        self._token = token

    def get_released_items(self):
        """
        Get released items.

        Data will be used later to check if released resource is actually deployed
        :return: List[ReleasedResource]
        """
        return []


class RmReleaserMixin(object):
    """Mixin for releases via Release Machine."""

    def __init__(self, c_info):
        super(RmReleaserMixin, self).__init__()
        self._release_info = []
        self._c_info = c_info

    @property
    def release_info(self):
        """Release info getter for RM events."""
        return self._release_info


class SbReleaserMixin(RmReleaserMixin):
    """Mixin for releases via Release Machine using sandbox-tasks."""

    def __init__(self, task, c_info):
        super(SbReleaserMixin, self).__init__(c_info)
        self._task = task

    def _set_release_number_attributes(self, resources):
        for resource in resources:
            try:
                resource.major_release_num = self._task.Parameters.major_release_num
                resource.minor_release_num = self._task.Parameters.minor_release_num
                logging.info(
                    "Release number attributes set for resource #%s:\n"
                    "    - major_release_num: %s\n"
                    "    - minor_release_num: %s\n",
                    resource.id,
                    self._task.Parameters.major_release_num,
                    self._task.Parameters.minor_release_num,
                )
            except Exception as e:
                logging.warning("Failed to set release number attributes: %s", e, exc_info=True)


class BaseReleaseWatcher(object):
    """Base class to implement release and deploy monitoring."""

    def __init__(self, parameters, token):
        self._token = token
        self._parameters = parameters

    def last_release(self, release_stage=None):
        """
        Get last release data.

        :return: List[ReleasedResource]
        """
        raise NotImplementedError

    def last_deploy(self, release_stage=None):
        """
        Get last deploy data.

        :return: List[DeployedResource]
        """
        raise NotImplementedError


class BasicReleaseWatcher(BaseReleaseWatcher):
    """Base class to implement release and deploy monitoring for release machine."""

    def __init__(self, c_info, token=None):
        token = token or sdk2.Vault.data(
            c_info.releases_cfg__token_vault_owner,
            c_info.releases_cfg__token_vault_name,
        )
        super(BasicReleaseWatcher, self).__init__(None, token)

        LOGGER.info("Init %s", self.__class__.__name__)
        self._c_info = c_info
        self._sb_rest_client = Client()

    @decorators.memoize
    def _get_revision(self, build_url):
        arc_hash, svn_revision = "", ""
        if build_url:
            rev = sdk2.svn.Arcadia.get_revision(build_url)
            if rev.isdigit():
                svn_revision = rev
            else:
                arc_hash = rev
        LOGGER.info("Got arc_hash and revision: %s, %s", arc_hash, svn_revision)
        return arc_hash, svn_revision

    def last_deploy_proto(self, item_data, deploy_info):
        # type: (typing.Union[ri.SandboxResourceData, ri.DockerImageData], ri.DeployInfo) -> typing.List[typing.Any]
        """Get last deployed data in proto format for specified releasable item and its deploy info."""
        return []


class DeployData(object):
    def to_dict(self):
        raise NotImplementedError

    @classmethod
    def from_dict(cls, dictionary):
        return cls(**dictionary)


class AfterDeployResults(object):
    pass


class ReleasableItemWithResource(object):
    __slots__ = ("resource", "releasable_item")

    def __init__(self, resource, releasable_item):
        # type: (sdk2.Resource, ri.ReleasableItem) -> typing.NoReturn
        self.resource = resource
        self.releasable_item = releasable_item


def press_release_button_on_build_task(task_id, where_to_release, sb_rest_client, major_release_num=None):
    """
    Press release button on sandbox task.

    Used mostly for backward compatibility.
    """
    try:
        msg = "Release task {}".format(task_id)
        sb_rest_client.release({
            "to": [],
            "params": {},
            "task_id": task_id,
            "cc": [],
            "message": msg,
            "type": where_to_release,
            "subject": msg if not major_release_num else "{}: {}".format(msg, major_release_num),
        })
    except Exception as e:
        if e.message.endswith(" is already in progress"):  # RMINCIDENTS-162
            return core.Ok("Release button of '{}' was pressed by another task".format(task_id))
        else:
            return core.Error("Release process of '{}' failed:\n{}".format(task_id, e))
    return core.Ok("Release button of '{}' was pressed successfully".format(task_id))
