"""
This module provides releaser and release watcher classes compatible with the "kosher" release stack
(also known as the Kvasov-Korum scheme). These classes can be used by ReleaseRmComponent2 for the components
that are configured to use kosher releases

- https://st.yandex-team.ru/RMDEV-2380
- https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/kosher_release/
"""

import logging

import typing  # noqa: UnusedImport

from sandbox import sdk2
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine.helpers.deploy import sandbox_releaser
from sandbox.projects.common import kosher_release
from sandbox.projects.common import decorators
from sandbox.projects.common import time_utils
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine.core import releasable_items


LOGGER = logging.getLogger(__name__)


class KosherSandboxReleaserFailure(Exception):
    pass


class KosherSandboxReleaser(sandbox_releaser.SandboxReleaser):

    def _do_release_one_item(self, release_item):  # type: (typing.Any) -> typing.List[rm_core.Result]

        resource, _ = release_item

        kosher_release_result = self._do_release_kosher(resource.id)

        LOGGER.info("Release result (kosher): %s", kosher_release_result)

        if not kosher_release_result.ok:
            raise KosherSandboxReleaserFailure(kosher_release_result.result)

        old_release_results = super(KosherSandboxReleaser, self)._do_release_one_item(release_item)

        LOGGER.info("Sandbox native release result: %s", old_release_results)

        if not all(i.ok for i in old_release_results):
            LOGGER.warning(
                "Sandbox native release process failed:\n{}\n "
                "However the overall release process is not going to be considered a total failure. "
                "Only kosher-release results are taken into consideration by {}. We do execute Sandbox native "
                "release process but we do not rely on the results. If you find this frustrating please contact "
                "Release Machine support",
                [str(res) for res in old_release_results],
                self.__class__.__name__,
            )

        # We do not rely on the Sandbox native release results here. We only consider kosher-release results
        return [kosher_release_result]

    @decorators.retries(
        max_tries=5,
        default_instead_of_raise=True,
        default_value=rm_core.Error("Kosher release failed. Please check task logs"),
    )
    def _do_release_kosher(self, resource_id):   # type: (typing.Any) -> rm_core.Result
        kosher_release.release(
            resource_id=resource_id,
            stage=self._release_stage,
            sb_rest_client=self._task.server,
            skip_sandbox_native_release=True,  # the call of super-method will task care of it
        )

        return rm_core.Ok("Resource #{} successfully released (by means of Kosher-release scheme".format(resource_id))


class KosherSandboxReleaseWatcher(sandbox_releaser.SandboxReleaseWatcher):

    def _get_released_resources_data(self, release_stage=None):
        result = []

        for res_info, deploy_info in self._c_info.releases_cfg__iter_over_deploy_info(release_stage):

            attrs = {"released": deploy_info.level}
            attrs.update(res_info.attributes or {})

            last_released_resource_id = kosher_release.find_release(
                resource_type=res_info.resource_type,
                stage=deploy_info.level,
                sb_rest_client=self._sb_rest_client,
                custom_attrs=res_info.attributes or None,
            )

            last_released_resource = sdk2.Resource[last_released_resource_id]

            LOGGER.debug("Last released resource for '%s': %s", res_info, last_released_resource_id)

            released_item = rm_core.ReleasedItem(
                res_info.resource_name,
                last_released_resource,
                res_info.build_ctx_key,
            )

            major_num, minor_num = self._c_info.get_release_numbers(released_item)

            result.append({
                "id": last_released_resource_id,
                "build_task_id": last_released_resource.task_id,
                "major_release": major_num,
                "minor_release": minor_num,
                "timestamp": int(time_utils.datetime_to_timestamp(last_released_resource.updated)),
                "component": self._c_info.name,
                "status": deploy_info.level,
                "owner": last_released_resource.owner,
                "resource_name": res_info.resource_name,
            })

        return result

    def last_deploy_proto(self, item_data, deploy_info):
        # type: (releasable_items.SandboxResourceData, releasable_items.SandboxInfo) -> typing.List[typing.Any]

        from release_machine.common_proto import release_and_deploy_pb2

        last_released_resource_id = kosher_release.find_release(
            resource_type=item_data.resource_type,
            stage=deploy_info.stage,
            sb_rest_client=self._sb_rest_client,
            custom_attrs=item_data.attributes or None,
        )

        if not last_released_resource_id:
            return []

        last_released_resource = sdk2.Resource[last_released_resource_id]

        release_time = kosher_release.get_release_time(last_released_resource_id)
        major_num, minor_num = self._c_info.get_release_numbers_from_resource(last_released_resource, item_data)
        build_arc_url = rm_utils.get_input_or_ctx_field(last_released_resource.task_id, item_data.build_ctx_key)
        arc_hash, svn_revision = self._get_revision(build_arc_url)

        service_version = release_and_deploy_pb2.ServiceVersion(
            stage_label=deploy_info.stage,
            arc_hash=arc_hash,
            svn_revision=svn_revision,
            major_release_number=major_num,
            minor_release_number=minor_num,
            timestamp=int(time_utils.datetime_to_timestamp(release_time)),
            sandbox_resource=release_and_deploy_pb2.SbResourceData(
                resource_id=last_released_resource.id,
                resource_type=item_data.resource_type,
                build_task_id=last_released_resource.task_id,
            ),
        )
        logging.info("Got service_version:\n%s", service_version)
        return [service_version]
