from collections import defaultdict
from concurrent.futures import ThreadPoolExecutor
import difflib
import itertools as it
import logging
import json
import typing

from sandbox import sdk2
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers.deploy import basic_releaser
from sandbox.projects.release_machine.helpers.deploy import nanny_releaser
from sandbox.projects.release_machine import security as rm_sec


LOGGER = logging.getLogger(__name__)

COMPONENTS_IGNORING_SB_TASK_RELEASE_RESULTS = {"begemot"}  # SPI-25445


@decorators.decorate_all_public_methods(decorators.log_start_and_finish(LOGGER))
class NannyPushReleaser(basic_releaser.SbReleaserMixin, basic_releaser.BasicReleaser):
    """
    Implements push releases to nanny (by committing directly to services and avoiding ticket integration).

    Process consists of two stages:
      - Updating services directly (main part)
      - Pressing release button on sandbox tasks (for backward compatibility)
    """
    def __init__(self, task, c_info):
        super(NannyPushReleaser, self).__init__(task, c_info)
        self._nanny_client = NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=rm_sec.get_rm_token(task))

    def do_release(self):
        services_with_release_items = self.group_releasable_items_by_services()
        n_services = len(services_with_release_items)
        if n_services == 0:
            eh.check_failed("No services found for release!")
        elif n_services == 1:
            results = [self._push_service(*list(services_with_release_items.items())[0])]
        else:
            with ThreadPoolExecutor(max_workers=len(services_with_release_items)) as executor:
                results = list(executor.map(self._push_service, *zip(*services_with_release_items.items())))
        if all(i.ok for i in results):
            build_tasks_to_release_sb = {
                i.resource.task_id
                for i in self._releasable_items_with_resources
                if any(
                    isinstance(deploy_info, ri.SandboxInfo)
                    for deploy_info in i.releasable_item.iter_deploy_infos(self._task.Parameters.where_to_release)
                )
            }
            if build_tasks_to_release_sb:
                with ThreadPoolExecutor(max_workers=len(build_tasks_to_release_sb)) as executor:
                    button_press_results = executor.map(
                        basic_releaser.press_release_button_on_build_task,
                        build_tasks_to_release_sb,
                        it.repeat(self._task.Parameters.where_to_release),
                        it.repeat(self._sb_rest_client),
                        it.repeat(self._task.Parameters.major_release_num),
                    )
                if self._c_info.name in COMPONENTS_IGNORING_SB_TASK_RELEASE_RESULTS:  # SPI-25445
                    logging.warning(
                        "Component %s belongs to the set of components that ignore SB task release results. This means "
                        "that we tried to release each of the given releasable items however the release results are "
                        "not going to affect the overall task result (here they are btw: %s). "
                        "See SPI-25445 for details.",
                        self._c_info.name,
                        list(button_press_results),
                    )
                else:
                    results.extend(button_press_results)
            else:
                LOGGER.info("No build tasks to release")
        return results

    def group_releasable_items_by_services(self):
        # type: () -> typing.Dict[typing.AnyStr, typing.List[basic_releaser.ReleasableItemWithResource]]
        result = defaultdict(list)
        for i in self._releasable_items_with_resources:
            for deploy_info in i.releasable_item.iter_deploy_infos(self._task.Parameters.where_to_release):
                if not isinstance(deploy_info, ri.NannyDeployInfo):
                    continue
                for nanny_service in nanny_releaser.get_all_nanny_services(deploy_info, self._nanny_client):
                    result[nanny_service.name].append(i)
        return result

    @decorators.memoized_property
    def _releasable_items_with_resources(self):
        # type: () -> typing.List[basic_releaser.ReleasableItemWithResource]
        result = []
        for item in self._c_info.releases_cfg__releasable_items:
            LOGGER.debug("Considering: %s", item)
            stage_deploy_infos = list(item.iter_deploy_infos(self._task.Parameters.where_to_release))
            if not stage_deploy_infos:
                LOGGER.debug("Skip it, no deploy infos for stage '%s' found", self._task.Parameters.where_to_release)
                continue
            resource_id = self._task.Parameters.component_resources.get(item.name)
            if not resource_id:
                LOGGER.debug("Skip it, no resource id specified")
                continue
            resource = sdk2.Resource[resource_id]
            if isinstance(item.data, ri.SandboxResourceData) and resource.type != item.data.resource_type:
                eh.check_failed("Resource has type {} instead of {}!".format(resource.type, item.data.resource_type))
            if isinstance(item, ri.ReleasableItem):
                result.append(basic_releaser.ReleasableItemWithResource(resource, item))
            elif isinstance(item, ri.DynamicReleasableItem):
                result.extend(list(self._dynamically_construct_releasable_items(resource, item)))
            else:
                eh.check_failed("Unknown releasable item type: {}".format(item))
        return result

    def _dynamically_construct_releasable_items(self, resource, source_item):
        # type: (sdk2.Resource, ri.DynamicReleasableItem) -> typing.Generator[basic_releaser.ReleasableItemWithResource]
        for nanny_service, resources_data in fu.json_load(sdk2.ResourceData(resource).path).items():
            LOGGER.debug("Processing service %s", nanny_service)
            for resource_data in resources_data:
                disk_type = ri.DiskType.HDD
                deploy_infos = [
                    ri.single_nanny_service(nanny_service, stage=self._task.Parameters.where_to_release)
                ]
                if isinstance(resource_data, list):
                    dynamic_resource_id, local_path = resource_data
                else:
                    dynamic_resource_id, local_path = resource_data["resource_id"], resource_data["local_path"]
                    if resource_data.get("storage") == "/ssd":
                        disk_type = ri.DiskType.SSD
                    if resource_data.get("sandbox_release", True):
                        deploy_infos.append(ri.SandboxInfo(stage=self._task.Parameters.where_to_release))
                dynamic_resource = sdk2.Resource[dynamic_resource_id]
                dynamically_constructed_item = ri.ReleasableItem(
                    name="{}__{}".format(source_item.name, dynamic_resource_id),
                    data=ri.SandboxResourceData(
                        dynamic_resource.type,
                        build_ctx_key=source_item.data.build_ctx_key,
                        dst_path=local_path,
                        disk_type=disk_type,
                    ),
                    deploy_infos=deploy_infos,
                )
                yield basic_releaser.ReleasableItemWithResource(dynamic_resource, dynamically_constructed_item)

    def _push_service(self, service_name, releasable_items_with_resources):
        # type: (typing.AnyStr, typing.List[basic_releaser.ReleasableItemWithResource]) -> rm_core.Result
        self._task.set_info("Start releasing to service '{}'".format(service_name))
        service_resources = self._nanny_client.get_service_resources(service_name)
        sb_resources_old_version = self._group_by_local_path(service_resources["content"]["sandbox_files"])
        new_service_resources = self._construct_service_resources(releasable_items_with_resources)
        sb_resources_new_version = self._group_by_local_path(new_service_resources)
        updated_version = self._updated_resources(
            sb_resources_old_version, sb_resources_new_version,
            allow_to_add=self._c_info.releases_cfg__allow_to_add_resources,
            allow_to_remove=self._c_info.releases_cfg__allow_to_remove_resources,
        )
        service_resources["content"]["sandbox_files"] = updated_version
        try:
            release_label = self._c_info.get_release_subject(
                self._task.Parameters.where_to_release,
                self._task.Parameters.major_release_num,
                self._task.Parameters.minor_release_num,
                add_info=self._task.Parameters.additional_release_notes,
            )
            LOGGER.info("Going to push service update to nanny")
            update_response = self._nanny_client.update_service_resources(service_name, {
                "content": service_resources["content"],
                "comment": "[push] {}\n(committed by RM directly using task {})".format(release_label, self._task.id),
                "snapshot_id": service_resources["snapshot_id"]
            })
            LOGGER.info("Nanny service '%s' update response:\n%s", service_name, pretty_json_dump(update_response))
        except Exception as e:
            self._task.set_info("Failed releasing to service '{}'".format(service_name))
            return rm_core.Error("Failed to push new configuration to service {}:\n{}".format(service_name, e))

        from release_machine.common_proto import events_pb2 as rm_proto_events

        for i in releasable_items_with_resources:
            self._release_info.append(rm_proto_events.ReleasedResourceInfo(
                release_label=release_label,
                resource_id=str(i.resource.id),
                resource_type=str(i.resource.type),
                build_task_id=str(i.resource.task_id),
            ))

        self._task.set_info("Finish pushing to service '{}'".format(service_name))
        return rm_core.Ok("Release process to service '{}' was started successfully".format(service_name))

    @staticmethod
    def _group_by_local_path(sb_files_list):
        return {
            i["local_path"]: i for i in sb_files_list
        }

    def _construct_service_resources(self, releasable_items_with_resources):
        # type: (typing.List[basic_releaser.ReleasableItemWithResource]) -> typing.Generator[typing.Dict[str, str]]
        for i in releasable_items_with_resources:
            build_task_info = self._sb_rest_client.task[i.resource.task_id].read()
            local_path = i.releasable_item.data.dst_path
            if local_path is None:
                eh.check_failed("{} should have dst_path specified!".format(i.releasable_item.data))
            if "/" in local_path:  # nanny is unable to work with nested folders
                local_path = "+rm+{}".format(local_path.replace('/', '+'))
            resource_description = {
                "task_type": str(build_task_info["type"]),
                "task_id": str(i.resource.task_id),
                "resource_id": str(i.resource.id),
                "resource_type": str(i.resource.type),
                "local_path": local_path,
            }
            yield resource_description

    @staticmethod
    def _updated_resources(old, new, allow_to_add=True, allow_to_remove=False):
        result = []
        old_keys = set(old.keys())
        new_keys = set(new.keys())
        new_only_keys = new_keys - old_keys
        if new_only_keys:
            if allow_to_add:
                LOGGER.info("Added:\n%s", "\n".join(pretty_json_dump(new[i]) for i in new_only_keys))
                result.extend(new[i] for i in new_only_keys)
        LOGGER.info("Changed:")
        changed = []
        for key in old_keys & new_keys:
            elem = {}
            elem.update(old[key])
            elem.update(new[key])
            changed.append("{}:\n{}".format(key, "\n".join(list(difflib.unified_diff(
                pretty_json_dump(old[key]).split("\n"),
                pretty_json_dump(elem).split("\n"),
                n=0,
            ))[3:])))
            result.append(elem)
        LOGGER.info("\n".join(changed))
        old_only_keys = old_keys - new_keys
        if old_only_keys:
            if allow_to_remove:
                LOGGER.info("Removed:\n%s", "\n".join(pretty_json_dump(old[i]) for i in old_only_keys))
            else:
                result.extend(old[i] for i in old_only_keys)

        return result


def pretty_json_dump(data):
    return json.dumps(data, sort_keys=True, indent=2, separators=('', ': '))
