import datetime as dt
import logging
import itertools
import time
from dateutil.tz import gettz
from typing import (  # noqa: UnusedImport
    Any,
    Dict,
    List,
    Tuple,
    Optional,
    Union,
)

from sandbox import sdk2
from sandbox.common.rest import Client
from sandbox.projects.common import decorators
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.release_machine.core import (
    DeployedResource,
    ReleasedItem,
    Ok,
    Error,
    const as rm_const,
    releasable_items as ri
)
from sandbox.projects.release_machine.helpers.deploy import basic_releaser
from sandbox.projects.release_machine.helpers.deploy import sandbox_releaser
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine import security as rm_sec

LOGGER = logging.getLogger(__name__)
END_STATUSES = frozenset(["DEPLOY_SUCCESS", "CLOSED", "CANCELLED", "DEPLOY_FAILED"])


class NannySimpleReleaser(sandbox_releaser.SandboxSimpleReleaser):

    def __init__(self, input_parameters, token):
        super(NannySimpleReleaser, self).__init__(input_parameters, token)
        self._nanny_client = NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=token)
        self._nanny_release_reqs_prev = {}
        self._nanny_deploy_results = {}

    @decorators.memoized_property
    def _sb_rest_client(self):
        return Client(auth=self._token)

    @property
    def common_release_data(self):
        return self._input.config.common_release_data.nanny_common_release_data

    def before_release(self):
        super(NannySimpleReleaser, self).before_release()
        self._nanny_release_reqs_prev = get_nanny_release_reqs(self._release_items, self._sb_rest_client)

    def get_data_for_deploy_results(self):
        return [v.to_dict() for v in self._nanny_deploy_results.values()]

    def get_deploy_result(self, data):
        return get_nanny_deploy_results(data, self._nanny_client)

    def _wait_release_results(self):
        wait_results = super(NannySimpleReleaser, self)._wait_release_results()
        nanny_release_results = wait_new_nanny_release_reqs(
            self._nanny_release_reqs_prev,
            self._release_items,
            self._sb_rest_client,
        )
        if nanny_release_results.ok:
            self._nanny_deploy_results = {k: NannyDeployData(v[-1], k) for k, v in nanny_release_results.result.items()}
            return wait_results + [Ok("Got all nanny release requests")]
        return wait_results + [nanny_release_results]


class NannyReleaser(sandbox_releaser.SandboxReleaser):

    def _get_release_params(self, release_item):
        params = super(NannyReleaser, self)._get_release_params(release_item)
        if self._c_info.notify_cfg__use_startrek:
            self._c_info.notify_cfg__st__add_nanny_reports(
                self._task.Parameters.where_to_release, params["params"], self._task.get_st_issue_key()
            )
        return params

    def __init__(self, task, c_info):
        super(NannyReleaser, self).__init__(task, c_info)
        self._nanny_client = NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=rm_sec.get_rm_token(task))
        self._nanny_release_reqs_prev = {}
        self._nanny_deploy_results = {}

    def before_release(self):
        super(NannyReleaser, self).before_release()
        self._nanny_release_reqs_prev = get_nanny_release_reqs(self._release_items, self._sb_rest_client)

    def after_release(self, release_results):
        for i in self._release_info:
            i.nanny_ticket_key = self._nanny_deploy_results[int(i.build_task_id)].request_id
        super(NannyReleaser, self).after_release(release_results)

    def get_data_for_deploy_results(self):
        return [v.to_dict() for v in self._nanny_deploy_results.values()]

    def get_deploy_result(self, data):
        return get_nanny_deploy_results(data, self._nanny_client)

    def _wait_release_results(self):
        wait_results = super(NannyReleaser, self)._wait_release_results()
        nanny_release_results = wait_new_nanny_release_reqs(
            self._nanny_release_reqs_prev,
            self._release_items,
            self._sb_rest_client,
        )
        if nanny_release_results.ok:
            self._nanny_deploy_results = {k: NannyDeployData(v[-1], k) for k, v in nanny_release_results.result.items()}
            return wait_results + [Ok("Got all nanny release requests")]
        return wait_results + [nanny_release_results]


def wait_new_nanny_release_reqs(nanny_release_reqs_prev, release_items, sb_rest_client):
    wait_time = 30
    wait_attempts = 5
    for current_attempt in itertools.count():
        LOGGER.debug("Waiting for nanny release requests. Attempt %s", current_attempt)
        nanny_release_reqs = get_nanny_release_reqs(release_items, sb_rest_client)
        wait_for = [k for k, v in nanny_release_reqs.items() if len(v) <= len(nanny_release_reqs_prev.get(k, []))]
        if not wait_for:
            return Ok(nanny_release_reqs)
        if current_attempt <= wait_attempts:
            LOGGER.info("Wait %s sec for nanny release request appearing in build tasks: %s", wait_time, wait_for)
            time.sleep(wait_time)
        else:
            return Error("Timed out waiting for nanny release requests in build tasks: {}".format(wait_for))


def get_nanny_release_reqs(release_items, sb_rest_client):
    # type: (List[ReleasedItem], Any) -> Dict[int, List[str]]
    """
    Get all nanny release requests for each build task id from task context.

    Build task could have several release requests because it could be released several times.
    """
    build_task_ids = set(resource.task_id for resource, _ in release_items)
    ctx_key = "context.nanny_release_requests"
    reqs = rm_utils.get_tasks_fields(
        ctx_key,
        {"limit": len(build_task_ids), "id": list(build_task_ids)},
        sb_rest_client
    )
    return {int(i["id"]): i.get(ctx_key) or [] for i in reqs}


def get_nanny_deploy_results(data, nanny_client):
    release_request = NannyDeployData.from_dict(data)
    if release_request.state in END_STATUSES:
        LOGGER.debug("Nothing to do: %s", release_request)
        return
    info = nanny_client.get_release_request_info(release_request.request_id)
    LOGGER.debug("Got nanny release request info: %s", info)
    state = info.get("status")
    release_request.state = state
    if state in END_STATUSES:
        end_time = info.get("end_time", {}).get("$date")
        if end_time:
            end_time_iso = dt.datetime.utcfromtimestamp(end_time / 1000).replace(
                tzinfo=gettz('Europe/Moscow')
            ).isoformat()
            release_request.end_time = end_time_iso
        return Ok(release_request.to_dict())
    else:
        return Error(release_request.to_dict())


class NannyDeployData(basic_releaser.DeployData):
    __slots__ = ("request_id", "build_task_id", "state", "end_time")

    def __init__(self, request_id, build_task_id, state="OPEN", end_time=None):
        self.request_id = request_id
        self.build_task_id = build_task_id
        self.state = state
        self.end_time = end_time

    def __str__(self):
        result = "NannyReleaseRequest<{}>[{}]".format(self.request_id, self.state)
        if self.end_time:
            result += ". End time = {} MSK".format(self.end_time)
        return result

    def to_dict(self):
        return {
            "request_id": self.request_id,
            "build_task_id": self.build_task_id,
            "state": self.state,
            "end_time": self.end_time,
        }

    __repr__ = __str__


class NannyReleaseWatcher(sandbox_releaser.SandboxReleaseWatcher):
    def __init__(self, c_info, token):
        super(NannyReleaseWatcher, self).__init__(c_info, token)
        self.nc = NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=self._token)

    @decorators.memoize
    def _get_service_data(self, service):
        LOGGER.info("Getting data from nanny service '%s'", service)
        current_state = self.nc.get_service_current_state(service)
        active_runtime_data = self.nc.get_service_active_runtime_attrs(service)
        return current_state, active_runtime_data

    def last_deploy(self, release_stage=None):
        deploy_data = []
        for res_info, deploy_info in self._c_info.releases_cfg__iter_over_deploy_info(release_stage):
            services = self._c_info.get_deploy_services(deploy_info, self.nc)
            service_versions = []
            for service in services:
                try:
                    service_version = self._get_one_service_version(service, res_info, deploy_info.level)
                    if service_version:
                        service_versions.append(service_version)
                except Exception as ex:
                    LOGGER.exception("Error occurred with _get_one_service_version in '%s' service:\n%s", service, ex)
            if service_versions:
                deploy_data.append(deploy_info.chooser(service_versions))
        return deploy_data

    @staticmethod
    def _get_deploy_time(current_state, active_runtime_data):
        for snapshot in current_state['content'].get('active_snapshots', []):
            if snapshot['snapshot_id'] == active_runtime_data['_id']:
                # Check if snapshot has actual ACTIVE state RMDEV-701, or DEACTIVATE_PENDING state RMDEV-2319
                if snapshot['state'] in ['ACTIVE', "DEACTIVATE_PENDING"]:
                    return int(snapshot['entered'] / 1000)
                else:
                    LOGGER.debug(
                        "Found snapshot is not in ACTIVE or DEACTIVATE_PENDING state. Skip it. Snapshot content: %s",
                        snapshot,
                    )
                    continue

    @staticmethod
    def _get_resource_and_build_task(resource_type, active_runtime_data):
        for res in active_runtime_data["content"]["resources"]["sandbox_files"]:
            LOGGER.debug("Current resource type: %s", res["resource_type"])
            if res["resource_type"] == resource_type:
                LOGGER.info("Got resource data: %s", res)
                return res["task_id"], res["resource_id"]
        return None, None

    def _get_one_service_version(self, service, res_info, deploy_level):
        res_type = res_info.resource_type
        LOGGER.info("Getting version of resource type %s in service: %s", res_type, service)
        current_state, active_runtime_data = self._get_service_data(service)
        deploy_time = self._get_deploy_time(current_state, active_runtime_data)
        if deploy_time is None:
            LOGGER.warning("Unable to deploy_time for active snapshot, skip this service: %s", service)
            return
        build_task_id, resource_id = self._get_resource_and_build_task(res_type, active_runtime_data)
        if build_task_id is None:
            LOGGER.warning("Unable to find build task id, skip this service: %s", service)
            return
        major_num, minor_num = self._c_info.get_release_numbers_from_resource(sdk2.Resource[resource_id], res_info)

        return DeployedResource(
            id=resource_id,
            build_task_id=build_task_id,
            timestamp=deploy_time,
            major_release=major_num,
            minor_release=minor_num,
            component=self._c_info.name,
            status=deploy_level,
            info=active_runtime_data["meta_info"].get("ticket_info", {}).get("release_id", ""),
            resource_name=res_info.resource_name,
        )

    def last_deploy_proto(self, item_data, deploy_info):
        # type: (ri.SandboxResourceData, ri.NannyDeployInfo) -> List[Any]
        LOGGER.info("Getting versions of %s", item_data)
        from release_machine.common_proto import release_and_deploy_pb2

        versions = []
        for service in get_all_nanny_services(deploy_info, self.nc):
            try:
                current_state, active_runtime_data = self._get_service_data(service.name)
            except Exception:
                LOGGER.error("Unable to get version of service %s. Skip it", service.name)
                continue
            deploy_time = self._get_deploy_time(current_state, active_runtime_data)
            build_task_id, resource_id = self._get_resource_and_build_task(item_data.resource_type, active_runtime_data)
            if deploy_time is None or build_task_id is None:
                LOGGER.warning(
                    "Unable to get deploy_time or build task id for active snapshot. "
                    "Deploy time: %s, Build task id: %s. "
                    "Save empty service version for this service: %s",
                    deploy_time, build_task_id, service
                )
                service_version = release_and_deploy_pb2.ServiceVersion(
                    stage_label=deploy_info.stage,
                    tags=service.tags if service.tags else None,
                    nanny=release_and_deploy_pb2.NannyData(service=service.name)
                )
            else:
                last_released_resource = sdk2.Resource[resource_id]
                major_num, minor_num = self._c_info.get_release_numbers_from_resource(last_released_resource, item_data)
                build_arc_url = rm_utils.get_input_or_ctx_field(last_released_resource.task_id, item_data.build_ctx_key)
                arc_hash, svn_revision = self._get_revision(build_arc_url)
                service_version = release_and_deploy_pb2.ServiceVersion(
                    stage_label=deploy_info.stage,
                    arc_hash=arc_hash,
                    svn_revision=svn_revision,
                    major_release_number=major_num,
                    minor_release_number=minor_num,
                    timestamp=deploy_time,
                    tags=service.tags if service.tags else None,
                    sandbox_resource=release_and_deploy_pb2.SbResourceData(
                        resource_id=int(last_released_resource.id),
                        resource_type=item_data.resource_type,
                        build_task_id=int(build_task_id),
                    ),
                    nanny=release_and_deploy_pb2.NannyData(
                        service=service.name,
                        release_ticket=active_runtime_data["meta_info"].get("ticket_info", {}).get("release_id", ""),
                    )
                )
            LOGGER.info("Got service_version:\n%s", service_version)
            versions.append(service_version)
        return versions


def get_all_nanny_services(deploy_info, nanny_client):
    # type: (ri.NannyDeployInfo, Any) -> Tuple[ri.DeployService]
    if not deploy_info.dashboards:
        return deploy_info.services
    service_names = {i.name for i in deploy_info.services}
    services_from_dashboards = []
    for dashboard in deploy_info.dashboards:
        for i in nanny_client.get_dashboard_services(dashboard):
            if i not in service_names:
                services_from_dashboards.append(ri.DeployService(i))
                service_names.add(i)
    return deploy_info.services + tuple(services_from_dashboards)
