import json
import logging
import re
import tempfile
import time
import datetime as dt

from dateutil.tz import gettz

from sandbox.projects.rasp.qloud import api as rasp_qloud_api
from sandbox.projects.release_machine.core import (
    Ok,
    Error,
    DeployedResource,
    releasable_items as ri
)

from sandbox import sdk2
from sandbox.projects.release_machine.helpers.deploy import basic_releaser
from sandbox.projects.release_machine.helpers.vcs_indexer_client import VcsIndexerClient

LOGGER = logging.getLogger(__name__)
REGISTRY_RE = re.compile(
    r'registry\.yandex\.net/(?P<proj>.*)/(?P<app>.*):.*'
    r'stable-(?P<major>\d+)-(?P<minor>\d+)\.(?P<revision>\d+)\.(?P<task_id>\d+)'
)
REVISION_REGISTRY_RE = re.compile(r'registry\.yandex\.net/(?P<proj>.*)/(?P<app>.*):(?P<revision>\d+)')


class QloudReleaser(basic_releaser.SbReleaserMixin, basic_releaser.BasicReleaser):
    END_STATUSES = frozenset(["DEPLOYED"])

    def __init__(self, task, c_info):
        super(QloudReleaser, self).__init__(task, c_info)
        self._qloud_env = task.Parameters.qloud_environment
        self._registry_name = task.Parameters.registry_name
        self._qloud_environment_dump_arcadia_path = task.Parameters.qloud_environment_dump_arcadia_path
        self._qloud_environment_dump_svn_revision = task.Parameters.qloud_environment_dump_svn_revision
        self._qloud_custom_environment_name = task.Parameters.qloud_custom_environment_name
        token = sdk2.Vault.data(task.Parameters.qloud_vault_owner, task.Parameters.qloud_vault_name)
        self._qloud_pub_api = rasp_qloud_api.QloudPublicApi(token)
        self._qloud_priv_api = rasp_qloud_api.QloudPrivateApi(token)
        self._release_items = list(self._get_release_items())  # type: List[Tuple[sdk2.Resource, Any]]
        self._released_qloud_environment_version = None

    def _get_release_items(self):
        for res_info in self._c_info.releases_cfg__resources_info:
            LOGGER.info("%s. Resource name: %s", res_info, res_info.resource_name)
            resource_id = self._task.Parameters.component_resources.get(res_info.resource_name, None)
            if not resource_id:
                LOGGER.info("No id specified for this resource")
                continue
            resource = sdk2.Resource[resource_id]
            if resource.type != res_info.resource_type:
                LOGGER.error("Resource has type %s instead of %s", resource.type, res_info.resource_type)
            yield resource, res_info

    def get_data_for_deploy_results(self):
        component_spec = rasp_qloud_api.ComponentSpecification.parse_from_path(self._qloud_env)
        environment_info = self._qloud_pub_api.get_environment_version_info(
            component_spec.environment_id,
            self._released_qloud_environment_version,
        )
        deploy_data = QloudDeployData(
            component_spec.environment_id,
            environment_info["version"],
            environment_info["status"],
        )
        return [deploy_data.to_dict()]

    def get_deploy_result(self, data):
        deploy_data = QloudDeployData.from_dict(data)
        if deploy_data.status in self.END_STATUSES:
            LOGGER.debug("Nothing to do: %s", deploy_data)
            return
        environment_version_info = self._qloud_pub_api.get_environment_version_info(
            deploy_data.environment_id,
            deploy_data.version,
        )
        LOGGER.debug("Got Qloud environment version info: %s", environment_version_info)
        status = environment_version_info.get("status")
        deploy_data.status = status
        if status in self.END_STATUSES:
            end_timestamp = environment_version_info.get("statusHistory", {}).get("messages", [])[-1]['time']
            end_time_iso = dt.datetime.utcfromtimestamp(end_timestamp / 1000).replace(
                tzinfo=gettz('Europe/Moscow')
            ).isoformat()
            deploy_data.end_time = end_time_iso
            return Ok(deploy_data.to_dict())
        else:
            return Error(deploy_data.to_dict())

    def do_release(self):
        component_spec = rasp_qloud_api.ComponentSpecification.parse_from_path(self._qloud_env)

        if self._qloud_environment_dump_arcadia_path:
            LOGGER.info("Using dump from arcadia path %s", self._qloud_environment_dump_arcadia_path)
            env_dump = self._load_dump_from_arcadia_path()
        else:
            LOGGER.info("Using dump from qloud environment %s", component_spec.environment_id)
            try:
                env_dump = self._qloud_pub_api.dump_environment(component_spec.environment_id)
            except Exception as e:
                msg = "Failed to dump environment\n{}".format(e)
                LOGGER.exception(msg)
                return [Error(msg)]

        if self._qloud_custom_environment_name:
            LOGGER.info("Setting custom environment name to env id: %s",
                        self._qloud_custom_environment_name)
            component_spec = component_spec.with_environment(self._qloud_custom_environment_name)

        if env_dump["objectId"] != component_spec.environment_id:
            LOGGER.warning("Overriding env id in dump: %s -> %s",
                           env_dump["objectId"], component_spec.environment_id)
            env_dump["objectId"] = component_spec.environment_id

        registry_url, registry_tag = self._split_registry_tag()
        component_descr = "Component(path={}, registry={}, tag={})".format(
            self._qloud_env, registry_url, registry_tag
        )

        LOGGER.info("Updating %s", component_descr)

        components_to_update = list(self._components_to_update(env_dump, component_spec.name))
        if not components_to_update:
            return [Error("No component for path {} were found".format(self._qloud_env))]

        for component in components_to_update:
            component["properties"]["hash"] = self._qloud_priv_api.docker_hash(registry_url, registry_tag)
            component["properties"]["repository"] = "registry.yandex.net/{}:{}".format(registry_url, registry_tag)

        env_dump["comment"] = self._build_release_comment()

        try:
            upload_environment_result = self._qloud_pub_api.upload_environment(env_dump)
            self._released_qloud_environment_version = upload_environment_result["version"]
        except Exception as e:
            msg = "Failed to update {}\n{}".format(component_descr, e)
            LOGGER.exception(msg)
            return [Error(msg)]
        return [Ok("{} updated successfully".format(component_descr))]

    def _build_release_comment(self):
        if not self._release_items or len(self._release_items) != 1:
            return "Deploy to qloud from sandbox RM"

        return self._c_info.get_release_notes(
            self._task.Parameters.where_to_release,
            [res_info.name for _, res_info in self._release_items],
            self._task.author,
            self._task.Parameters.major_release_num,
            self._task.Parameters.minor_release_num,
        )

    def _split_registry_tag(self):
        if ":" in self._registry_name:
            registry_url, registry_tag = self._registry_name.split(":")
        else:
            registry_url, registry_tag = self._registry_name, "latest"
        if registry_url.startswith("registry.yandex.net/"):
            registry_url = registry_url[len("registry.yandex.net/"):]

        return registry_url, registry_tag

    def _load_dump_from_arcadia_path(self):
        with tempfile.NamedTemporaryFile(suffix='.json') as local_file:
            sdk2.svn.Arcadia.export(
                sdk2.svn.Arcadia.trunk_url(self._qloud_environment_dump_arcadia_path),
                local_file.name,
                revision=self._qloud_environment_dump_svn_revision
            )

            with open(local_file.name) as fp:
                return json.load(fp)

    @staticmethod
    def _components_to_update(env, components_to_update_str):
        component_names = {name for name in components_to_update_str.split(",") if name}

        for component in env.get("components", []):
            if (
                component["componentType"] == "standard" and
                ((component["componentName"] in component_names) or ("*" in component_names))
            ):
                yield component


class QloudDeployData(basic_releaser.DeployData):
    __slots__ = ("environment_id", "version", "status", "end_time")

    def __init__(self, environment_id, version, status="NEW", end_time=None):
        self.environment_id = environment_id
        self.version = version
        self.status = status
        self.end_time = end_time

    def __str__(self):
        result = "QloudReleaseRequest<{}/{}>[{}]".format(self.environment_id, self.version, self.status)
        if self.end_time:
            result += ". End time = {} MSK".format(self.end_time)
        return result

    def to_dict(self):
        return {
            "environment_id": self.environment_id,
            "version": self.version,
            "status": self.status,
            "end_time": self.end_time,
        }

    __repr__ = __str__


class QloudReleaseWatcher(basic_releaser.BasicReleaseWatcher):
    def last_deploy_proto(self, item_data, deploy_info):
        # type: (Union[ri.SandboxResourceData, ri.DockerImageData], ri.QloudDeployInfo) -> List[Any]
        """The code here is not clean, but qloud is deprecated, so it will be deleted soon."""
        LOGGER.info("Getting versions of %s", item_data)
        from release_machine.common_proto import release_and_deploy_pb2
        qloud_pub_api = rasp_qloud_api.QloudPublicApi(self._token)
        versions = []
        for service in deploy_info.services:
            environment_id, _ = service.name.rsplit(".", 1)
            dump = qloud_pub_api.get_component_info(service.name)
            LOGGER.debug("Qloud component info dump: %s", dump)
            if dump is None:
                LOGGER.warning("Service {} not found".format(service.name))
                continue
            actual_states = [i.get("actualState") for i in dump["runningInstances"]]
            if not all(actual_state == "ACTIVE" for actual_state in actual_states):
                LOGGER.warning("Not all actual states are active: {}".format(actual_states))
                continue
            else:
                LOGGER.info("All %s instances are in active state", len(actual_states))
            repository = dump["repository"]
            LOGGER.info("Parsing repository %s", repository)

            match = deploy_info.version_getter.match(repository)
            if match is not None:
                versions.append(release_and_deploy_pb2.ServiceVersion(
                    stage_label=deploy_info.stage,
                    arc_hash="",
                    svn_revision="",  # todo: get revision from build task input
                    major_release_number=match.group("major"),
                    minor_release_number=match.group("minor"),
                    timestamp=_get_deploy_time(qloud_pub_api, environment_id),
                    tags=service.tags if service.tags else None,
                    docker_image=release_and_deploy_pb2.DockerImageData(),
                    qloud=release_and_deploy_pb2.QloudData(environment=service.name)
                ))
                continue
            match = REGISTRY_RE.match(repository)
            if match is not None:  # that means, we have SandboxResourceData
                build_task_id = int(match.group("task_id"))
                res_type = str(item_data.resource_type)
                resources_found = self._sb_rest_client.resource.read(
                    type=res_type,
                    task_id=build_task_id,
                    limit=1,
                )["items"]
                if not resources_found:
                    LOGGER.warning("Unable to find resource with task_id {task_id} and type {res_type}".format(
                        task_id=build_task_id,
                        res_type=res_type,
                    ))
                    continue
                resource_id = resources_found[0]["id"]
                versions.append(release_and_deploy_pb2.ServiceVersion(
                    stage_label=deploy_info.stage,
                    arc_hash="",
                    svn_revision=match.group("revision"),
                    major_release_number=match.group("major"),
                    minor_release_number=match.group("minor"),
                    timestamp=_get_deploy_time(qloud_pub_api, environment_id),
                    tags=service.tags if service.tags else None,
                    sandbox_resource=release_and_deploy_pb2.SbResourceData(
                        resource_id=resource_id,
                        resource_type=res_type,
                        build_task_id=build_task_id,
                    ),
                    qloud=release_and_deploy_pb2.QloudData(environment=service.name)
                ))
                continue
            match = REVISION_REGISTRY_RE.match(repository)
            if match is not None:
                svn_revision = match.group("revision")
                vcs_indexer = VcsIndexerClient()
                rev_info = vcs_indexer.batch_info([svn_revision], sandbox_mode=False)[0]
                for path in rev_info["paths"]:
                    release_numbers = self._c_info.svn_cfg__get_release_numbers(path["path"])
                    if release_numbers:
                        major_release_number, minor_release_number = release_numbers
                        versions.append(release_and_deploy_pb2.ServiceVersion(
                            stage_label=deploy_info.stage,
                            arc_hash="",
                            svn_revision=svn_revision,
                            major_release_number=major_release_number,
                            minor_release_number=minor_release_number,
                            timestamp=_get_deploy_time(qloud_pub_api, environment_id),
                            tags=service.tags if service.tags else None,
                            docker_image=release_and_deploy_pb2.DockerImageData(),
                            qloud=release_and_deploy_pb2.QloudData(environment=service.name)
                        ))
                        break
                continue
            logging.warning("No match for repository: %s", repository)
        return versions

    def last_release(self, release_stage=None):
        return []  # todo

    def last_deploy(self, release_stage=None):
        qloud_pub_api = rasp_qloud_api.QloudPublicApi(self._token)
        result = []
        for res_info, deploy_info in self._c_info.releases_cfg__iter_over_deploy_info(release_stage):
            for service in deploy_info.services:
                LOGGER.debug("Handling %s", service)
                dump = qloud_pub_api.get_component_info(service)
                if dump is None:
                    LOGGER.warning("Service {} not found".format(service))
                    continue
                actual_states = [i.get("actualState") for i in dump["runningInstances"]]
                if not all(actual_state == "ACTIVE" for actual_state in actual_states):
                    LOGGER.warning("Not all actual states are active: {}".format(actual_states))
                    continue
                repository = dump["repository"]
                LOGGER.info("Parsing repository %s", repository)
                match = REGISTRY_RE.match(repository)
                if match is not None:
                    data = self._get_data_from_extended_match(res_info, deploy_info, match)
                    if data:
                        environment_id, _ = service.rsplit(".", 1)
                        data.timestamp = _get_deploy_time(qloud_pub_api, environment_id) or int(time.time())
                        result.append(data)
                    continue
                match = REVISION_REGISTRY_RE.match(repository)
                if match is not None:
                    data = self._get_data_from_revision_match(res_info, deploy_info, match)
                    if data:
                        environment_id, _ = service.rsplit(".", 1)
                        data.timestamp = _get_deploy_time(qloud_pub_api, environment_id) or int(time.time())
                        result.append(data)
                    continue
                logging.warning("No match for repository: %s", repository)
        return result

    def _get_data_from_revision_match(self, res_info, deploy_info, match):
        LOGGER.info("Getting data from simple match: %s", match)
        rev = match.group("revision")
        vcs_indexer = VcsIndexerClient()
        rev_info = vcs_indexer.batch_info([rev], sandbox_mode=False)[0]
        for path in rev_info["paths"]:
            release_numbers = self._c_info.svn_cfg__get_release_numbers(path["path"])
            if release_numbers:
                major_release, minor_release = release_numbers
                return DeployedResource(
                    id=0,
                    build_task_id=0,
                    major_release=major_release,
                    minor_release=minor_release,
                    component=self._c_info.name,
                    status=deploy_info.level,
                )
        LOGGER.warning("Unable to find deploy info from simple match")

    def _get_data_from_extended_match(self, res_info, deploy_info, match):
        LOGGER.info("Getting data from extended match: %s", match)
        task_id = int(match.group("task_id"))
        res_type = str(res_info.resource_type)
        resources_found = self._sb_rest_client.resource.read(
            type=res_type,
            task_id=task_id,
            limit=1,
        )["items"]
        if resources_found:
            resource_id = resources_found[0]["id"]
            LOGGER.debug("Resource of type %s found: %s", res_info.resource_type, resource_id)
            return DeployedResource(
                id=resource_id,
                build_task_id=task_id,
                major_release=match.group("major"),
                minor_release=match.group("minor"),
                component=self._c_info.name,
                status=deploy_info.level,
            )
        LOGGER.warning("Unable to find resource with task_id {task_id} and type {res_type}".format(
            task_id=task_id,
            res_type=res_type,
        ))


def _get_deploy_time(qloud_pub_api, environment_id):
    environment_version_info = qloud_pub_api.get_environment_info(environment_id)
    LOGGER.debug("Got Qloud environment info: %s", environment_version_info)
    for status_history_item in reversed(environment_version_info.get("statusHistory", {}).get("messages", [])):
        if status_history_item.get("status") == "DEPLOYED":
            return int(status_history_item.get("time") / 1000)
    return 0
