import logging

from sandbox import sdk2
from sandbox.projects.common import decorators
from sandbox.projects.release_machine.core import (
    Ok,
    Error,
    DeployedResource,
)
from sandbox.projects.release_machine.helpers.deploy import basic_releaser
from sandbox.projects.samogon import api as samogon_api

LOGGER = logging.getLogger(__name__)


class SamogonDeployData(basic_releaser.DeployData):
    __slots__ = ("project", "url", "resource_id")

    def __init__(self, project, url, resource_id):
        self.resource_id = resource_id
        self.project = project
        self.url = url

    def __str__(self):
        result = "SamogonDeployData[{}, {}, {}]".format(
            self.project,
            self.url,
            self.resource_id,
        )
        return result

    def to_dict(self):
        return {
            "project": self.project,
            "url": self.url,
            "resource_id": self.resource_id,
        }

    __repr__ = __str__


@decorators.decorate_all_public_methods(decorators.log_start_and_finish(LOGGER))
class SamogonBasicReleaser(basic_releaser.BasicReleaser):
    def _do_release_one_item(self, pkg, project, namespace):
        try:
            self._samogon_deploy_api.deploy(pkg, project, namespace)
        except Exception as e:
            return Error("Failed to release: {}".format(e))
        return Ok("Release process of package '{}' to {}:{} was started successfully".format(
            pkg, project, namespace
        ))

    def get_deploy_result(self, data):
        deploy_data = SamogonDeployData.from_dict(data)
        cluster_url = "https://clusterapi-{}".format(deploy_data.url)
        samogon_cluster_api = samogon_api.SamogonClusterAPI(cluster_url, self._token)
        deployed_resource_ids = samogon_cluster_api.get_deployed_resource_ids()
        result = Ok if all(int(res_id) == int(deploy_data.resource_id) for res_id in deployed_resource_ids if
                           res_id is not None) else Error
        return result(deploy_data.to_dict())


@decorators.decorate_all_public_methods(decorators.log_start_and_finish(LOGGER))
class SamogonSimpleReleaser(basic_releaser.SimpleReleaserMixin, SamogonBasicReleaser):
    def __init__(self, input_parameters, token):
        super(SamogonSimpleReleaser, self).__init__(input_parameters, token)
        self._samogon_deploy_api = samogon_api.SamogonDeployAPI("https://api-deploy.n.yandex-team.ru", token)

    def do_release(self):
        LOGGER.info("Start releasing")
        results = []
        for item in self._input.release_items:
            results.append(self._do_release_one_item(
                samogon_api.SamogonPkgSandbox(item.samogon_release_item.sb_resource),
                item.samogon_release_item.project,
                item.samogon_release_item.namespace,
            ))
        return results

    def get_data_for_deploy_results(self):
        return [self._make_deploy_data(item).to_dict() for item in self._input.release_items]

    @staticmethod
    def _make_deploy_data(release_item):
        item = release_item.samogon_release_item
        return SamogonDeployData(
            item.project,
            item.url,
            item.sb_resource.id,
        )


@decorators.decorate_all_public_methods(decorators.log_start_and_finish(LOGGER))
class SamogonReleaser(basic_releaser.SbReleaserMixin, SamogonBasicReleaser):
    def __init__(self, task, c_info):
        super(SamogonReleaser, self).__init__(task, c_info)
        self._token = sdk2.Vault.data(self._task.Parameters.samogon_oauth_token_vault_name)
        self._samogon_deploy_api = samogon_api.SamogonDeployAPI("https://api-deploy.n.yandex-team.ru", self._token)

    def do_release(self):
        return [
            self._do_release_one_item(
                samogon_api.SamogonPkgSandbox(self._task.Parameters.component_resource),
                self._task.Parameters.project,
                self._task.Parameters.namespace,
            ),
            basic_releaser.press_release_button_on_build_task(
                self._task.Parameters.component_resource.task_id,
                self._task.Parameters.where_to_release,
                self._sb_rest_client,
            )
        ]

    def get_data_for_deploy_results(self):
        return [SamogonDeployData(
            self._task.Parameters.project,
            self._task.Parameters.clusterapi_url,
            self._task.Parameters.component_resource.id,
        ).to_dict()]


class SamogonReleaseWatcher(basic_releaser.BaseReleaseWatcher):

    def __init__(self, parameters, token):
        super(SamogonReleaseWatcher, self).__init__(parameters, token)
        self._deploy_data = SamogonDeployData.from_dict(parameters)
        cluster_url = "https://clusterapi-{}".format(self._deploy_data.url)
        self._samogon_cluster_api = samogon_api.SamogonClusterAPI(cluster_url, self._token)

    def last_release(self, release_stage=None):
        raise NotImplementedError

    def last_deploy(self, release_stage=None):
        deployed_resource_ids = self._samogon_cluster_api.get_deployed_resource_ids()
        return [DeployedResource(int(resource_id)) for resource_id in deployed_resource_ids]
