import logging
import itertools
import requests
import time
from typing import (  # noqa: UnusedImport
    Any,
    Dict,
    List,
)

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import resource_selectors
from sandbox.projects.common import time_utils as tu
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine.helpers.deploy import basic_releaser


LOGGER = logging.getLogger(__name__)


class AnotherReleaseIsInProgress(Exception):
    pass


@decorators.decorate_all_public_methods(decorators.log_start_and_finish(LOGGER))
class SandboxBasicReleaser(basic_releaser.BasicReleaser):
    @decorators.memoized_property
    def _release_items(self):
        # type: () -> List[Tuple[sdk2.Resource, Any]]
        raise NotImplementedError

    def _get_release_params(self, release_item):
        # type: (Any) -> Dict
        raise NotImplementedError

    @property
    def _release_stage(self):
        # type: () -> str
        raise NotImplementedError

    def do_release(self):
        # type: () -> List[rm_core.Result]
        results = []
        released_task_ids = set()
        for release_item in self._release_items:
            resource, _ = release_item
            if resource.task_id in released_task_ids:
                LOGGER.info(
                    "Release process for resource '%s' was started successfully along with another item", resource.id
                )
                continue
            one_item_results = self._do_release_one_item(release_item)
            results.extend(one_item_results)
            if (release_result.ok for release_result in one_item_results):
                released_task_ids.add(resource.task_id)
            else:
                LOGGER.error("Release failed:\n%s", "\n".join(map(str, one_item_results)))
        if all(i.ok for i in results):
            results.extend(self._wait_release_results())
        return results

    def _do_release_one_item(self, release_item):
        # type: (Any) -> List[rm_core.Result]
        try:
            release_params = self._get_release_params(release_item)
        except Exception as e:
            return [
                rm_core.Error(
                    "Failed to get release params for {release_item}: {exception}\n{traceback}".format(
                        release_item=release_item,
                        exception=e,
                        traceback=eh.shifted_traceback(),
                    ),
                ),
            ]
        return [self._press_release_button_on_build_task(release_params)]

    @decorators.retries(
        5, 10,
        exceptions=AnotherReleaseIsInProgress,
        default_instead_of_raise=True,
        default_value=rm_core.Error("Sandbox release process failed")
    )
    def _press_release_button_on_build_task(self, release_params):
        # type: (Dict) -> rm_core.Result
        try:
            self._sb_rest_client.release(release_params)
        except requests.exceptions.HTTPError as http_error:
            error = str(http_error)
        except Exception as e:
            error = str(e)
        else:
            return rm_core.Ok("Release button in build task {} was pressed successfully".format(
                release_params.get("task_id")
            ))
        if "is already in progress" in error:  # RMINCIDENTS-162
            LOGGER.warning("Release button in build task %s was pressed by another task", release_params.get("task_id"))
            raise AnotherReleaseIsInProgress()
        LOGGER.exception(error)
        return rm_core.Error("Sandbox release process failed:\n{}\n{}".format(error, eh.shifted_traceback()))

    def _wait_release_results(self):
        # type: (Any) -> List[rm_core.Result]
        wait_time = 10  # 10 sec
        wait_timeout = 7 * 60  # 7 min
        start_time = int(time.time())
        for current_attempt in itertools.count():
            wait_results = []
            resources_info = self._sb_rest_client.resource.read(
                id=[resource.id for resource, _ in self._release_items],
                limit=len(self._release_items),
            )["items"]
            for i in resources_info:
                wait_results.append(check_resource_released_attr(i, self._release_stage))
            if any(not r.ok for r in wait_results) and int(time.time()) - start_time < wait_timeout:
                wait_for = [
                    (item[0], wait_results[i].result)
                    for i, item in enumerate(self._release_items) if not wait_results[i].ok
                ]
                LOGGER.info("Waiting for release of items #%s: %s", current_attempt, wait_for)
                time.sleep(wait_time)
            else:
                return wait_results


class SandboxSimpleReleaser(basic_releaser.SimpleReleaserMixin, SandboxBasicReleaser):

    @property
    def common_release_data(self):
        return self._input.config.common_release_data.sandbox_common_release_data

    @decorators.memoized_property
    def _release_items(self):
        # type: () -> List[Tuple[sdk2.Resource, Any]]
        return [
            (sdk2.Resource[i.sandbox_release_item.id], i) for i in self._input.release_items
        ]

    def _get_release_params(self, release_item):
        # type: (Any) -> Dict
        return {
            "type": str(self.common_release_data.release_stage),
            "subject": str(self.common_release_data.release_subject),
            "message": str(self.common_release_data.release_notes),
            "cc": str(self.common_release_data.release_followers),
            "to": [],
            "params": {},
            "task_id": release_item[0].task_id,
        }

    @decorators.memoized_property
    def _release_stage(self):
        # type: () -> str
        return str(self.common_release_data.release_stage)


class SandboxReleaser(basic_releaser.SbReleaserMixin, SandboxBasicReleaser):
    @decorators.memoized_property
    def _release_items(self):
        # type: () -> List[Tuple[sdk2.Resource, ReleasedResourceInfo]]
        result = []

        for res_info in self._c_info.releases_cfg__resources_info:

            LOGGER.info("Consider %s. Resource name: %s", res_info, res_info.resource_name)

            resource_id = self._task.Parameters.component_resources.get(res_info.resource_name, None)

            if not resource_id:
                LOGGER.info("No id specified for this resource (skipping it)")
                continue

            resource = sdk2.Resource[resource_id]

            if resource.type != res_info.resource_type:
                LOGGER.error(
                    "Resource #%s has type %s while expected %s (specified in component config)",
                    resource_id,
                    resource.type,
                    res_info.resource_type,
                )

            result.append((resource, res_info))

        return result

    def _get_release_params(self, release_item):
        # type: (Any) -> Dict
        if self._task.Parameters.additional_release_parameters:
            add_release_parameters = self._task.Parameters.additional_release_parameters.copy()
        else:
            add_release_parameters = {}
        resource, released_res_info = release_item
        return {
            "to": [],
            "params": add_release_parameters,
            "task_id": resource.task_id,
            "cc": self._c_info.get_release_followers(self._task.Parameters.where_to_release),
            "message": self._c_info.get_release_notes(
                self._task.Parameters.where_to_release,
                [released_res_info.name],
                self._task.author,
                self._task.Parameters.major_release_num,
                self._task.Parameters.minor_release_num,
                custom_release_notes=self._task.Parameters.additional_release_notes
            ),
            "type": self._task.Parameters.where_to_release,
            "subject": self._c_info.get_release_subject(
                self._task.Parameters.where_to_release,
                self._task.Parameters.major_release_num,
                self._task.Parameters.minor_release_num,
            ),
        }

    @property
    def _release_stage(self):
        # type: () -> str
        return self._task.Parameters.where_to_release

    def _do_release_one_item(self, release_item):
        # type: (Any) -> List[rm_core.Result]
        results = super(SandboxReleaser, self)._do_release_one_item(release_item)
        if all(i.ok for i in results):
            from release_machine.common_proto import events_pb2 as rm_proto_events

            release_params = self._get_release_params(release_item)
            self._release_info.append(rm_proto_events.ReleasedResourceInfo(
                release_label=str(release_params["subject"]),
                build_task_id=str(release_params["task_id"]),
                resource_id=str(release_item[0].id),
                resource_type=str(release_item[0].type),
            ))
        return results

    def after_release(self, release_results):
        super(SandboxBasicReleaser, self).after_release(release_results)
        if all(i.ok for i in release_results):
            self._set_release_number_attributes([i[0] for i in self._release_items])


class SandboxReleaseWatcher(basic_releaser.BasicReleaseWatcher):
    def _get_released_resources_data(self, release_stage=None):
        result = []
        for res_info, deploy_info in self._c_info.releases_cfg__iter_over_deploy_info(release_stage):
            attrs = {"released": deploy_info.level}
            attrs.update(res_info.attributes or {})
            last_released_res = sdk2.Resource.find(
                type=res_info.resource_type,
                status=ctr.State.READY,
                attrs=attrs,
            ).first()
            LOGGER.debug("Last released resource for '%s': %s", res_info, last_released_res.id)
            released_item = rm_core.ReleasedItem(res_info.resource_name, last_released_res, res_info.build_ctx_key)
            major_num, minor_num = self._c_info.get_release_numbers(released_item)
            result.append({
                "id": last_released_res.id,
                "build_task_id": last_released_res.task_id,
                "major_release": major_num,
                "minor_release": minor_num,
                "timestamp": int(tu.datetime_to_timestamp(last_released_res.updated)),
                "component": self._c_info.name,
                "status": deploy_info.level,
                "owner": last_released_res.owner,
                "resource_name": res_info.resource_name,
            })
        return result

    def last_release(self, release_stage=None):
        return [rm_core.ReleasedResource(**data) for data in self._get_released_resources_data(release_stage)]

    def last_deploy(self, release_stage=None):
        """
        For SandboxReleaser deploy === release, so find last released resources.

        :return: List[rm_core.DeployedResource]
        """
        return [rm_core.DeployedResource(**data) for data in self._get_released_resources_data(release_stage)]

    def last_deploy_proto(self, item_data, deploy_info):
        # type: (SandboxResourceData, SandboxInfo) -> List[Any]
        from release_machine.common_proto import release_and_deploy_pb2
        last_released_resource_id, release_time = resource_selectors.by_last_released_task(
            item_data.resource_type, item_data.attributes, deploy_info.stage
        )
        if not last_released_resource_id:
            return []
        last_released_resource = sdk2.Resource[last_released_resource_id]
        major_num, minor_num = self._c_info.get_release_numbers_from_resource(last_released_resource, item_data)
        build_arc_url = rm_utils.get_input_or_ctx_field(last_released_resource.task_id, item_data.build_ctx_key)
        arc_hash, svn_revision = self._get_revision(build_arc_url)
        service_version = release_and_deploy_pb2.ServiceVersion(
            stage_label=deploy_info.stage,
            arc_hash=arc_hash,
            svn_revision=svn_revision,
            major_release_number=major_num,
            minor_release_number=minor_num,
            timestamp=tu.to_unixtime(release_time),
            sandbox_resource=release_and_deploy_pb2.SbResourceData(
                resource_id=last_released_resource.id,
                resource_type=item_data.resource_type,
                build_task_id=last_released_resource.task_id,
            ),
        )
        logging.info("Got service_version:\n%s", service_version)
        return [service_version]


def check_resource_released_attr(resource_info, stage):
    # type: (Dict, str) -> rm_core.Result
    if resource_info["attributes"].get("released") != stage:
        return rm_core.Error(
            "Attribute 'released': {observed_attr_value} != {expected_attr_value}".format(
                observed_attr_value=resource_info["attributes"].get("released"),
                expected_attr_value=stage,
            ),
        )
    elif resource_info["task"].get("status") != ctt.Status.RELEASED:
        return rm_core.Error(
            "Build task status: {observed_status} != {expected_status}".format(
                observed_status=resource_info["task"].get("status"),
                expected_status=ctt.Status.RELEASED,
            ),
        )
    else:
        return rm_core.Ok("Resource {} was released successfully".format(resource_info["id"]))
