import collections
import datetime as dt
import itertools as it
import logging
import json
import re
import socket
import time
from dateutil.tz import gettz
from typing import (  # noqa: UnusedImport
    Any,
    Dict,
    Generator,
    List,
    Optional,
    Tuple,
    Union,
)

from sandbox import sdk2
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string
from sandbox.projects.release_machine.core import releasable_items as ri  # noqa: UnusedImport
from sandbox.projects.release_machine.helpers.deploy import basic_releaser
from sandbox.projects.release_machine.helpers.deploy import sandbox_releaser
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine import security as rm_sec


LOGGER = logging.getLogger(__name__)
DeployTicketData = collections.namedtuple("DeployTicketData", ("ticket_id", "stage_id", "progress"))
RELEASE_EXISTS_RE = re.compile(r"Release \"(.*)\" already exists")


def _stringify_resource_attributes(resource):
    # ttl resource attribute can have value which represents infinity or -infinity.
    # This values cannot be parsed by YP json decoder. So we cast values of all
    # attributes to strings.
    return {k: str(v) for k, v in resource}


class YaDeploySimpleReleaser(sandbox_releaser.SandboxSimpleReleaser):
    """Release sandbox resources to ya.deploy."""

    def __init__(self, input_parameters, token):
        from yp import client
        super(YaDeploySimpleReleaser, self).__init__(input_parameters, token)
        self._common_yp_client = client.YpClient("xdc.yp.yandex.net:8090", config={'token': token})
        self._tickets_to_wait = []

    @property
    def common_release_data(self):
        return self._input.config.common_release_data.ya_deploy_common_release_data

    @decorators.memoized_property
    def _release_items(self):
        # type: () -> List[Tuple[sdk2.Resource, Any]]
        return [
            (sdk2.Resource[i.ya_deploy_release_item.id], i) for i in self._input.release_items
        ]

    def do_release(self):
        # type: () -> List[rm_core.Result]
        # these errors will be rewritten if release to yp created ok
        yp_release_object_result = rm_core.Error("Creating yp release was skipped due to previous fail")
        sb_release_button_result = [rm_core.Error("Pressing release button was skipped due to previous fail")]
        yp_commit_tickets_results = []
        construct_release_data_result = self.construct_release_data_object()
        if construct_release_data_result.ok:
            yp_release_object_result = create_yp_release(construct_release_data_result.result, self._common_yp_client)
            if yp_release_object_result.ok:
                if self._input.config.commit_release_tickets:
                    time.sleep(3)  # initial wait for tickets creation
                    release_processing_status = wait_for_release_processing(
                        self._release_meta_id,
                        self._common_yp_client,
                    )
                    if release_processing_status.ok:
                        all_tickets = select_deploy_tickets(self._release_meta_id, self._common_yp_client)
                        if not all_tickets:
                            yp_commit_tickets_results.append(rm_core.Error(
                                "No deploy tickets found for release object: {}. "
                                "Please, check release rule in your ya.deploy stage".format(self._release_meta_id)
                            ))
                        else:
                            tickets_to_commit = filter_deploy_tickets(
                                all_tickets,
                                filter_by_closed=True,
                                filter_by_stage=self.common_release_data.stage_name,
                            )
                            for i in tickets_to_commit:
                                self._tickets_to_wait.append(i.ticket_id)
                                yp_commit_tickets_results.append(commit_release_ticket(
                                    i.ticket_id, self._common_yp_client,
                                    message=self.common_release_data.release_ticket_commit_message
                                ))
                    else:
                        yp_commit_tickets_results.append(release_processing_status)
                # replace this with line after it if you do not want to wait for release of SB-task while debugging
                sb_release_button_result = super(YaDeploySimpleReleaser, self).do_release()
                # sb_release_button_result = [rm_core.Ok("TMP skip this")]

        return [
            construct_release_data_result,
            yp_release_object_result,
        ] + yp_commit_tickets_results + sb_release_button_result

    @decorators.memoized_property
    def _release_artifact_name(self):
        from release_machine.tasklets.simple_releaser.proto import simple_releaser_pb2
        return simple_releaser_pb2.ReleaseArtifactType.Name(self.common_release_data.release_artifact_type)

    def construct_release_data_object(self):
        # type: () -> rm_core.Result
        try:
            if self._release_artifact_name == "SANDBOX_RESOURCE":
                return rm_core.Ok(self._release_resource_data_obj())
            elif self._release_artifact_name == "DOCKER_IMAGE":
                return rm_core.Ok(self._release_docker_data_obj())
            return rm_core.Error("Unknown artifact type = {}".format(self.common_release_data.release_artifact_type.name))
        except Exception as e:
            return rm_core.Error("Release data construction failed:\n{}\n\n{}".format(e, eh.shifted_traceback()))

    def _release_resource_data_obj(self):
        # type: () -> Dict
        title = self.common_release_data.release_subject
        description = self.common_release_data.release_notes
        first_resource = self._release_items[0][0]  # let the first release item be main
        task_id = first_resource.task_id
        build_task_info = self._sb_rest_client.task[task_id].read()
        task_type = str(build_task_info["type"])  # this type will be used in release rule
        return {
            "meta": {
                "id": self._release_meta_id,
            },
            "spec": {
                "title": title,
                "description": description,
                "sandbox": {
                    "title": title,
                    "description": description,
                    "task_id": str(task_id),  # todo: use release task id (or its analogue in tasklet world)
                    "task_type": task_type,
                    "release_author": "robot-srch-releaser",  # todo get author from input
                    "release_type": self.common_release_data.release_stage,
                    "resources": [
                        {
                            "resource_id": str(resource.id),
                            "type": str(resource.type),
                            "description": resource.description,
                            "skynet_id": resource.skynet_id,
                            "attributes": _stringify_resource_attributes(resource),
                        } for resource, _ in self._release_items
                    ],
                },
            },
        }

    def _release_docker_data_obj(self):
        # type: () -> Dict
        registry = "registry.yandex.net"
        images_data = [
            string.left_strip(i.ya_deploy_release_item.attributes["resource_version"], "{}/".format(registry)).split(":")
            for _, i in self._release_items
        ]
        return {
            "meta": {
                "id": self._release_meta_id
            },
            "spec": {
                "title": self.common_release_data.release_subject,
                "description": self.common_release_data.release_notes,
                "docker": {
                    "images": [
                        {
                            "name": image_name,
                            "tag": image_tag,
                            "digest": "EMPTY",
                            "registry_host": registry
                        } for image_name, image_tag in images_data
                    ],
                    "release_type": self.common_release_data.release_stage,
                    "release_author": "robot-srch-releaser",  # todo get author from input
                }
            },
        }

    @decorators.memoized_property
    def _release_meta_id(self):
        return "RM-{}-{}-{}".format(
            self._release_artifact_name.replace("_", "-"),
            self.common_release_data.release_stage.upper(),
            self._release_items[0][0].task_id
        )

    def get_data_for_deploy_results(self):
        if self._tickets_to_wait:
            return [YaDeployTicketData(i).to_dict() for i in self._tickets_to_wait]
        return [YaDeployData(self._release_meta_id).to_dict()]

    def get_deploy_result(self, data):
        try:
            return get_deploy_result_by_deploy_ticket(self._common_yp_client, YaDeployTicketData.from_dict(data))
        except TypeError:
            return get_deploy_result_by_release_object(self._common_yp_client, YaDeployData.from_dict(data))

    def after_deploy(self, deploy_results):
        try:
            deploy_tickets_data = [YaDeployTicketData.from_dict(data) for data in deploy_results]
            stages = self._common_yp_client.get_objects(
                "deploy_ticket",
                [i.ticket_id for i in deploy_tickets_data],
                ["/meta/stage_id"],
            )
        except TypeError:
            deploy_data = [YaDeployData.from_dict(data) for data in deploy_results]
            stages = list(it.chain.from_iterable(self._common_yp_client.select_objects(
                "deploy_ticket",
                filter="[/spec/release_id]='{}'".format(i.release_object_id),
                selectors=["/meta/stage_id"],
                limit=100,
            ) for i in deploy_data))
        return self._resolve_endpoints(stages)

    @decorators.retries(4, delay=4)
    def _resolve_endpoints(self, stages):
        from infra.yp_service_discovery.api import api_pb2
        from infra.yp_service_discovery.python.resolver.resolver import Resolver
        from release_machine.tasklets.simple_releaser.proto import simple_releaser_pb2
        logging.info("Going to resolve endpoints for stages: %s", stages)
        deploy_units_info = self._common_yp_client.get_objects(
            "stage", [i[0] for i in stages],
            ["/status/deploy_units"],
        )
        logging.info("Got deploy units info: %s", deploy_units_info)
        client_name = "release_machine:{}".format(socket.gethostname())
        resolver = Resolver(client_name=client_name, timeout=20)
        new_endpoints = []
        for i in deploy_units_info:
            for deploy_unit_name, deploy_unit_data in i[0].items():
                try:
                    if "replica_set" in deploy_unit_data:
                        cluster_statuses = deploy_unit_data["replica_set"]["cluster_statuses"]
                    elif "multi_cluster_replica_set" in deploy_unit_data:
                        cluster_statuses = deploy_unit_data["multi_cluster_replica_set"]["cluster_statuses"]
                    else:
                        logging.warning(
                            "Skip resolving %s! Unable to find replica set data in %s",
                            deploy_unit_name, deploy_unit_data
                        )
                        cluster_statuses = {}
                except KeyError:
                    logging.warning("Skip resolving %s! Empty pod set in %s", deploy_unit_name, deploy_unit_data)
                    continue

                for cluster_name, cluster_data in cluster_statuses.items():
                    request = api_pb2.TReqResolveEndpoints()
                    request.cluster_name = cluster_name
                    request.endpoint_set_id = cluster_data["endpoint_set_id"]
                    request.client_name = client_name
                    result = _resolve_single_cluster_endpoints(resolver, request)
                    new_endpoints.append(simple_releaser_pb2.YaDeployServiceEndpoint(
                        deploy_unit=deploy_unit_name,
                        cluster_name=cluster_name,
                        endpoint_set=result.endpoint_set,
                    ))
        return YaDeployAfterDeployResults(service_endpoints=new_endpoints)


@decorators.retries(3)
def _resolve_single_cluster_endpoints(resolver, request):
    return resolver.resolve_endpoints(request)


class YaDeployReleaser(sandbox_releaser.SandboxReleaser):
    """Release rm component sandbox resources to ya.deploy."""

    def __init__(self, task, c_info):
        super(YaDeployReleaser, self).__init__(task, c_info)

        from yp import client
        self._token = rm_sec.get_rm_token(task)
        self._common_yp_client = client.YpClient("xdc.yp.yandex.net:8090", config={"token": self._token})

    def do_release(self):
        # type: () -> List[rm_core.Result]
        # these errors will be rewritten if release to yp created ok
        yp_release_object_result = rm_core.Error("Creating yp release was skipped due to previous fail")
        sb_release_button_result = [rm_core.Error("Pressing release button was skipped due to previous fail")]
        construct_release_data_result = self.construct_release_data_object()
        if construct_release_data_result.ok:
            yp_release_object_result = create_yp_release(construct_release_data_result.result, self._common_yp_client)
            if yp_release_object_result.ok:
                # replace this with line after it if you do not want to wait for release of SB-task while debugging
                sb_release_button_result = super(YaDeployReleaser, self).do_release()
                # sb_release_button_result = [rm_core.Ok("TMP skip this")]

        return [construct_release_data_result, yp_release_object_result] + sb_release_button_result

    def construct_release_data_object(self):
        # type: () -> rm_core.Result
        try:
            title = self._c_info.get_release_subject(
                self._task.Parameters.where_to_release,
                self._task.Parameters.major_release_num,
                self._task.Parameters.minor_release_num,
            )
            description = self._c_info.get_release_notes(
                self._task.Parameters.where_to_release,
                [res_info.name for _, res_info in self._release_items],
                self._task.author,
                self._task.Parameters.major_release_num,
                self._task.Parameters.minor_release_num,
            )
            if self._c_info.releases_cfg__use_release_task_as_ti_task_type:
                task_id = self._task.id
                task_type = str(self._task.type)
            else:
                first_resource, _ = self._release_items[0]  # let the first release item be main
                task_id = first_resource.task_id
                build_task_info = self._sb_rest_client.task[task_id].read()
                task_type = str(build_task_info["type"])  # this type will be used in release rule
            resources = []
            for resource, res_info in self._release_items:
                attrs = {
                    "major_release_num": str(self._task.Parameters.major_release_num),
                    "minor_release_num": str(self._task.Parameters.minor_release_num),
                    "rm_release_item_name": str(res_info.name),
                }
                if res_info.attributes:
                    attrs.update(res_info.attributes)
                resources.append({
                    "resource_id": str(resource.id),
                    "type": str(resource.type),
                    "description": resource.description,
                    "skynet_id": resource.skynet_id,
                    "attributes": attrs,
                })
            release_data = {
                "meta": {
                    "id": self._release_meta_id,
                },
                "spec": {
                    "title": title,
                    "description": description,
                    "sandbox": {
                        "title": title,
                        "description": description,
                        "task_id": str(task_id),
                        "task_type": task_type,
                        "release_author": str(self._task.author),
                        "release_type": self._task.Parameters.where_to_release,
                        "resources": resources,
                    }
                },
            }
            return rm_core.Ok(release_data)
        except Exception as e:
            msg = "Release data construction failed"
            logging.exception(msg)
            return rm_core.Error("{}:\n{}".format(msg, e))

    @decorators.memoized_property
    def _release_meta_id(self):
        return "SANDBOX-RM-{}-{}".format(self._task.Parameters.where_to_release.upper(), self._task.id)

    def get_data_for_deploy_results(self):
        return [YaDeployData(self._release_meta_id).to_dict()]

    def get_deploy_result(self, data):
        return get_deploy_result_by_release_object(self._common_yp_client, YaDeployData.from_dict(data))


def get_yp_release(release_id, selectors, common_yp_client):
    try:
        yp_release_info = common_yp_client.get_object(
            "release", release_id,
            selectors=selectors
        )
        LOGGER.debug(
            "Got yt release info:\n%s",
            json.dumps(dict(zip(selectors, yp_release_info)), indent=2),
        )
        return yp_release_info
    except Exception as e:  # case, when release was not created yet
        LOGGER.info(e)


def create_yp_release(release_data, common_yp_client):
    # type: (Dict, Any) -> rm_core.Result
    LOGGER.info("Release data:\n%s", json.dumps(release_data, indent=2))
    try:
        common_yp_client.create_object("release", attributes=release_data)
    except Exception as e:
        existing_release_id_found = RELEASE_EXISTS_RE.search(str(e))
        if existing_release_id_found:
            existing_release_id = existing_release_id_found.group(1)
            release_author, release_creation_time = get_yp_release(
                existing_release_id, ["/meta/author_id", "/meta/creation_time"], common_yp_client
            )
            return rm_core.Ok("Release object already exists: {}. Created by {} at {}".format(
                existing_release_id, release_author, _ts_to_iso(release_creation_time / 1000000.0)
            ))
        LOGGER.exception("Failed to create release object")
        return rm_core.Error("Creating release object [{}] failed:\n{}".format(release_data["meta"]["id"], e))
    return rm_core.Ok("Release process was started successfully. Release object id: '{}'.".format(release_data["meta"]["id"]))


def commit_release_ticket(deploy_ticket_id, common_yp_client, message=None):
    # type: (List[DeployTicketData], Any, str) -> rm_core.Result
    LOGGER.info("Committing ticket %s", deploy_ticket_id)
    try:
        return _commit_release_ticket(deploy_ticket_id, common_yp_client, message)
    except Exception as e:
        return rm_core.Error("Failed to commit ticket:\n{}".format(e))


@decorators.retries(3, delay=2)
def _commit_release_ticket(deploy_ticket_id, common_yp_client, message=None):
    # type: (List[DeployTicketData], Any, str) -> rm_core.Result
    try:
        response = common_yp_client.update_object(
            "deploy_ticket", deploy_ticket_id,
            set_updates=[{
                "path": "/control/commit",
                "value": {
                    "options": {
                        "message": message or "Autocommit from release task",
                        "patch_selector": {"type": "full"},
                        "reason": "ManualCommit by RM",
                    }
                }
            }]
        )
        LOGGER.info("Commit ticket response: %s", response)
        return rm_core.Ok("Ticket {} was committed successfully.".format(deploy_ticket_id))
    except Exception as e:
        if 'has terminal state "Commit"' in str(e):
            return rm_core.Ok("Ticket {} was already committed, skip commit.".format(deploy_ticket_id))
        raise e


def wait_for_release_processing(release_id, common_yp_client):
    # type: (str, Any) -> rm_core.Result
    start_time = time.time()
    total_wait_time_sec = 180  # 3 min
    for wait_cycle_time in it.count(2):
        release_object_status = get_yp_release(release_id, ["/status/processing/finished"], common_yp_client)
        release_object_status = "Release not created yet" if release_object_status is None else release_object_status[0]
        if isinstance(release_object_status, dict) and release_object_status.get("status"):
            return rm_core.Ok("Release object processing finished. Status: {}".format(release_object_status.get("status")))
        time_spent = time.time() - start_time
        if time_spent > total_wait_time_sec:
            return rm_core.Error("Failed to wait for release process finish")
        LOGGER.info("Wait %s sec for deploy tickets (already waited %s sec)", wait_cycle_time, time_spent)
        time.sleep(wait_cycle_time)


def select_deploy_tickets(release_id, common_yp_client):
    # type: (str, Any) -> List[DeployTicketData]
    tickets = common_yp_client.select_objects(
        "deploy_ticket",
        filter="[/spec/release_id]='{}'".format(release_id),
        selectors=["/meta/id", "/meta/stage_id", "/status/progress"],
        limit=100,
    )
    LOGGER.info("Got (%s) tickets:\n%s", len(tickets), "\n".join(map(str, tickets)))
    return [DeployTicketData(*i) for i in tickets]


def filter_deploy_tickets(tickets_data, filter_by_closed=True, filter_by_stage=None):
    # type: (List[DeployTicketData], bool, Optional[str]) -> Generator[DeployTicketData]
    for ticket_data in tickets_data:
        if filter_by_closed and isinstance(ticket_data.progress, dict) and ticket_data.progress.get("closed"):
            LOGGER.info("Ticket %s already closed, skip it", ticket_data.ticket_id)
            continue
        if filter_by_stage and ticket_data.stage_id != filter_by_stage:
            LOGGER.info("Ticket stage %s != %s, skip it", ticket_data.stage_id, filter_by_stage)
            continue
        yield ticket_data


def get_deploy_result_by_release_object(common_yp_client, deploy_data):
    # type: (Any, YaDeployData) -> Optional[rm_core.Result]
    if deploy_data.status == "closed":
        return
    release_object_status = common_yp_client.get_object(
        "release",
        deploy_data.release_object_id,
        selectors=["/status"]
    )[0]
    LOGGER.debug("Release object status (%s): %s", type(release_object_status), release_object_status)
    release_object_progress = release_object_status.get("progress", {})
    LOGGER.debug("Release object progress (%s): %s", type(release_object_progress), release_object_progress)
    status = release_object_progress.get("closed", {}).get("status")
    end_time = release_object_progress.get("end_time", {}).get("seconds")
    if status == "true":
        deploy_data.status = "closed"
        result = rm_core.Ok
    else:
        result = rm_core.Error
    if end_time:
        deploy_data.end_time = _ts_to_iso(end_time)
    return result(deploy_data.to_dict())


def get_deploy_result_by_deploy_ticket(common_yp_client, deploy_tickets_data):
    # type: (Any, YaDeployTicketData) -> Optional[rm_core.Result]
    if deploy_tickets_data.end_time is not None:
        return
    ticket_progress = common_yp_client.get_object(
        "deploy_ticket", deploy_tickets_data.ticket_id,
        selectors=["/status/progress"]
    )[0]
    LOGGER.debug("Ticket %s progress: %s", deploy_tickets_data.ticket_id, ticket_progress)
    if not isinstance(ticket_progress, dict):
        LOGGER.debug("No info about ticket progress found")
        return rm_core.Error(deploy_tickets_data.to_dict())  # no progress -> nothing to update
    status = ticket_progress.get("closed", {}).get("status")
    end_time = ticket_progress.get("end_time", {}).get("seconds")
    LOGGER.info("Got ticket status and end time: %s, %s", status, end_time)
    if status == "true":
        deploy_tickets_data.status = "closed"
        result = rm_core.Ok
        if end_time:
            deploy_tickets_data.end_time = _ts_to_iso(end_time)
    else:
        result = rm_core.Error
    return result(deploy_tickets_data.to_dict())


def _ts_to_iso(timestamp):
    return dt.datetime.utcfromtimestamp(timestamp).replace(tzinfo=gettz("Europe/Moscow")).isoformat()


class YaDeployData(basic_releaser.DeployData):
    __slots__ = ("release_object_id", "status", "end_time")

    def __init__(self, release_object_id, status=None, end_time=None):
        self.release_object_id = release_object_id
        self.end_time = end_time
        self.status = status

    def __str__(self):
        return "YaDeployData[{}]".format(self.release_object_id)

    def to_dict(self):
        return {
            "release_object_id": self.release_object_id,
            "end_time": self.end_time,
            "status": self.status,
        }

    __repr__ = __str__


class YaDeployTicketData(basic_releaser.DeployData):
    __slots__ = ("ticket_id", "status", "end_time")

    def __init__(self, ticket_id, status=None, end_time=None):
        self.ticket_id = ticket_id
        self.status = status
        self.end_time = end_time

    def __str__(self):
        return "YaDeployTicketData{}".format(self.ticket_id)

    def to_dict(self):
        return {
            "ticket_id": self.ticket_id,
            "end_time": self.end_time,
            "status": self.status,
        }

    __repr__ = __str__


class YaDeployAfterDeployResults(basic_releaser.AfterDeployResults):
    __slots__ = ("service_endpoints",)

    def __init__(self, service_endpoints):
        self.service_endpoints = service_endpoints

    def __str__(self):
        return "YaDeployTicketData (service_endpoints={})".format(self.service_endpoints)

    __repr__ = __str__


class YaDeployReleaseWatcher(basic_releaser.BasicReleaseWatcher):
    def __init__(self, c_info, token=None):
        super(YaDeployReleaseWatcher, self).__init__(c_info, token)

        from yp import client
        self._yp = client.YpClient("xdc.yp.yandex.net:8090", config={"token": self._token})

    def last_deploy(self, release_stage=None):
        pass

    def last_release(self, release_stage=None):
        pass

    def last_deploy_proto(self, item_data, deploy_info):
        # type: (Union[ri.SandboxResourceData, ri.DockerImageData], ri.YaDeployInfo) -> List[Any]
        LOGGER.info("Getting versions of %s", item_data)
        from release_machine.common_proto import release_and_deploy_pb2

        deploy_units_info = self._yp.get_objects(
            "stage",
            [s.name for s in deploy_info.services],
            ["/spec/deploy_units", "/status/deploy_units"],
        )

        versions = []

        for service, du_info in zip(deploy_info.services, deploy_units_info):
            LOGGER.info("Deploy unit info: %s", du_info)

            service_version = release_and_deploy_pb2.ServiceVersion(
                stage_label=deploy_info.stage,
                tags=service.tags if service.tags else None,
                ya_deploy=release_and_deploy_pb2.YaDeployData(stage=service.name),
            )

            try:
                self._update_service_version(item_data, du_info, service_version, release_and_deploy_pb2)
            except Exception:
                LOGGER.exception("Failed to update service version. Will post empty one.")

            LOGGER.info("Got service_version:\n%s", service_version)
            versions.append(service_version)

        return versions

    def _update_service_version(self, item_data, du_info, service_version, release_and_deploy_pb2):

        LOGGER.info("Trying to update local service version info")

        du_spec, du_status = du_info

        for sub_unit_name, data in du_spec.items():

            LOGGER.info("Considering deploy unit: %s", sub_unit_name)

            ready = du_status[sub_unit_name].get("ready")

            if not ready or not ready.get("status"):
                LOGGER.warning(
                    "Not ready deploy unit, skip it: %s. Status: %s",
                    sub_unit_name, du_status[sub_unit_name]
                )
                continue

            service_version.timestamp = ready.get("last_transition_time", {}).get("seconds")
            pod_template_spec = _get_pod_template_spec(data)

            LOGGER.info("Got pod template spec: %s", pod_template_spec)

            if pod_template_spec is None:
                LOGGER.warning("Unable to find pod_template_spec, skip %s", sub_unit_name)
                continue

            pod_agent_payload_spec = pod_template_spec["spec"]["pod_agent_payload"]["spec"]
            res = pod_agent_payload_spec.get("resources")

            if res:

                LOGGER.info("Resources (spec) found: %s", res)

                resources_and_layers = res.get("static_resources", []) + res.get("layers", [])

                if resources_and_layers:
                    sb_res = _get_sb_resource_data(item_data, resources_and_layers)
                    if sb_res:
                        self.__update_service_version_sb(
                            item_data, service_version, release_and_deploy_pb2,
                            sb_res.get("resource_id"), sb_res.get("task_id"),
                        )
                        break

            LOGGER.info("Resources and layers not found. Will try to find images for boxes")

            if item_data.docker_image_data_re:

                boxes = pod_agent_payload_spec.get("boxes", [])
                images_for_boxes = data["images_for_boxes"]

                for box in boxes:

                    logging.info("Considering box %s", box)

                    image_data = images_for_boxes.get(box["id"])

                    if image_data:

                        match = item_data.docker_image_data_re.match(image_data.get("tag"))

                        if match:

                            task_id = match.group("task_id")

                            logging.info("Got task ID: %s", task_id)

                            try:
                                sb_res = self._sb_rest_client.resource.read(
                                    type=item_data.resource_type,
                                    task_id=task_id,
                                    limit=1,
                                )["items"][0]
                            except IndexError:
                                logging.exception("Deploy unit processing failed")
                                continue

                            self.__update_service_version_sb(
                                item_data, service_version, release_and_deploy_pb2,
                                sb_res["id"], sb_res["task"]["id"],
                            )
                            break

            else:
                LOGGER.info("docker_image_data_re is not set. Set it in config")
            LOGGER.info("Images for boxes not found. Continue with next sub unit")

    def __update_service_version_sb(
        self, item_data, service_version, release_and_deploy_pb2,
        sb_resource_id=None, sb_task_id=None,
    ):
        if sb_resource_id:
            last_released_resource = sdk2.Resource[sb_resource_id]
            major_num, minor_num = self._c_info.get_release_numbers_from_resource(last_released_resource, item_data)
            build_arc_url = rm_utils.get_input_or_ctx_field(last_released_resource.task_id, item_data.build_ctx_key)
            arc_hash, svn_revision = self._get_revision(build_arc_url)
            service_version.arc_hash = arc_hash
            service_version.svn_revision = svn_revision
            service_version.major_release_number = major_num
            service_version.minor_release_number = minor_num
        service_version.sandbox_resource.CopyFrom(release_and_deploy_pb2.SbResourceData(
            resource_id=int(sb_task_id) if sb_task_id else 0,
            resource_type=item_data.resource_type,
            build_task_id=int(sb_task_id) if sb_task_id else 0,
        ))


def _get_pod_template_spec(deploy_units_data):
    LOGGER.info("... getting pod template spec")

    if "replica_set" in deploy_units_data:
        LOGGER.info("'replica_set' key found in deploy unit data")
        return deploy_units_data["replica_set"]["replica_set_template"]["pod_template_spec"]

    elif "multi_cluster_replica_set" in deploy_units_data:
        LOGGER.info("'multi_cluster_replica_set' key found in deploy unit data")
        return deploy_units_data["multi_cluster_replica_set"]["replica_set"]["pod_template_spec"]

    LOGGER.info("Neither of ['replica_set', 'multi_cluster_replica_set'] found in %s", deploy_units_data)


def _get_sb_resource_data(item_data, resources_and_layers):
    for i in resources_and_layers:
        sb_res = i.get("meta", {}).get("sandbox_resource")
        if sb_res:
            LOGGER.info("Current resource: %s", sb_res)
            if sb_res["resource_type"] != item_data.resource_type:
                LOGGER.debug("Skip it, %s != %s", sb_res["resource_type"], item_data.resource_type)
                continue
            if item_data.attributes and any(
                sb_res["attributes"].get(k) != v for k, v in item_data.attributes.items()
            ):
                LOGGER.debug("Skip it because of different attributes")
                continue
            return sb_res
