from collections import OrderedDict
from dateutil.parser import parse as parse_datetime
import json
import logging
import os
import shutil
import typing  # noqa: UnusedImport

from sandbox import sdk2
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import time_utils as tu
from sandbox.projects.release_machine.core import releasable_items as ri  # noqa: UnusedImport
from sandbox.projects.release_machine.core import (
    DeployedResource,
    Ok,
    Error,
    const as rm_const,
)
from sandbox.projects.release_machine.helpers import svn_helper as rm_svn
from sandbox.projects.release_machine.helpers.deploy import basic_releaser


LOGGER = logging.getLogger(__name__)

REMOTE_YA_CONF_FOLDER = "arcadia:/arc/trunk/arcadia/build"
LOCAL_YA_CONF_FOLDER = "arcadia_build"
YA_CONF_FILE_NAME = "ya.conf.json"


class YaToolReleaser(basic_releaser.SbReleaserMixin, basic_releaser.BasicReleaser):
    def do_release(self):
        sdk2.svn.Arcadia.checkout(REMOTE_YA_CONF_FOLDER, LOCAL_YA_CONF_FOLDER, depth="files")
        local_ya_conf_file = os.path.join(LOCAL_YA_CONF_FOLDER, YA_CONF_FILE_NAME)
        with open(local_ya_conf_file, "r") as in_f:
            conf = json.load(in_f, object_pairs_hook=OrderedDict)
        set_ya_conf_field(conf, self._task.Parameters.change_json_path, self._task.Parameters.build_task_id)
        updated_conf = json.dumps(conf, indent=4, separators=(',', ': ')) + "\n"
        fu.write_file(local_ya_conf_file, updated_conf)
        LOGGER.info("Diff:")
        LOGGER.info(sdk2.svn.Arcadia.diff(LOCAL_YA_CONF_FOLDER))

        release_results = [
            basic_releaser.press_release_button_on_build_task(
                self._task.Parameters.build_task_id,
                self._task.Parameters.where_to_release,
                self._sb_rest_client,
            ),
            self._commit_updated_config(),
        ]
        if all(i.ok for i in release_results):

            from release_machine.common_proto import events_pb2 as rm_proto_events

            self._release_info.append(rm_proto_events.ReleasedResourceInfo(
                release_label="Release task {}: {}".format(
                    self._task.Parameters.build_task_id, self._task.Parameters.major_release_num
                ),
                build_task_id=str(self._task.Parameters.build_task_id),
            ))
        return release_results

    def _commit_updated_config(self):
        ssh_key = rm_svn.get_ssh_key(self, rm_const.COMMON_TOKEN_OWNER, rm_const.ARC_ACCESS_TOKEN_NAME)
        with ssh_key:
            try:
                sdk2.svn.Arcadia.commit(
                    LOCAL_YA_CONF_FOLDER,
                    "Update '{}' {} to {} value".format(
                        self._task.Parameters.where_to_release,
                        self._c_info.name,
                        self._task.Parameters.build_task_id
                    ),
                    user=rm_const.ROBOT_RELEASER_USER_NAME
                )
            except sdk2.svn.SvnError as e:
                LOGGER.debug("SvnError while committing (probably, it is OK): %s", e, exc_info=True)
                review = rm_const.REVIEW_URL_RE.search(str(e))
                if review:
                    return Ok("Review was posted: {}".format(lb.review_link(review.group(1))))
                else:
                    return Error("Review was not created!")
            except Exception as e:
                msg = "Unknown error: {}".format(e)
                LOGGER.exception(msg)
                return Error(msg)
        return Ok("Updated config was successfully committed!")

    def after_release(self, release_results):
        super(YaToolReleaser, self).after_release(release_results)
        if all(i.ok for i in release_results):
            resources_info = self._sb_rest_client.resource.read(
                task_id=self._task.Parameters.build_task_id,
                limit=10,
            )["items"]
            self._set_release_number_attributes(i[0] for i in get_release_items(self._c_info, resources_info))


class YaToolReleaseWatcher(basic_releaser.BasicReleaseWatcher):
    def last_release(self, release_stage=None):
        pass

    def last_deploy(self, release_stage=None):
        remote_ya_conf_file = os.path.join(REMOTE_YA_CONF_FOLDER, YA_CONF_FILE_NAME)
        conf = self._get_ya_conf(remote_ya_conf_file)

        result = []
        for res_info, deploy_info in self._c_info.releases_cfg__iter_over_deploy_info(release_stage):
            for service in deploy_info.services:
                got_build_task_id = get_ya_conf_field_val(conf, service)
                LOGGER.debug("Got build task id for service %s: %s", service, got_build_task_id)
                if got_build_task_id.ok:
                    build_task_id = got_build_task_id.result
                    resources_info = self._sb_rest_client.resource.read(
                        task_id=build_task_id,
                        limit=10,
                    )["items"]
                    resource = get_release_resource(res_info, resources_info)
                    major, minor = self._c_info.get_release_numbers_from_resource(resource, res_info)
                    result.append(DeployedResource(
                        build_task_id=build_task_id,
                        timestamp=get_deploy_timestamp(remote_ya_conf_file, build_task_id),
                        major_release=major,
                        minor_release=minor,
                        component=self._c_info.name,
                        status=deploy_info.level,
                        resource_name=res_info.resource_type,
                    ))
        return result

    def last_deploy_proto(self, item_data, deploy_info):
        # type: (ri.SandboxResourceData, ri.YaToolDeployInfo) -> typing.List[typing.Any]
        from release_machine.common_proto import release_and_deploy_pb2
        remote_ya_conf_file = os.path.join(REMOTE_YA_CONF_FOLDER, YA_CONF_FILE_NAME)
        conf = self._get_ya_conf(remote_ya_conf_file)
        result = []
        for service in deploy_info.services:
            got_build_task_id = get_ya_conf_field_val(conf, service.name)
            if got_build_task_id.ok:
                build_task_id = got_build_task_id.result
                task_info = self._sb_rest_client.task[build_task_id].context.read()
                resources_info = self._sb_rest_client.resource.read(
                    task_id=build_task_id,
                    limit=10,
                )["items"]
                build_arc_url = task_info.get(item_data.build_ctx_key)
                resource = get_release_resource(item_data, resources_info)
                major, minor = self._c_info.get_release_numbers_from_resource(resource, item_data)
                arc_hash, svn_revision = self._get_revision(build_arc_url)
                service_version = release_and_deploy_pb2.ServiceVersion(
                    stage_label=deploy_info.stage,
                    arc_hash=arc_hash,
                    svn_revision=str(svn_revision),
                    major_release_number=str(major),
                    minor_release_number=str(minor),
                    timestamp=int(get_deploy_timestamp(remote_ya_conf_file, build_task_id)),
                    sandbox_resource=release_and_deploy_pb2.SbResourceData(
                        resource_id=int(resource.id),
                        resource_type=item_data.resource_type,
                        build_task_id=int(build_task_id),
                    ),
                )
                logging.info("Got service_version:\n%s", service_version)
                result.append(service_version)
        return result

    @staticmethod
    def _get_ya_conf(remote_ya_conf_file):
        local_ya_conf_file = os.path.join(LOCAL_YA_CONF_FOLDER, YA_CONF_FILE_NAME)
        if os.path.exists(LOCAL_YA_CONF_FOLDER):
            shutil.rmtree(LOCAL_YA_CONF_FOLDER)
        os.mkdir(LOCAL_YA_CONF_FOLDER)
        sdk2.svn.Arcadia.export(remote_ya_conf_file, local_ya_conf_file)
        conf = fu.json_load(local_ya_conf_file)
        return conf


def set_ya_conf_field(conf, path, new_field):
    conf_part = conf
    path_iter = path.split(".")
    for key in path_iter[:-1]:
        if key not in conf_part:
            raise KeyError("Key {} not found in ya.conf.json".format(key))
        conf_part = conf_part[key]

    conf_part[path_iter[-1]] = new_field


def get_ya_conf_field_val(conf, path):
    conf_part = conf
    for key in path.split("."):
        if key not in conf_part:
            return Error("Key {} not found in {}".format(key, conf_part))
        conf_part = conf_part[key]
    return Ok(conf_part)


def get_deploy_timestamp(remote_ya_conf_file, build_task_id):
    # type: (typing.AnyStr, typing.Union[typing.AnyStr, int]) -> float
    blame = sdk2.svn.Arcadia.blame(remote_ya_conf_file, verbose=True)
    build_task_id_str = str(build_task_id)
    for i in blame.split("\n"):
        parts = i.split()
        if build_task_id_str in "".join(parts[5:]):
            deploy_datetime = parse_datetime(" ".join(parts[2:5]))
            ts = tu.datetime_to_timestamp(deploy_datetime)
            LOGGER.info("Got timestamp for deploy (%s): %s", deploy_datetime.isoformat(), ts)
            return ts


def get_release_items(
    c_info,  # type: typing.Any
    resources_info,  # type: typing.List[dict]
):
    # type: (...) -> List[Tuple[sdk2.Resource, ri.ReleasableItem]]
    for releasable_item in c_info.releases_cfg__releasable_items:
        yield get_release_resource(releasable_item.data, resources_info), releasable_item


def get_release_resource(item_data, resources_info):
    # type: (typing.Union[ri.SandboxResourceData, config.ReleasedResourceInfo], typing.List[dict]) -> sdk2.Resource
    for i in resources_info:
        if (
            i["type"] == item_data.resource_type and (
                not item_data.attributes or
                not (set(item_data.attributes.items()) - set(i["attributes"].items()))
            )
        ):
            return sdk2.Resource[i["id"]]
