# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import base64
import hashlib
import logging
from concurrent.futures import as_completed, ThreadPoolExecutor
from multiprocessing.dummy import Pool as ThreadPool
from typing import (  # noqa: UnusedImport
    Any,
    Dict,
    Iterable,
    Generator,
    List,
    Optional,
    Set,
    Text,
    Union,
)

import sandbox.common.itertools as it
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.client as rm_client
import sandbox.projects.release_machine.events as rm_events
import sandbox.projects.common.error_handlers as eh
from sandbox.projects.common import decorators

LOGGER = logging.getLogger(__name__)
EVENTS_BY_TASK_NAME = {
    "GENERATE_YAPPY_BETA": rm_events.EventType.NewBetaGeneration,
    "RELEASE_RM_COMPONENT": rm_events.EventType.ReleaseCreated,
    "RELEASE_RM_COMPONENT_2": rm_events.EventType.ReleaseCreated,
    "RELEASE_RM_COMPONENT_AB": rm_events.EventType.ReleaseCreated,
    "CREATE_STARTREK_TICKET": rm_events.EventType.TicketHistory,
    "CREATE_WIKI_PAGE": rm_events.EventType.NewWikiPage,
    "CLONE_TESTENV_DB": rm_events.EventType.TestenvDbClone,
    "ROLLBACK_COMMIT": rm_events.EventType.RollbackCommit,
    "MERGE_TO_STABLE": rm_events.EventType.MergeCommit,
}
EVENTS_BY_TASK_PREFIX = {
    "BUILD": rm_events.EventType.BuildTest,
    "YA_BUILD": rm_events.EventType.BuildTest,
    "YA_PACKAGE": rm_events.EventType.BuildTest,
    "YA_MAKE": rm_events.EventType.BuildTest,
    "KOSHER_YA_MAKE": rm_events.EventType.BuildTest,
    "LAUNCH_METRICS": rm_events.EventType.AcceptanceTest,
    "ACCEPTANCE": rm_events.EventType.AcceptanceTest,
}
# tests to show in current state
STATE_TESTS = frozenset([
    rm_events.EventType.GenericTest,
    rm_events.EventType.GenericDiff,
    rm_events.EventType.AcceptanceTest,
    rm_events.EventType.BuildTest,
    rm_events.EventType.NewBetaGeneration,
])
SCOPE_NUMBER_KEY = "scope_number"
INFERRED_RM_PROTO_EVENT_KEY = "inferred_rm_proto_event_data"
RM_PROTO_EVENT_CTX_KEY = "context.rm_proto_event"
RM_PROTO_EVENT_READY_CTX_KEY = "context.event_ready"

RELEASE_STATUSES = {"RELEASING", "RELEASED"}
BUILD_TASK_PREFIXES = frozenset([
    "YA_BUILD",
    "YA_PACKAGE",
    "BUILD",
    "YA_MAKE",
    "KOSHER_YA_MAKE",
])
DO_NOW_CRAWL_TASKS = frozenset([
    "CHECK_TASKS_CORRECTNESS",  # this task is just trigger for other tasks, no need to crawl it
    "CREATE_STARTREK_TICKET",  # this task is crawled by ReleaseMonitorCrawlTickets
])


class GSIDInfo(object):
    def __init__(self, gsid, job_name='', scope_number='', revision='', revision1='', revision2=''):
        self.gsid = gsid
        self.job_name = job_name
        self.scope_number = scope_number
        self.revision = revision
        self.revision1 = revision1
        self.revision2 = revision2

    def upd_svn_info(self, event_type):

        if event_type == rm_events.EventType.GenericDiff:
            svn_info = rm_const.GSID_SVNDIFF_RE.search(self.gsid)
            if svn_info:
                self.revision1 = svn_info.group("rev1")
                self.revision2 = svn_info.group("rev2")
        else:
            svn_info = rm_const.GSID_SVN_RE.search(self.gsid)
            if svn_info:
                self.revision = svn_info.group("svn_revision")

    def upd_te_info(self, c_info):
        te_info = rm_const.GSID_TE_RE.search(self.gsid)
        if te_info:
            self.job_name = te_info.group("job_name")
            if c_info.is_branched:
                m = c_info.testenv_cfg__branch_db_regex.match(te_info.group("db_name"))
                self.scope_number = m.group(1) if m else ""


def post_events(events, send_to_testing=True):
    """DEPRECATED! Use post_proto_events"""
    logging.error("THIS METHOD DOES NOT WORK ANYMORE")
    return False


def _do_post_proto_events(event_messages, rm_client_prod=None, rm_client_dev=None, already_a_json=False):
    result_ok = True

    if not event_messages:
        return result_ok

    LOGGER.info("Posting %d events", len(event_messages))

    pool = ThreadPool(2)

    prod_result, dev_result = None, None

    if already_a_json:
        prod_method = rm_client_prod.post_proto_events_as_json
        dev_method = rm_client_dev.post_proto_events_as_json
    else:
        prod_method = rm_client_prod.post_proto_events
        dev_method = rm_client_dev.post_proto_events

    try:

        if rm_client_prod:
            prod_result = pool.apply_async(prod_method, args=(event_messages, ))

        if rm_client_dev:
            LOGGER.debug("Sending event to testing")
            dev_result = pool.apply_async(dev_method, args=(event_messages, ))

        result_ok = (prod_result.get() is not None) if rm_client_prod else True

    finally:
        LOGGER.debug("Closing thread pool")
        pool.close()
        if rm_client_dev and dev_result:
            LOGGER.debug("Testing events: %s", "DONE" if dev_result.ready() else "STILL IN PROGRESS")
    return result_ok


def post_proto_events(event_messages, prod=True, dev=True, already_a_json=False):
    LOGGER.info("Posting proto events begins")
    result_ok = True
    rm_client_prod = rm_client.RMClient() if prod else None
    rm_client_dev = rm_client.RMClient(rm_const.Urls.RM_TESTING_URL) if dev else None
    for chunk in it.chunker(event_messages, size=300):
        result_ok = result_ok and _do_post_proto_events(chunk, rm_client_prod, rm_client_dev, already_a_json)
    LOGGER.info("Posting finished %s", "OK" if result_ok else "with problems")
    return result_ok


def get_event_type(task_type):
    if task_type in EVENTS_BY_TASK_NAME:
        return EVENTS_BY_TASK_NAME[task_type]
    for task_prefix, event_type in EVENTS_BY_TASK_PREFIX.items():
        if task_type.startswith(task_prefix):
            return event_type
    return rm_events.EventType.GenericTest


def get_event_task_status(task_info):
    return task_info.get("task_status") or task_info.get("status") or ""


def infer_type(task_info):
    gsid = task_info.get("context.__GSID", "")

    if "SVNDIFF" in gsid:
        return rm_events.EventType.GenericDiff

    task_type = task_info.get("type", "")
    event_type = get_event_type(task_type)

    if event_type == rm_events.EventType.BuildTest:
        for build_task_prefix in BUILD_TASK_PREFIXES:
            if (
                task_type.startswith(build_task_prefix) and
                get_event_task_status(task_info) in RELEASE_STATUSES
            ):
                event_type = None
                break

    return event_type


def rm_event_data_by_task_info_generator(c_info, tasks_info_list, skip_none=True):
    """

    :param c_info: component info instance
    :param tasks_info_list: a list of task info dictionaries;
    the following dict keys are expected (some are optional however):
    * `id` — task id
    * `type` — task type
    * `status` — task status
    * `context.__GSID`
    * `revision` —  (optional) svn revision
    * `job_name` — (optional)  TestEnv job name

    :param skip_none:
    :return:
    """
    LOGGER.debug("Retrieving RM event data for %s from %s", c_info.name, tasks_info_list)
    for task_info in tasks_info_list:
        if task_info["type"] in DO_NOW_CRAWL_TASKS:
            continue
        task_id = task_info["id"]
        event_type = infer_type(task_info)
        LOGGER.info("Process new event %s. Task id: %s", event_type, task_id)
        rm_proto_event_data = task_info.get(RM_PROTO_EVENT_CTX_KEY, None)
        if not rm_proto_event_data and skip_none:
            LOGGER.info("No rm_proto_event_data, in task %s. Skip it", task_id)
            continue
        yield task_info, event_type


def upd_state_event(event_type, task_info, c_info):
    if event_type not in STATE_TESTS:
        return
    LOGGER.info("Event is used for state. Try to infer some info from task")
    gsid = task_info["context.__GSID"]
    gsid_info = GSIDInfo(gsid)
    gsid_info.upd_svn_info(event_type)
    gsid_info.upd_te_info(c_info)
    _upd_proto_event_info(event_type, task_info, c_info, gsid_info)


def get_job_name_and_scope_number_from_gsid(c_info, gsid):
    te_info = rm_const.GSID_TE_RE.search(gsid)

    job_name, scope_number = "", ""

    if te_info:
        job_name = te_info.group("job_name")

        if c_info.is_branched:
            m = c_info.testenv_cfg__branch_db_regex.match(te_info.group("db_name"))
            scope_number = m.group(1) if m else ""

    return job_name, scope_number


def _upd_proto_event_info(event_type, task_info, c_info, gsid_info):

    LOGGER.debug("Update protobuf event data")

    if task_info.get(RM_PROTO_EVENT_CTX_KEY):
        LOGGER.debug("The task defines a custom %s", RM_PROTO_EVENT_CTX_KEY)
        return

    if event_type not in (
        rm_events.EventType.GenericTest,
        rm_events.EventType.GenericDiff,
        rm_events.EventType.BuildTest,
    ):
        LOGGER.debug("%s events are not processed as proto events by this crawler", event_type)
        return

    try:

        from release_machine.common_proto import events_pb2, test_results_pb2

        event_hash = hashlib.md5(
            "{}{}{}".format(
                gsid_info.gsid,
                rm_const.EVENT_HASH_SEPARATOR,
                task_info['time.updated'],
            ).encode("utf-8")
        ).hexdigest()

        task_status_str = task_info['status'].upper()

        event_proto = events_pb2.EventData(
            general_data=events_pb2.EventGeneralData(
                hash=event_hash,
                component_name=task_info.get('input_parameters.component_name') or c_info.name,
                referrer='',  # populate later
            ),
            task_data=events_pb2.EventSandboxTaskData(
                task_id=int(task_info.get('id')),
                status=task_status_str,
                created_at=task_info.get('started_at', task_info['time.created']),
                updated_at=task_info.get('finished_at', task_info['time.updated']),
            ),
        )
        if task_status_str == 'SUCCESS' or task_status_str == 'RELEASED':
            test_result_status = test_results_pb2.TestResult.TestStatus.OK
        else:
            test_result_status = test_results_pb2.TestResult.TestStatus.WARN

        if event_type == rm_events.EventType.GenericTest:
            event_proto.generic_test_data.job_name = gsid_info.job_name
            event_proto.generic_test_data.scope_number = gsid_info.scope_number
            event_proto.generic_test_data.revision = gsid_info.revision

            event_proto.generic_test_data.test_result.status = test_result_status

        elif event_type == rm_events.EventType.GenericDiff:

            if gsid_info.revision1:
                event_proto.generic_diff_data.revision1 = gsid_info.revision1
                event_proto.generic_diff_data.revision2 = gsid_info.revision2

            event_proto.generic_diff_data.job_name = gsid_info.job_name
            event_proto.generic_diff_data.scope_number = gsid_info.scope_number

        else:  # BuildTest
            event_proto.build_test_data.job_name = gsid_info.job_name
            event_proto.build_test_data.scope_number = gsid_info.scope_number
            event_proto.build_test_data.revision = gsid_info.revision
            event_proto.build_test_data.test_result.status = test_result_status

        task_info[INFERRED_RM_PROTO_EVENT_KEY] = event_proto

        LOGGER.debug("New protobuf event generated: %s", event_proto)

    except Exception as e:
        eh.log_exception("Unable to build proto event for {}, {}".format(event_type, task_info), e)


def update_rm_event_data_generator(c_info, rm_event_data_generator):
    for task_info, event_type in rm_event_data_generator:
        upd_state_event(event_type, task_info, c_info)
        yield task_info, event_type


def updated_rm_event_data_by_task_info_generator(c_info, task_info_list):
    return update_rm_event_data_generator(
        c_info,
        rm_event_data_by_task_info_generator(c_info, task_info_list, skip_none=False),
    )


def parse_rm_proto_event_from_b64_encoded_str(rm_proto_event_encoded):
    """
    Parse base64-encoded RM Proto Event Message into appropriate protobuf message object

    :raises TypeError: invalid input or invalid base64 encoding
    :raises base64.binascii.Error: invalid base64 encoding
    :raises google.protobuf.message.DecodeError: cannot parse EventData protobuf from the given string

    :param rm_proto_event_encoded: str/unicode, base64-encoded event protobuf message
    """
    from release_machine.common_proto import events_pb2 as rm_proto_events

    event_message = rm_proto_events.EventData()
    event_message.ParseFromString(base64.b64decode(rm_proto_event_encoded))

    return event_message


def convert_sandbox_task_status_into_rm_event_enum_status(task_status):
    from release_machine.common_proto import events_pb2 as rm_proto_events
    return getattr(rm_proto_events.EventSandboxTaskData.TaskStatus, task_status)


@decorators.decorate_all_public_methods(decorators.log_start_and_finish(LOGGER))
class EventHelper(object):
    def __init__(self, use_prod_client=True, use_testing_client=True):
        from release_machine.release_machine.services.release_engine.services.Event import EventClient

        self._prod_event_client = EventClient.from_address(rm_const.Urls.RM_HOST) if use_prod_client else None
        self._test_event_client = EventClient.from_address(rm_const.Urls.RM_TESTING_HOST) if use_testing_client else None

    def post_grpc_events(self, events):
        """
        Post event using grpc clients.

        :param events: List[EventData]
        :return: True if all prod queries were ok, False otherwise.
        """
        from release_machine.release_machine.proto.structures import message_pb2

        LOGGER.info("Posting %d events", len(events))
        prod_futures = []
        testing_futures = []
        with ThreadPoolExecutor(2) as executor:
            for chunk in it.chunker(events, size=300):
                prod_future, testing_future = self._submit_method(
                    executor, "post_proto_events", message_pb2.PostProtoEventsRequest(events=chunk)
                )
                prod_futures.append(prod_future)
                testing_futures.append(testing_future)
            prod_results = list(self._get_future_results(prod_futures))
            testing_results = list(self._get_future_results(testing_futures))
            LOGGER.info("Prod grpc event results:\n%s", "\n".join(map(str, prod_results)))
            LOGGER.info("Test grpc event results:\n%s", "\n".join(map(str, testing_results)))
        return all(i.ok for i in prod_results)

    @staticmethod
    def _get_future_results(futures, wait=True):
        futures = [future for future in futures if future is not None]
        iterator = as_completed(futures) if wait else futures
        for i in iterator:
            try:
                yield rm_core.Ok(i.result() if i else None)
            except Exception as e:
                yield rm_core.Error(e)

    def _submit_method(self, executor, method_name, *args):
        # type: (ThreadPoolExecutor, str, *Any) -> Generator[Optional[Future]]
        for client in [self._prod_event_client, self._test_event_client]:
            if client is None:
                yield
            else:
                method = getattr(client, method_name)
                yield executor.submit(method, *args)
