"""
    This module MUST NOT contain/import sandbox-specific things!
    Has unit tests! Run them if you change something here!
    python merge_helper_ut.py
"""

import logging
import re
import sandbox.projects.common.link_builder as lb
import sandbox.projects.release_machine.core as rm_core
from sandbox.sdk2.vcs.svn import Arcadia


MERGE_RE = re.compile(r"\[mergeto:(.+?)\]", re.IGNORECASE)

CHECK_SYNTAX = (
    "Probably commit message has incorrect syntax.\n"
    "Please check that syntax is correct ( https://nda.ya.ru/3TKQ8Y )"
)

IGNORED_RE = [
    (re.compile(r"__ignored__\s*:\s*(r\d+(,[\s]*r\d+)*)", re.IGNORECASE), True),
    (re.compile(r"__ignored__\s*:\s*(\d+(,[\s]*\d+)*)", re.IGNORECASE), False),
]

ARC_IGNORED_RE = [
    (re.compile(r"__ignored__\s*:\s*([a-z\d]+(,[\s]*[a-z\d]+)*)", re.IGNORECASE), False),
]

MERGE_REG_EXPRESSIONS = [
    (re.compile(r"^Merge[\s]*from[\s]*trunk[\s]*:[\s]*(r\d+(,[\s]*r\d+)*)", re.IGNORECASE), True),
    (re.compile(r"^Merge[\s]*from[\s]*trunk[\s]*:[\s]*(\d+(,[\s]*\d+)*)", re.IGNORECASE), False),
]

ROLLBACK_REG_EXPRESSIONS = [
    (re.compile(r"^Rollback[\s]*:[\s]*(r\d+(,[\s]*r\d+)*)", re.IGNORECASE), True),
    (re.compile(r"^Rollback[\s]*:[\s]*(\d+(,[\s]*\d+)*)", re.IGNORECASE), False),
]

ARC_MERGE_REG_EXPRESSIONS = [
    (re.compile(r"^Merge[\s]*from[\s]*trunk[\s]*:[\s]*([a-z\d]+(,[\s]*[a-z\d]+)*)", re.IGNORECASE), False),
]

ARC_ROLLBACK_REG_EXPRESSIONS = [
    (re.compile(r"^Rollback[\s]*:[\s]*([a-z\d]+(,[\s]*[a-z\d]+)*)", re.IGNORECASE), False),
]

ARC_CHERRY_PICK_REG_EXPRESSION = re.compile(r"\n\(cherry picked from commit ([a-z0-9]+)\)")
ARC_REVERT_REG_EXPRESSION = re.compile(r"\nThis reverts commit ([a-z0-9]+)\.")


class CommitResult(object):
    SUCCEEDED = "SUCCEEDED"
    FAILED = "FAILED"


class CommitNotification(object):
    def __init__(self, merge_path, revs, zero_diff=False):
        self.merge_path = merge_path
        self.str_revs_short = ", ".join(map(str, revs))
        self.str_revs_long = "\n".join(lb.revision_link(rev, link_type=lb.LinkType.plain) for rev in revs)
        self.zero_diff_msg = " (WITH EMPTY CHANGESET)" if zero_diff else ""


class MergeNotification(CommitNotification):
    def short(self, is_failed):
        if is_failed:
            return "Revisions {} merge to {} FAILED".format(self.str_revs_short, self.merge_path.short)
        else:
            return "Revisions {} merged to {}{}".format(self.str_revs_short, self.merge_path.short, self.zero_diff_msg)

    def long(self, is_failed):
        if is_failed:
            return (
                "Merge to '{component_branch_name}' FAILED "
                "for the following revisions:\n{revs_long_list}"
            ).format(
                component_branch_name=self.merge_path.short,
                revs_long_list=self.str_revs_long,
            )
        else:
            return (
                "The following revisions were successfully merged{zero_diff_msg} "
                "to {component_branch_name}:\n"
                "{revs_long_list}\n"
                "Please note that by merging revision into stable branch YOU ARE TAKING RESPONSIBILITY "
                "FOR THAT COMMIT(S) and all its (maybe negative) consequences.\n"
                "Please ensure that manual testing was done correctly "
                "or ask your duty release engineer to run them if you need "
                "(see RMDEV-1499 and https://nda.ya.ru/t/phBBvABn3hF5mP for details)."
            ).format(
                component_branch_name=self.merge_path.short,
                revs_long_list=self.str_revs_long,
                zero_diff_msg=self.zero_diff_msg,
            )


class RollbackNotification(CommitNotification):
    def short(self, is_failed):
        if is_failed:
            return "Revisions {} rollback FAILED in branch {}".format(self.str_revs_short, self.merge_path.short)
        else:
            return "Revisions {} were rolled back from branch {}".format(self.str_revs_short, self.merge_path.short)

    def long(self, is_failed):
        if is_failed:
            return (
                "Rollback for component '{component_name}' FAILED\n"
                "in branch '{component_branch_name}' "
                "for the following revisions:\n{revs_long_list}"
            ).format(
                component_name=self.merge_path.component_name,
                component_branch_name=self.merge_path.short,
                revs_long_list=self.str_revs_long,
            )
        else:
            return (
                "The following revisions were ROLLED BACK for component '{component_name}' "
                "in branch '{component_branch_name}':\n"
                "{revs_long_list}"
            ).format(
                component_name=self.merge_path.component_name,
                component_branch_name=self.merge_path.short,
                revs_long_list=self.str_revs_long,
            )


class Action(object):
    MERGE = 'merge'
    ROLLBACK = 'rollback'

    NOTIFY = {
        MERGE: MergeNotification,
        ROLLBACK: RollbackNotification,
    }
    COMMIT_HEAD = {
        MERGE: "Merge from trunk:",
        ROLLBACK: "Rollback:",
    }

    @staticmethod
    def startrek_msg(action, revs_list, is_failed):
        status = CommitResult.FAILED if is_failed else CommitResult.SUCCEEDED
        return "{} action for revisions {} **{}**".format(
            action.capitalize(),
            revs_list,
            status
        )


def parse_mergeto(message, allowed_components):
    """
        Extract mergeto information from commit message
        Raises exception on invalid marker.
        :param message: commit message text
        :return: Dict per component name with lists of branch numbers
    """
    parse_result = {}

    for items in MERGE_RE.findall(message):
        for component_name, branch_nums in _parse_mergeto_items(items, allowed_components):
            parse_result[component_name] = branch_nums

    return parse_result


def extract_branch_nums(branch_nums):
    """
        Splits branch numbers string like "127, 126 128" into tokens and sort them
        :param branch_nums: input branch nums descriptor string
        :return: list of branch numbers (as strings)
    """
    chunks = str(branch_nums).replace("'", "").replace(",", " ").split(' ')
    branch_nums = []
    for num in chunks:
        if num:
            if num != '+' and not num.isdigit():
                raise ValueError("Invalid branch number: '{}', it should be numeric.".format(num))
            branch_nums.append(num)
    if '+' in branch_nums:
        if len(branch_nums) == 1 or len(branch_nums) > 2:
            raise IOError("There is \"+\" in branch_nums but it's not branch_num,+. Aborting")
        if branch_nums == ['+', '+']:
            raise ValueError("There is two \"+\" in branch_nums but it's not branch_num,+. Aborting")
    return [str(num) for num in sorted(branch_nums, key=lambda x: 1e9 if x == '+' else int(x))]


def _parse_mergeto_items(items, allowed_components):
    """
        Should be <component-name>(:branch-numbers)
        :param items: item to parse
        :return: generator of tuples (component name, sorted branch numbers list)
    """
    for item in items.split(';'):
        item = item.strip()
        component_name = item
        branch_nums = []
        if ':' in item:
            splitted_item = item.split(':')
            if len(splitted_item) != 2:
                raise ValueError("Too many colons in mergeto item. ")
            component_name = splitted_item[0]
            branch_nums = extract_branch_nums(splitted_item[1])

        if component_name not in allowed_components:
            raise Exception(
                "Invalid component name: '{component_name}' in mergeto item `{item}`".format(
                    component_name=component_name,
                    item=item,
                ))

        yield component_name, branch_nums


def _all_paths_zip(all_paths):
    """
        Transposes "paths matrix".
        Please DO NOT remove debug output from here.
        :param all_paths: List of DiffItem-s
        :return: Max common path (as generator)
    """
    level = 0
    while True:
        logging.debug("Selecting paths column at level %s...", level)
        column = [diff_item.path_list[level] if level < len(diff_item.path_list) else None for diff_item in all_paths]
        logging.debug("Got path column %s", column)
        if len(set(column)) == 1 and column[0] is None:
            logging.debug("Detected 'None' column at level %s", level)
            # all items are None
            break

        level += 1
        yield column


def get_max_common_path(all_paths):
    max_common_path = []
    for items in _all_paths_zip(all_paths):
        items_set = set(items)
        if len(items_set) > 1:
            # non-common elements in `items` found
            break
        max_common_path.append(list(items_set)[0])

    logging.info("Preliminary max common path: %s", max_common_path)

    # common path adjustment according to SEARCH-3121 bug
    for diff_item in all_paths:
        if diff_item.path_list == max_common_path:
            logging.debug(
                "Found max common path in all_paths: %s, operation is %s",
                max_common_path, diff_item.operation,
            )
            if diff_item.operation in {"A", "D", "R"}:
                max_common_path.pop()
                logging.info("Adjusted max common path: %s, operation: %s", max_common_path, diff_item.operation)
                break

    logging.info("Max common path: %s", max_common_path)
    return max_common_path


def prepare_diff_items(diff_paths):
    return [DiffItem(operation=diff_item[0], path=diff_item[1]) for diff_item in diff_paths]


def update_message(message, merge_revision):
    """
    Insert merged revision links into commit message
    Insert "Merged as" message into commit message
    :return: formatted message, list of merged revision links
    """
    lines = message.split("\n")
    merge_from_line = lines[0]
    revs_to_merge = []  # todo: move it outside the function
    # FIXME(mvel): not case sensitive, use common regexes
    if "erge from" in merge_from_line:
        action_and_revs = merge_from_line.split(":")
        if len(action_and_revs) == 2:
            revs_to_merge = _get_revs_from_message(merge_from_line)
            links_revs_to_merge = [lb.revision_link(r, link_type=lb.LinkType.wiki) for r in revs_to_merge]
            lines[0] = "{action}: {link}".format(action=action_and_revs[0], link=", ".join(links_revs_to_merge))
        else:
            # other cases are wrong merge line
            logging.warning("Wrong merge line detected: %s", merge_from_line)

    link_merged_rev = lb.revision_link(merge_revision, link_type=lb.LinkType.wiki)
    result_revision_line = "Merged as: {link}".format(link=link_merged_rev)
    lines.insert(0, result_revision_line)
    return "\n".join(lines), revs_to_merge


def _get_revs_from_message(message):
    merged = get_merges_from_commit_message(message)
    rollbacked = get_rollbacks_from_commit_message(message)
    revs = merged or rollbacked
    return revs


class DiffItem(object):
    def __init__(self, operation, path):
        self.operation = operation
        self.path_list = path.split("/")


def check_branch_numbers_correctness(branch_numbers):
    """
    Checks that branch_numbers is not empty and all of them are positive
    :param branch_numbers: list of ints (because of input task parameters)
    :return: Error or Ok class object
    """
    if not branch_numbers:
        return rm_core.Error("There are no branches, please enter correct branches(s).")
    for branch in branch_numbers:
        if branch <= 0:
            return rm_core.Error("All branch numbers should be positive integers.")
    return rm_core.Ok()


def check_branch_existence(comp_name, branch, rev_to_checkout=None):
    """
    Checks that path to branch exists on revision (if not None)
    :param comp_name: component name, string
    :param branch: path to branch, string
    :param rev_to_checkout: revision to checkout, int
    :return: Error or Ok class object
    """
    if not Arcadia.check(branch, revision=rev_to_checkout):
        rev_info = rev_to_checkout or Arcadia.info(Arcadia.ARCADIA_BASE_URL)['commit_revision']
        return rm_core.Error("[{}] Path {} doesn't exist on revision {}".format(comp_name, branch, rev_info))
    return rm_core.Ok()


def check_revs_correctness(revs):
    """
    Checks that all revs are correct
    :param revs: revisions, list of strings
    :return: Error or Ok class object
    """
    arc_hashes = []
    svn_revs = []
    if not revs:
        return rm_core.Error("There are no revisions, please enter correct revision(s).")
    for rev in revs:
        logging.debug("Rev: %s", rev)
        if len(rev) == rm_core.rm_const.ARC_HASH_LENGTH:
            logging.debug("It should be arc hash %s", rev)
            arc_hashes.append(rev)
            continue
        if not rev.isdigit() or len(rev) < 7:
            return rm_core.Error(
                "You've entered wrong svn revision {rev}: it should be a digit with at least 7 characters".format(
                    rev=rev,
                )
            )
        svn_revs.append(rev)
    if svn_revs and arc_hashes:
        return rm_core.Error(
            "You've entered svn revisions {revs} and arc hashes {hashes}, "
            "please choose one format and try again".format(
                revs=", ".join(svn_revs),
                hashes=", ".join(arc_hashes),
            )
        )
    return rm_core.Ok()


def split_and_check_revs_correctness(revs_string):
    """
        Splits revisions string like "100, r200, 300  400"
        or "605babfb59230d9e251145ab790ad6800aff9517, a205c0b86e62d346798285a8546c0a7be33136ce" into tokens.
        :param revs_string: input revision descriptor string
        :return: splitted revisions list
    """
    # split by spaces, non-breaking spaces, commas, semicolons, dots as well (see also SEARCH-6141)
    chunks = (
        str(revs_string)
        .replace(',', ' ')
        .replace('.', ' ')
        .replace(';', ' ')
        .replace('@', ' ')
        .replace('-', ' ')
        .replace('\xc2\xa0', ' ')
    ).split(' ')
    revs = [rev for rev in chunks if rev]
    # Arc hashes are 40 symbols long, and don't have prefix `r`. Svn revisions can have it, so we should strip them.
    revs = [rev.strip("r") if len(rev) < rm_core.rm_const.ARC_HASH_LENGTH else rev for rev in revs]
    logging.debug("Got revisions before check: %s", revs)
    check = check_revs_correctness(revs)
    if check.ok:
        return rm_core.Ok(revs)
    else:
        return check


def _find_matches(reg_expressions, message, svn_commit=False):
    for line in message.split("\n"):
        for reg_exp, has_prefix in reg_expressions:
            matches = reg_exp.match(line.strip())
            if matches:
                offset = 1 if has_prefix else 0
                revs = [s.strip()[offset:] for s in matches.group(1).split(",")]
                if svn_commit:
                    revs = list(map(int, revs))
                # logging.debug("Find matches in line: %s", line)
                return revs
    # logging.debug("Cannot find matches in message: %s", message)
    return []


def get_ignored_revisions_from_commit_message(message):
    """
    Extract list of revisions from message like "__ignored__: r3616687,  r3619024, r3619045"
    """
    merges = _find_matches(IGNORED_RE, message, svn_commit=True)
    return merges


def get_arc_ignored_revisions_from_commit_message(message):
    """
    Extract list of revisions from message like "__ignored__: 01b48d21efbc775d2b8eaddd7598c23923d7db88"
    """
    merges = _find_matches(ARC_IGNORED_RE, message, svn_commit=False)
    return merges


def get_merges_from_commit_message(message):
    """
    Extract list of revisions from message like "Merge from trunk: r3616687,  r3619024, r3619045"
    """
    merges = _find_matches(MERGE_REG_EXPRESSIONS, message, svn_commit=True)
    # logging.debug("Found merges: %s", merges)
    return merges


def get_rollbacks_from_commit_message(message):
    """
    Extract list of revisions from message like "Rollback: r3616687, r3619024, r3619045"
    """
    rollbacks = _find_matches(ROLLBACK_REG_EXPRESSIONS, message, svn_commit=True)
    # logging.debug("Found rollbacks: %s", rollbacks)
    return rollbacks


def get_arc_merges_from_commit_message(message):
    """
    Extract list of revisions from message like
    "Merge from trunk: 01b48d21efbc775d2b8eaddd7598c23923d7db88,  e3b28216cfd972a6f04cb3ba66238299f800fea3"
    """
    merges = _find_matches(ARC_MERGE_REG_EXPRESSIONS, message, svn_commit=False)
    # logging.debug("Found merges: %s", merges)
    return merges


def get_arc_rollbacks_from_commit_message(message):
    """
    Extract list of revisions from message like
    "Rollback: 4189b018085a2a57bf4206f005b8f38aa1553198, 01b48d21efbc775d2b8eaddd7598c23923d7db88"
    """
    rollbacks = _find_matches(ARC_ROLLBACK_REG_EXPRESSIONS, message, svn_commit=False)
    # logging.debug("Found rollbacks: %s", rollbacks)
    return rollbacks


def get_arc_cherry_pick_from_commit_message(message):
    """
    Extract arc hash from message like
    "Add functions for parsing commit messages\n(cherry picked from commit fc61e74cc4428630b76824372226a5a5ef9ac933)"
    """
    cherry_picked = ARC_CHERRY_PICK_REG_EXPRESSION.search(message)
    if cherry_picked:
        cherry_picked_commit = cherry_picked.group(1)
        logging.debug("Found cherry-picked commit: %s", cherry_picked_commit)
        return [cherry_picked_commit]
    logging.debug("Could not find cherry-picked commit in msg %s", message)
    return []


def get_arc_revert_from_commit_message(message):
    """
    Extract arc hash from message like
    "Revert: `Add functions for parsing commit messages`\nThis reverts commit 94f725a903fe0e0f72a1eb8ff2b507bee8261951."
    """
    reverted = ARC_REVERT_REG_EXPRESSION.search(message)
    if reverted:
        reverted_commit = reverted.group(1)
        logging.debug("Found reverted commit: %s", reverted_commit)
        return [reverted_commit]
    logging.debug("Could not find reverted commit in msg %s", message)
    return []
