import datetime
import logging

from collections import defaultdict
from dateutil import parser as dt_parser
from dateutil import tz

from sandbox.projects.common import requests_wrapper


METRICS_API_URL = "https://metrics-qgaas.metrics.yandex-team.ru/api/"


class MetricsApi(object):

    def __init__(self, token):
        self.headers = self._set_headers(token)

    @staticmethod
    def _set_headers(token):
        return {
            'Authorization': 'OAuth {}'.format(token)
        }

    def update_basket(self, basket_id, json_queries):
        """
        Update specified basket with passed queries in json format.
        :param basket_id: basket id to update
        :param json_queries: new queries for basket in json format
        """
        logging.info("Try to update basket '%s' with queries:\n%s", basket_id, json_queries)
        response = requests_wrapper.put(
            "{}/basket/{}/query-generator".format(METRICS_API_URL, basket_id),
            json={
                "type": "RAW",
                "queries": json_queries,
            },
            headers=self.headers,
        )
        logging.info("Basket update response [%s]:\n%s", response.status_code, response.text)
        response.raise_for_status()


def get_aggregated_launchset_detalization(mlm_tasks):

    def visit(task_list, result_dict):
        now = datetime.datetime.utcnow().replace(tzinfo=tz.tzutc())
        then = datetime.datetime.utcfromtimestamp(0).replace(tzinfo=tz.tzutc())
        for item in task_list:
            phase = item['tags'].get('phase')
            if phase in ('CALCULATE', None):
                visit(item['children'] or [], result_dict)
                continue
            item_started_at = dt_parser.parse(item['startedTime'])
            item_finished_at = dt_parser.parse(item['finishedTime'])
            result_dict[phase]['started_at'] = min(
                result_dict[phase].get('started_at', now),
                item_started_at,
            )
            result_dict[phase]['finished_at'] = max(
                result_dict[phase].get('finished_at', then),
                item_finished_at,
            )

    result = defaultdict(dict)
    visit(mlm_tasks, result)

    return result
